# Copyright (C) 2008 Karline Soetaert (K.Soetaert@nioo.knaw.nl) and Heloise Lavigne
# Revised by James Orr, 2012-01-17
#
# This file is part of seacarb.
#
# Seacarb is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation; either version 2 of the License, or any later version.
#
# Seacarb is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with seacarb; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#


#--------------------------------------------------------------
# Conversion factors for converting dissociation constants 
# from total pH scale to free pH scale (ktotal2free) 
# from total pH scale to seawater pH scale (ktotal2SWS)
# and from free pH scale to seawater scale (kfree2SWS)
# kfree = ktotal * ktotal2free
# kSWS  = ktotal * ktotal2SWS
# kSWS  = kfree  * kfree2SWS
# kfree = kSWS * kSWS2free
# ktotal = kfree * kfree2total
# ktotal = kSWS * kSWS2total
#--------------------------------------------------------------


"kconv" <- function (S=35,T=25,P=0,kf)

{
	#--------------------------------------------------------------
  # CONVERT equilibrium constants to free scale:
	#--------------------------------------------------------------
	#------------------ Ks ----------------------------------------
	#       Dickson and Goyet (1994), Chapter 5, p.13
	#       (required for total2free)
	#       Equilibrium constant for HSO4- = H+ + SO4--
	#
	#       K_S  = [H+]free [SO4--] / [HSO4-]
	#       pH-scale: free scale !!!
	#
	#       the term log(1-0.001005*S) converts from
	#       mol/kg H2O to mol/kg soln
	Ks = Ks(S=S, T=T, P=P)                 # on free pH scale
	ST  = 0.14/96.062/1.80655*S    # (mol/kg soln) total sulfate
	total2free  = 1/(1+ST/Ks)      # Kfree = Ktotal*total2free
	total2free <- as.numeric(total2free)	

	#---------------------------------------------------------------------
	# --------------------- Kf  ------------------------------------------
	#  Kf = [H+][F-]/[HF]
	#
	#   (Dickson and Riley, 1979 in Dickson and Goyet,
	#   1994, Chapter 5, p. 14)
	#   pH-scale: 'free'

	# When df is not given as an argument, use the global var dfg, if it exists
	# dfg is now defined in any previous call to Kf routine, when kf is given as an argument
#	kfg <- NULL; rm(kfg); # just to avoid a "note" during the compilation of the package
	if (missing(kf)) {
#          if (exists("kfg")) {
            kf <- get("kfg", envir = parent.frame()) 
#          } else {
#            kf <- "x"
#          }
        }

	Kf = Kf(S=S, T=T, P=P, kf=kf, pHscale="F")
	#Kf  = Kf*total2free       # convert Kf from total to free pH scale


	#------- sws2free -----------------------------------------------
	#
	#       convert from pH_sws ('seawater scale`) to pH ('free`):
	#       pH_sws = pH_free - log(1+S_T/K_S(S,T)+F_T/K_F(S,T))
	FT = 7e-5*(S/35)                  # (mol/kg soln) total fluoride
	free2SWS  = 1+ST/Ks+FT/Kf         # Kfree = Ksws*sws2free
	free2SWS <- as.numeric(free2SWS)
	total2SWS = total2free * free2SWS # KSWS = Ktotal*total2SWS
	total2SWS <- as.numeric(total2SWS)

return (list(ktotal2SWS=total2SWS, ktotal2free=total2free,kfree2SWS=free2SWS, kfree2total=(1/total2free), kSWS2total=(1/total2SWS), kSWS2free=(1/free2SWS)))
}

