% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/grim.R
\name{grim}
\alias{grim}
\title{The GRIM test (granularity-related inconsistency of means)}
\usage{
grim(
  x,
  n,
  items = 1,
  percent = FALSE,
  show_rec = FALSE,
  rounding = "up_or_down",
  threshold = 5,
  symmetric = FALSE,
  tolerance = .Machine$double.eps^0.5
)
}
\arguments{
\item{x}{String. The reported mean or percentage value.}

\item{n}{Integer. The reported sample size.}

\item{items}{Numeric. The number of items composing \code{x}. Default is 1, the
most common case.}

\item{percent}{Boolean. Set \code{percent} to \code{TRUE} if \code{x} is a percentage. This
will convert it to a decimal number and adjust the decimal count (i.e.,
increase it by 2). Default is \code{FALSE}.}

\item{show_rec}{Boolean. For internal use only.  If set to \code{TRUE}, the output
is a matrix that also contains intermediary values from GRIM-testing. Don't
specify this manually; instead, use \code{show_rec} in \code{grim_map()}. Default is
\code{FALSE}.}

\item{rounding}{String. Rounding method or methods to be used for
reconstructing the values to which \code{x} will be compared. Default is
\code{"up_or_down"} (from 5).}

\item{threshold}{Numeric. If \code{rounding} is set to \code{"up_from"}, \code{"down_from"},
or \code{"up_from_or_down_from"}, set \code{threshold} to the number from which the
reconstructed values should then be rounded up or down. Otherwise, this
argument plays no role. Default is \code{5}.}

\item{symmetric}{Boolean. Set \code{symmetric} to \code{TRUE} if the rounding of
negative numbers with \code{"up"}, \code{"down"}, \code{"up_from"}, or \code{"down_from"}
should mirror that of positive numbers so that their absolute values are
always equal. Default is \code{FALSE}.}

\item{tolerance}{Numeric. Tolerance of comparison between \code{x} and the
possible mean or percentage values. Default is circa 0.000000015
(1.490116e-08), as in \code{dplyr::near()}.}
}
\value{
Boolean. \code{TRUE} if \code{x}, \code{n}, and \code{items} are mutually consistent,
\code{FALSE} if not.
}
\description{
\code{grim()} checks if a reported mean value of integer data is
mathematically consistent with the reported sample size and the number of
items that compose the mean value.

Set \code{percent} to \code{TRUE} if \code{x} is a percentage. This will convert \code{x} to a
decimal number and adjust the decimal count accordingly.

The function is vectorized, but it is recommended to use \code{grim_map()} for
testing multiple cases.
}
\details{
The \code{x} values need to be strings because only strings retain
trailing zeros, which are as important for the GRIM test as any other
decimal digits.

Use \code{restore_zeros()} on numeric values (or values that were numeric values
at some point) to easily supply the trailing zeros they might once have
had. See documentation there.

Browse the source code in the grim.R file. \code{grim()} is a vectorized version
of the internal \code{grim_scalar()} function found there.
}
\section{Rounding}{
 Here are the options for the \code{rounding} argument.
Reconstructed mean or percentage values can be rounded in either of these
ways:
\itemize{
\item Rounded to \code{"even"} using base R's own \code{round()}.
\item Rounded \code{"up"} or \code{"down"} from 5. (Note that SAS, SPSS, Stata, Matlab,
and Excel round \code{"up"} from 5, whereas Python rounds \code{"down"} from 5.)
\item Rounded \code{"up_from"} or \code{"down_from"} some number, which then needs to be
specified via the \code{threshold} argument.
\item Given a \code{"ceiling"} or \code{"floor"} at the respective decimal place.
\item Rounded towards zero with \code{"trunc"} or away from zero with
\code{"anti_trunc"}.
The default, \code{"up_or_down"}, allows for numbers rounded either \code{"up"} or
\code{"down"} from 5 when GRIM-testing; and likewise for \code{"up_from_or_down_from"}
and \code{"ceiling_or_floor"}.
}

With \code{rounding = "up_or_down"}, if \code{n} is 40 or 80 and \code{x} has two decimal
places, very few values will test as inconsistent; but note that many will be
with either of \code{rounding = "up"} and \code{rounding = "down"}, or indeed with any
other rounding method. This is part of a more general pattern: \code{n} is 400 or
800 and \code{x} has three decimal places, etc.

For more information about these methods, see documentation for \code{round()},
\code{round_up()}, and \code{round_ceiling()}. These include all of the above ways of
rounding. How the reconstructed values are rounded can also be calibrated by
the \code{threshold} and \code{symmetric} arguments.
}

\examples{
# A mean of 5.19 is not consistent with a sample size of 28:
grim(x = "5.19", n = 28)    # `x` in quotes!

# However, it is consistent with a sample size of 32:
grim(x = "5.19", n = 32)

# For a scale composed of two items:
grim(x = "2.84", n = 16, items = 2)

# With percentages instead of means -- here, 71\%:
grim(x = "71", n = 43, percent = TRUE)

}
\references{
Brown, N. J. L., & Heathers, J. A. J. (2017). The GRIM Test: A
Simple Technique Detects Numerous Anomalies in the Reporting of Results in
Psychology. \emph{Social Psychological and Personality Science}, 8(4), 363–369.
https://journals.sagepub.com/doi/10.1177/1948550616673876
}
\seealso{
\code{grim_map()} applies \code{grim()} to any number of cases at once.
}
