% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/satdad_Rfunctions.R
\name{rArchimaxMevlog}
\alias{rArchimaxMevlog}
\title{r function for Archimax Mevlog models.}
\usage{
rArchimaxMevlog(n, ds, dist = "exp", dist.param = 1)
}
\arguments{
\item{n}{The number of observations.}

\item{ds}{An object of class \code{ds}.}

\item{dist}{The underlying distribution. A character string among \code{"exp"} (the default value), \code{"gamma"} and \code{"ext"}.}

\item{dist.param}{The parameter associated with the choice \code{dist}.  If \code{dist} is \code{"exp"}, then \code{dist.param} is a postive real, the parameter of an exponential distribution. The default value is 1. If \code{dist} is \code{"gamma"}, then \code{dist.param} is a vector that concatenates the shape  and  scale parameters (in this order) of a gamma distribution.}
}
\value{
returns a \code{n x d} matrix containing \code{n} realizations of a \code{d}-variate Archimax Mevlog random vector.
}
\description{
Random vectors generation for some Archimax Mevlog models.
}
\details{
We follow below  Algorithm 4.1 of p. 124 in Charpentier et al. (2014). Let \eqn{\psi} defined by \eqn{\psi(x)=\int_0^\infty \exp(-x t) dF(t)}, the Laplace transform of a positive random variable with  cumulative distribution function \eqn{F}.

Define the random vector \eqn{(U_1,...,U_d)} as \eqn{U_i=\psi(-\log(Y_i)/V)} where
\itemize{
\item{} \eqn{Z} has a multivariate extreme value distribution with stable tail dependence function \eqn{\ell} ; here \eqn{Z} has standard Frechet margins,
\item{} \eqn{(Y_1,...,Y_d)=(\exp(-1/Z_1),...,\exp(-1/Z_d))} the margin transform of \eqn{Z} so that \eqn{Y} is sampled from the extreme value copula associated with \eqn{\ell},
\item{} \eqn{V} has the distribution function \eqn{F},
\item{} \eqn{Y} and \eqn{V} are independent.
}
Then,  \eqn{U} is sampled from the Archimax copula \eqn{C(u_1,...,u_d) = \psi(\ell(\psi^{-1}(u_1),...,\psi^{-1}(u_d)))}.

We restrict here the function \eqn{\ell} to those associated with Mevlog models. See \code{\link[satdad]{ellMevlog}} and \code{\link[satdad]{gen.ds}}.

We restrict also the distribution of \eqn{V} to
\itemize{
\item{} exponential ; For a positive \eqn{\lambda}, set \eqn{dF(t)=\lambda \exp(-\lambda t) 1_{t>0} dt},  then \eqn{\psi(x)=\frac{\lambda}{x+\lambda}} and \eqn{\psi^{-1}(x)=\lambda \frac{1-x}{x}}.
 \item{} gamma ; For positive scale \eqn{\sigma} and shape \eqn{a}, set \eqn{dF(t)= \frac{1}{\sigma^a \Gamma(a)}t^{a-1}\exp(-t/\sigma)1_{t>0}}, then \eqn{\psi(x)=\frac{1}{(x+\sigma)^a}} and \eqn{\psi^{-1}(x)=\frac{x^{-1/a}-1}{\sigma}}.
 }
}
\examples{

## Fix a  5-dimensional asymmetric tail dependence structure
(ds5 <- gen.ds(d = 5))

## Generate a 1000-sample of Archimax Mevlog random vectors
## associated with ds5 and underlying distribution gamma
(shape5 <- runif(1, 0.01, 5))
(scale5 <- runif(1, 0.01, 5))
sample5.gamma <- rArchimaxMevlog(n = 1000, ds = ds5, dist = "gamma", dist.param = c(shape5, scale5))

## Compare theoretical (left) and empirical (right) tail dependographs
oldpar <- par(mfrow = c(1,2))
graphs(ds = ds5)
graphsEmp(sample = sample5.gamma, k = 100)
par(oldpar)

## Generate a 1000-sample of Archimax Mevlog random vectors
## associated with ds5 and underlying distribution exp
(lambda <- runif(1, 0.01, 5))
sample5.exp <- rArchimaxMevlog(n = 1000, ds = ds5, dist = "exp", dist.param = lambda)
## Compare theoretical (left) and empirical (right) tail dependographs
graphs(ds = ds5)
graphsEmp(sample = sample5.exp, k = 100)

}
\references{
Charpentier, A.,   Fougères,  A.-L.,  Genest, C. and  Nešlehová, J.G. (2014)
Multivariate Archimax copulas.
\emph{Journal of Multivariate Analysis}, \bold{126}, 118--136.
}
\seealso{
\code{\link[satdad]{rMevlog}},    \code{\link[satdad]{copArchimaxMevlog}}, \code{\link[satdad]{psiArchimaxMevlog}},  \code{\link[satdad]{psiinvArchimaxMevlog}}, \code{\link[satdad]{gen.ds}}
}
\author{
Cécile Mercadier (\code{mercadier@math.univ-lyon1.fr})
}
