# Copyright (c) 2019 Andrew Marx. All rights reserved.
# Licensed under GPLv3.0. See LICENSE file in the project root for details..

#' @include samc-class.R
NULL


#' Check landscape data
#'
#' Check that landscape inputs have valid values and matching properties.
#'
#' This function is used to ensure that landscape inputs (resistance, absorption,
#' fidelity, and occupancy) have valid values and the same properties. This includes
#' checking the CRS (if using RasterLayer inputs), dimensions, and locations of
#' cells with NA data. It can be used to directly compare two matrices or two
#' RasterLayers, or it can be used to check a \code{\link{samc-class}} object
#' against a matrix or RasterLayer.
#'
#' The function returns \code{TRUE} if the inputs have matching properties. Otherwise,
#' it will stop execution and print the error message generated by the
#' compareRaster() function from the raster package. This error will provide
#' some details about the difference between the two inputs.
#'
#' Note that the package assumes the different landscape inputs will be the same
#' type, either matrices or RasterLayers. Mixing RasterLayer data and matrix
#' data is not supported.
#'
#' @param a A \code{\link{samc-class}}, \code{\link{matrix}}, or \code{\link[raster]{RasterLayer-class}} object
#' @param b A \code{\link{matrix}} or \code{\link[raster]{RasterLayer-class}} object
#
#' @return See \emph{Details} section.
#'
#' @example inst/examples/example.R
#'
#' @export

setGeneric(
  "check",
  function(a, b) {
    standardGeneric("check")
  })

#' @rdname check
setMethod(
  "check",
  signature(a = "RasterLayer", b = "missing"),
  function(a){

    if (sum(is.infinite(a[]), na.rm = TRUE) > 0) {
      stop("Data contains Inf or -Inf element")
    } else if (sum(is.nan(a[]), na.rm = TRUE) > 0) {
      stop("Data contains NaN elements")
    }

    return(TRUE)
  })

#' @rdname check
setMethod(
  "check",
  signature(a = "matrix", b = "missing"),
  function(a){
    a <- raster::raster(a, xmn = 1, xmx = ncol(a), ymn = 1, ymx = nrow(a))

    check(a)
  })

#' @rdname check
setMethod(
  "check",
  signature(a = "RasterLayer", b = "RasterLayer"),
  function(a, b){
    check(a)
    check(b)

    a[] <- is.finite(a[])
    b[] <- is.finite(b[])

    raster::compareRaster(a, b, values = TRUE)
  })

#' @rdname check
setMethod(
  "check",
  signature(a = "matrix", b = "matrix"),
  function(a, b){
    a <- raster::raster(a, xmn = 1, xmx = ncol(a), ymn = 1, ymx = nrow(a))
    b <- raster::raster(b, xmn = 1, xmx = ncol(b), ymn = 1, ymx = nrow(b))

    check(a, b)
  })

#' @rdname check
setMethod(
  "check",
  signature(a = "samc", b = "RasterLayer"),
  function(a, b){
    check(b)

    b[] <- is.finite(b[])

    raster::compareRaster(a@map, b)
  })

#' @rdname check
setMethod(
  "check",
  signature(a = "samc", b = "matrix"),
  function(a, b){
    b <- raster::raster(b, xmn = 1, xmx = ncol(b), ymn = 1, ymx = nrow(b))

    check(a, b)
  })
