#' Sparse Gaussian Markov Random Field Mixtures
#'
#' @param x data.frame. A training data.
#' @param K integer. Number of mixture components. Set a large enough number
#'          because the algorithm identifies major dependency patterns from
#'          the data via the sparse mixture model.
#' @param rho double. Constant that multiplies the penalty term. An optimal
#'          value should be determined together with the threshold on the
#'          anomaly score, so the performance of anomaly detection is maximized.
#' @param m0 a numeric vector. Location parameter of Gauss-Laplace prior.
#'          Keep default if no prior information is available.
#' @param lambda0 double. Coefficient for scale parameter of Gauss-Laplace
#'          prior. Keep default if no prior information is available.
#' @param alpha double. Concentration parameter of Dirichlet prior.
#'          Keep default if no prior information is available.
#' @param pi_threshold double. Threshold to decide a number of states.
#'          If pi < pi_threshold, the states are rejected in the sense of
#'          sparse estimation.
#' @param max_iter integer. Maximum number of iterations.
#' @param tol double. The tolerance to declare convergence.
#' @param verbose logical.
#'
#' @return sGMRFmix object
#'
#' @examples
#' library(sGMRFmix)
#'
#' set.seed(314)
#' train_data <- generate_train_data()
#' fit <- sGMRFmix(train_data, K = 7, rho = 10)
#' fit
#'
#' @export
sGMRFmix <- function(x, K, rho, m0 = rep(0, M), lambda0 = 1, alpha = NULL,
                     pi_threshold = 1/K/100, max_iter = 500, tol = 1e-1,
                     verbose = TRUE) {
  if (!is.data.frame(x)) {
    x <- as.data.frame(x)
  }
  M <- ncol(x)
  if (verbose) message("################## sparseGaussMix #######################")
  fit <- sparseGaussMix(x, K = K, rho = rho, m0 = m0,
                        lambda0 = lambda0, max_iter = max_iter,
                        tol = tol, verbose = verbose)
  pi <- fit$pi
  m <- fit$m
  A <- fit$A
  if (verbose) message("\n################## GMRFmix ##############################")
  ind <- pi >= pi_threshold
  pi <- pi[ind]
  m <- m[ind]
  A <- A[ind]
  fit <- GMRFmix(x, pi = pi, m = m, A = A, alpha = alpha,
                 max_iter = max_iter, tol = tol, verbose = verbose)
  theta <- fit$theta
  H <- fit$H
  if (verbose) message("\n################## Finished #############################")
  mode <- apply(H, 1, function(row) {
    t <- table(row)
    as.integer(names(t)[which.max(t)])
  })
  result <- list(x = x, pi = pi, m = m, A = A, theta = theta, H = H, mode = mode,
                 Kest = length(pi), K = K, rho = rho, m0 = m0, lambda0 = lambda0,
                 pi_threshold = pi_threshold)
  class(result) <- "sGMRFmix"

  cl <- match.call()
  result$call <- cl

  result
}
