#' Make a simple map to visualize VertNet data.
#'
#' Plots record locations on a world or regional map using latitude/longitude
#' data returned by a VertNet search.
#'
#' @export
#' @import ggplot2 maps
#' @param input Output from \code{\link{vertsearch}}, 
#'    \code{\link{searchbyterm}}, or \code{\link{spatialsearch}}. Must
#' 		include columns "decimallatitude" and "decimallongitude"
#' @param mapdatabase The base map on which your data are displayed; what you
#'    choose here determines what you can choose in the region parameter; one
#'    of: county, state, usa, world, world2, france, italy, or nz
#' @param region The region in which your data are displayed; to see region names
#'    for the "world" database layer, run
#'    \code{sort(unique(map_data("world")$region))} after loading packages maps 
#'    and ggplot2; to see region names for the US "state" layer, run 
#'    \code{sort(unique(map_data("state")$region))}
#' @param geom Specifies the type of object being plotted; one of: geom_point or
#'    geom_jitter (do not use quotes)
#' @param jitter If geom = geom_jitter, the amount by which to jitter points in
#'    width, height, or both
#' @return Map of record locations displayed on the selected base map
#' @details \code{vertmap} uses decimal latitude and longitude data in records generated by
#'    an rvertnet search to display returned records on a specified base map. Taxa
#'    are color-coded by scientific name, if available. Adapt the vertmap code to
#'    construct maps according to your own specfications.
#' @examples \dontrun{
#' out <- vertsearch("Junco hyemalis") # get occurrence records
#' vertmap(out)                        # map occurrence records
#'
#' # Records are color coded by dwc term "scientificname" - sometimes unavailble
#' out <- vertsearch("mustela nigripes")
#' vertmap(input = out, mapdatabase = "state")
#'
#' # Use searchbyterm() to match records with mapped region
#' spec <- searchbyterm(genus = "ochotona", specificepithet = "princeps", state = "california",
#' limit = 200)
#' vertmap(input = spec, mapdatabase = "state", region = "california")
#'
#' # Many species
#' splist <- c("Accipiter erythronemius", "Aix sponsa", "Haliaeetus leucocephalus",
#' 		"Corvus corone", "Threskiornis molucca", "Merops malimbicus")
#' out <- lapply(splist, function(x) vertsearch(t=x, lim=100))
#' library("plyr")
#' out <- ldply(lapply(out, "[[", "data"))
#' vertmap(out)
#' }

vertmap <- function(input = NULL, mapdatabase = "world", region = ".", 
                    geom = geom_point, jitter = NULL) {

  if (!class(input) %in% c("list", "data.frame")) {
    stop("Input must be of class list or data.frame", call. = FALSE)
  }
	if (is(input, "list"))	input <- input$data

	if(inherits(input$decimallatitude, "NULL")) {
		stop("Need columns named 'decimallatitude' and 'decimallongitude'") } else {NULL}
	if(inherits(input$decimallongitude, "NULL")) {
		stop("Need columns named 'decimallatitude' and 'decimallongitude'") } else {NULL}
	if(inherits(input$scientificname, "NULL")) { name <- FALSE } else { name <- TRUE }

  input$decimallatitude <- as.numeric(as.character(input$decimallatitude))
	input$decimallongitude <- as.numeric(as.character(input$decimallongitude))
  if (name) {
  	input$scientificname <- as.character(input$scientificname)
  	input$scientificname <- do.call(c, lapply(input$scientificname,
      function(x) paste(strsplit(x, " ")[[1]][1:2],collapse=" ")))
  }

	tomap <- input[complete.cases(input$decimallatitude, input$decimallongitude), ]
	tomap <- tomap[tomap$decimallatitude < 90 & tomap$decimallatitude > -90, ]
	tomap <- tomap[tomap$decimallongitude < 180 & tomap$decimallongitude > -180, ]

	basemap <- map_data(map=mapdatabase, region=region) # get base-map data
  if (name) { 
    # Color record locations by scientificname
    ggplot(basemap, aes(long, lat)) + # Plot using lat/long of base map
    geom_polygon(aes(group=group), fill="white", color="gray40", size=0.2) +
	  geom(data=tomap, aes(decimallongitude, decimallatitude, colour = scientificname),
      alpha=0.4, size=3, position=jitter) +
    labs(x="Longitude (decimal degrees)", y="Latitude") +
    theme_bw(base_size=14)
  } else { 
    # Do not distinguish record locations by color
    ggplot(basemap, aes(long, lat)) +
    geom_polygon(aes(group=group), fill="white", color="gray40", size=0.2) +
    geom(data=tomap, aes(decimallongitude, decimallatitude),
         alpha=0.4, size=3, position=jitter) +
    labs(x="Longitude (decimal degrees)", y="Latitude") +
    theme_bw(base_size=14)
  }

}
