% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rrpca.R
\name{rrpca}
\alias{rrpca}
\title{Randomized robust principal component analysis (rrpca).}
\usage{
rrpca(A, lambda = NULL, maxiter = 50, tol = 1e-05, p = 10, q = 2,
  trace = FALSE, rand = TRUE)
}
\arguments{
\item{A}{array_like; \cr
a real \eqn{(m, n)} input matrix (or data frame) to be decomposed. \cr
na.omit is applied, if the data contain \eqn{NA}s.}

\item{lambda}{scalar, optional; \cr
tuning parameter (default \eqn{lambda = max(m,n)^-0.5}).}

\item{maxiter}{integer, optional; \cr
maximum number of iterations (default \eqn{maxiter = 50}).}

\item{tol}{scalar, optional; \cr
precision parameter (default \eqn{tol = 1.0e-5}).}

\item{p}{integer, optional; \cr
oversampling parameter for \eqn{rsvd} (default \eqn{p=10}), see \code{\link{rsvd}}.}

\item{q}{integer, optional; \cr
number of additional power iterations for \eqn{rsvd} (default \eqn{q=2}), see \code{\link{rsvd}}.}

\item{trace}{bool, optional; \cr
print progress.}

\item{rand}{bool, optional; \cr
if (\eqn{TRUE}), the \eqn{rsvd} routine is used, otherwise \eqn{svd} is used.}
}
\value{
\code{rrpca} returns a list containing the following components:
\describe{
   \item{L}{  array_like; \cr
             low-rank component; \eqn{(m, n)} dimensional array.
   }
   \item{S}{  array_like \cr
              sparse component; \eqn{(m, n)} dimensional array.
   }
}
}
\description{
Robust principal components analysis separates a matrix into a low-rank plus sparse component.
}
\details{
Robust principal component analysis (RPCA) is a method for the robust seperation of a
a rectangular \eqn{(m,n)} matrix \eqn{A} into a low-rank component \eqn{L} and a
sparse comonent \eqn{S}: 

\deqn{A = L + S}

To decompose the matrix, we use the inexact augmented Lagrange multiplier
method (IALM). The algorithm can be used in combination with either the randomized or deterministic SVD.
}
\examples{
library('rsvd')

# Create toy video
# background frame
xy <- seq(-50, 50, length.out=100)
mgrid <- list( x=outer(xy*0,xy,FUN="+"), y=outer(xy,xy*0,FUN="+") )
bg <- 0.1*exp(sin(-mgrid$x**2-mgrid$y**2))
toyVideo <- matrix(rep(c(bg), 100), 100*100, 100)

# add moving object
for(i in 1:90) {
  mobject <- matrix(0, 100, 100)
  mobject[i:(10+i), 45:55] <- 0.2
  toyVideo[,i] =  toyVideo[,i] + c( mobject )
}

# Foreground/Background separation
out <- rrpca(toyVideo, trace=TRUE)

# Display results of the seperation for the 10th frame
par(mfrow=c(1,4))
image(matrix(bg, ncol=100, nrow=100)) #true background
image(matrix(toyVideo[,10], ncol=100, nrow=100)) # frame
image(matrix(out$L[,10], ncol=100, nrow=100)) # seperated background
image(matrix(out$S[,10], ncol=100, nrow=100)) #seperated foreground
}
\references{
\itemize{
 \item [1] N. B. Erichson, S. Voronin, S. L. Brunton and J. N. Kutz. 2019.
       Randomized Matrix Decompositions Using {R}. 
       Journal of Statistical Software, 89(11), 1-48.
       \url{http://doi.org/10.18637/jss.v089.i11}.

  \item  [2] Lin, Zhouchen, Minming Chen, and Yi Ma.
         "The augmented lagrange multiplier method for exact
         recovery of corrupted low-rank matrices." (2010).
         (available at arXiv \url{http://arxiv.org/abs/1009.5055}).
}
}
\author{
N. Benjamin Erichson, \email{erichson@berkeley.edu}
}
