% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plotSlopes.R
\name{plotSlopes}
\alias{plotSlopes}
\alias{plotSlopes.lm}
\title{Generic function for plotting regressions and interaction effects}
\usage{
plotSlopes(model, plotx, ...)

\method{plotSlopes}{lm}(model, plotx, modx, n = 3, modxVals = NULL,
  plotxRange = NULL, interval = c("none", "confidence", "prediction"),
  plotPoints = TRUE, plotLegend = TRUE, legendTitle = NULL,
  legendPct = TRUE, col = NULL, llwd = 2, opacity = 100, ...)
}
\arguments{
\item{model}{Required. A fitted Regression}

\item{plotx}{Required. Name of one predictor from the fitted
model to be plotted on horizontal axis}

\item{...}{Additional arguments passed to methods. Often includes
arguments that are passed to plot. Any arguments that customize
plot output, such as lwd, cex, and so forth, may be
supplied. These arguments intended for the predict method will be
used: c("type", "se.fit", "interval", "level", "dispersion",
"terms", "na.action")}

\item{modx}{Optional. String for moderator variable name. May be
either numeric or factor. If omitted, a single predicted value line
will be drawn.}

\item{n}{Optional. Number of focal values of
\code{modx}, used by algorithms specified by modxVals; will be
ignored if modxVals supplies a vector of focal values.}

\item{modxVals}{Optional. Focal values of \code{modx} for which
lines are desired. May be a vector of values or the name of an
algorithm, "quantile", "std.dev.", or "table".}

\item{plotxRange}{Optional. If not specified, the observed
range of plotx will be used to determine the axis range.}

\item{interval}{Optional. Intervals provided by the
\code{predict.lm} may be supplied, either "confidence" (confidence
interval for the estimated conditional mean) or "prediction"
(interval for observed values of y given the rest of the model).
The level can be specified as an argument (which goes into ...
and then to the predict method)}

\item{plotPoints}{Optional. TRUE or FALSE: Should the plot include
the scatterplot points along with the lines.}

\item{plotLegend}{Optional. TRUE or FALSE: Include a default}

\item{legendTitle}{Optional. You'll get an automatically generated
title, such as "Moderator: modx", but if you don't like that,
specify your own string here.  legend. Set to FALSE if user wants
to customize a legend after the plot has been drawn.}

\item{legendPct}{Default = TRUE. Variable labels print with sample percentages.}

\item{col}{Optional. A color vector for predicted value lines (and
intervals if requested). If not specified, the R's builtin palette()
will be used. User may supply a vector of valid color names,
either explicitly c("pink","black", "gray70") or implicitly,
rainbow(10) or gray.colors(5). Color names will be recycled if there
are more focal values of \code{modx} than colors provided.}

\item{llwd}{Optional, default = 2. Line widths for predicted values. Can be
single value or a vector, which will be recycled as necessary.}

\item{opacity}{Optional, default = 100. A number between 1 and 255.
1 means "transparent" or invisible, 255 means very dark.
Determines the darkness of confidence interval regions}
}
\value{
Creates a plot and an output object that summarizes it.

The return object includes the "newdata" object that was
used to create the plot, along with the "modxVals" vector, the
values of the moderator for which lines were drawn, and the color
vector. It also includes the call that generated the plot.
}
\description{
This is a generic function for plotting regression
objects. So far, there is an implementation for \code{lm()} objects.
This allows interaction effects, but not nonlinearities like log(x1).
For that, please see \code{plotCurves}.

This is a "simple slope" plotter for linear regression objects
that are created by \code{lm()}.  The function \code{plotCurves()}
can handle nonlinear predictive equations and generalized linear
models. The term "simple slopes" was coined by psychologists
(Aiken and West, 1991; Cohen, et al 2002) for analysis of
interaction effects for particular values of a moderating
variable. The moderating variable may be continuous or
categorical, lines will be plotted for focal values of that
variable.
}
\details{
This function works well with lm models in which the predictor
formula includes interactions, but it does not work well with
nonlinear predictors (log(x) and poly(x)).  For that, please use
\code{plotCurves}. plotSlopes is needed only when one wants to
create an output object that can be used as input for
\code{testSlopes()}.

The argument \code{plotx} is the name of the horizontal plotting
variable; it must be numeric.  The argument \code{modx} is the
moderator variable. It may be either a numeric or a factor
variable. As of version 1.7, the modx argument may be omitted. A
single predicted value line will be drawn. That version also
introduced the arguments interval and n.

There are many ways to specify focal values using the arguments
\code{modxVals} and \code{n}. This changed in rockchalk-1.7.0.  If
\code{modxVals} is omitted, a default algorithm will be used,
selecting \code{n} values for plotting. \code{modxVals} may be a
vector of values (for a numeric moderator) or levels (for a
factor).  If modxVals is a vector of values, then the argument
\code{n} is ignored.  However, if modxVals is one of the name of
one of the algorithms, "table", "quantile", or "std.dev.", then
the argument \code{n} sets number of focal values to be selected.
For numeric \code{modx}, n defaults to 3, but for factors
\code{modx} will be the number of observed values of
\code{modx}. If modxVals is omitted, the defaults will be used
("table" for factors, "quantile" for numeric variables).

For the predictors besides \code{modx} and \code{plotx} (the ones
that are not explicitly included in the plot), predicted values
are calculated with variables set to the mean and mode, for numeric
or factor variables (respectively). Those values can be reviewed
in the newdata object that is created as a part of the output from
this function
}
\examples{
## Manufacture some predictors
set.seed(12345)

dat <- genCorrelatedData2 (N = 100, means = rep(0,4), sds = 1, rho = 0.2,
                           beta = c(0.3, 0.5, -0.45, 0.5, -0.1, 0, 0.6),
                           stde = 2)

dat$xcat1 <- gl(2, 50, labels = c("M", "F"))
dat$xcat2 <- cut(rnorm(100), breaks = c(-Inf, 0, 0.4, 0.9, 1, Inf),
                 labels = c("R", "M", "D", "P", "G"))
## incorporate effect of categorical predictors
dat$y <- dat$y + 1.9 * dat$x1 * contrasts(dat$xcat1)[dat$xcat1] +
           contrasts(dat$xcat2)[dat$xcat2 , ] \%*\% c(0.1, -0.1, 0, 0.2)

m1 <- lm(y ~ x1 * x2 + x3 + x4 + xcat1 + xcat2, data = dat)
summary(m1)

## New in rockchalk 1.7.x. No modx required:
plotSlopes(m1, plotx = "x1")

## Confidence interval, anybody?
plotSlopes(m1, plotx = "x1", interval = "conf")

## Prediction interval.
plotSlopes(m1, plotx = "x1", interval = "pred")

## Now experiment with a moderator variable
## let default quantile algorithm do its job
plotSlopes(m1, plotx = "x1", modx = "x2")
## previous uses default equivalent to
## plotSlopes(m1, plotx = "x1", modx = "x2", modxVals = "quantile")
## Want more focal values?
plotSlopes(m1, plotx = "x1", modx = "x2", n = 5)
## Pick focal values yourself?
plotSlopes(m1, plotx = "x1", modx = "x2", modxVals = c(-2, 0, 0.5))
## Alternative algorithm?
plotSlopes(m1, plotx = "x1", modx = "x2", modxVals = "std.dev.",
           main = "Uses \\"std.dev.\\" Divider for the Moderator",
           xlab = "My Predictor", ylab = "Write Anything You Want for ylab")

## Will catch output object from this one
m1ps <- plotSlopes(m1, plotx = "x1", modx = "x2", modxVals = "std.dev.", n = 5,
                 main = "Setting n = 5 Selects More Focal Values for Plotting")

m1ts <- testSlopes(m1ps)

plot(m1ts)

### Examples with categorical Moderator variable

m3 <- lm (y ~ x1 + xcat1, data = dat)
summary(m3)
plotSlopes(m3, modx = "xcat1", plotx = "x1")
plotSlopes(m3, modx = "xcat1", plotx = "x1", interval = "predict")


m4 <- lm (y ~ x1 * xcat1, data = dat)
summary(m4)
plotSlopes(m4, modx = "xcat1", plotx = "x1")
plotSlopes(m4, modx = "xcat1", plotx = "x1", interval = "conf")


m5 <- lm (y ~ x1 + x2 + x1 * xcat2, data = dat)
summary(m5)
plotSlopes(m5, modx = "xcat2", plotx = "x1")
m5ps <- plotSlopes(m5, modx = "xcat2", plotx = "x1", interval = "conf")

testSlopes(m5ps)



## Now examples with real data. How about Chilean voters?
library(car)
m6 <- lm(statusquo ~ income * sex, data = Chile)
summary(m6)
plotSlopes(m6, modx = "sex", plotx = "income")
plotSlopes(m6, modx = "sex", plotx = "income", col = c("yellow", "blue"))


m7 <- lm(statusquo ~ region * income, data= Chile)
summary(m7)
plotSlopes(m7, plotx = "income", modx = "region")

plotSlopes(m7, plotx = "income", modx = "region", plotPoints = FALSE)
plotSlopes(m7, plotx = "income", modx = "region", plotPoints = FALSE,
           interval = "conf")
plotSlopes(m7, plotx = "income", modx = "region", modxVals = c("SA","S", "C"),
           plotPoints = FALSE, interval = "conf")
## Same, choosing 3 most frequent values
plotSlopes(m7, plotx = "income", modx = "region", n = 3, plotPoints = FALSE,
           interval = "conf")


m8 <- lm(statusquo ~ region * income + sex + age, data= Chile)
summary(m8)
plotSlopes(m8, modx = "region", plotx = "income")


m9 <- lm(statusquo ~ income * age + education + sex + age, data = Chile)
summary(m9)
plotSlopes(m9, modx = "income", plotx = "age")

plotSlopes(m9, modx = "income", plotx = "age", plotPoints = FALSE)


## Convert education to numeric, for fun
Chile$educationn <- as.numeric(Chile$education)
m10 <- lm(statusquo ~ income * educationn + sex + age, data = Chile)
summary(m10)
plotSlopes(m10, plotx = "educationn",  modx = "income")

## Now, the occupational prestige data. Please note careful attention
## to consistency of colors selected
data(Prestige)
m11 <- lm(prestige ~ education * type, data = Prestige)

plotSlopes(m11, plotx = "education", modx = "type", interval = "conf")
dev.new()
plotSlopes(m11, plotx = "education", modx = "type",
           modxVals = c("prof"), interval = "conf")
dev.new()
plotSlopes(m11, plotx = "education", modx = "type",
           modxVals = c("bc"), interval = "conf")
dev.new()
plotSlopes(m11, plotx = "education", modx = "type",
           modxVals = c("bc", "wc"), interval = "conf")
}
\author{
Paul E. Johnson <pauljohn@ku.edu>
}
\references{
Aiken, L. S. and West, S.G. (1991). Multiple Regression:
Testing and Interpreting Interactions. Newbury Park, Calif: Sage Publications.

Cohen, J., Cohen, P., West, S. G., and Aiken, L. S. (2002).
Applied Multiple Regression/Correlation Analysis for the Behavioral
Sciences (Third.). Routledge Academic.
}
\seealso{
\code{\link[rockchalk]{testSlopes}} \code{\link[rockchalk]{plotCurves}}
}

