# ----------------------
# Author: Andreas Alfons
#         KU Leuven
# ----------------------

## workhorse function for classes inheriting from "seqModel"
cvSeqModel <- function(
    x,  # matrix of predictors
    y,  # response
    cost = rmspe, 
    K = 5, 
    R = 1, 
    foldType = c("random", "consecutive", "interleaved"), 
    folds = NULL, 
    selectBest = c("min", "hastie"), 
    seFactor = 1, 
    active,            # integer vector giving the sequence of predictors
    s = NULL,          # integer vector giving the steps of the submodels to be fitted
    groupwise = TRUE,  # logical indicating whether groupwise submodels should be fitted
    assign = NULL,     # integer vector giving the group assignment of the variables
    robust = FALSE,    # logical indicating whether methods are robust
    regFun = lm.fit,   # regression function
    regArgs = list(),  # additional arguments for regression function
    class,             # character string giving the class of the return object
    seed = NULL, 
    ...
) {
    ## initializations
    # if argument 'assign' is not supplied for groupwise models, check if 
    # predictor matrix has attribute "assign" (as generated by model.matrix())
    if(groupwise && is.null(assign)) {
        assign <- attr(x, "assign")
        if(is.null(assign)) {
            # take each predictor to be its own group
            assign <- seq_len(ncol(x))
        }
    }
    # check if the predictor matrix contains column for intercept and 
    # remove it if necessary
    if(isTRUE(assign[1] == 0)) assign <- assign[-1]
    n <- length(y)
    x <- addIntercept(as.matrix(x), check=TRUE)
    if(!isTRUE(n == nrow(x))) stop(sprintf("'x' must have %d rows", n))
    if(!is.null(s)) {
        s <- checkSteps(s, 0, length(active))  # check steps
        s <- sort(unique(s))  # make sure steps are in the correct order
    }
    ## construct call to fit models in cross-validation
#    call <- call("fitModels", s=s, assign=assign, robust=robust, 
#        active=active, final=FALSE, class=class)
    call <- as.call(list(fitModels, s=s, assign=assign, robust=robust, 
            active=active, final=FALSE, class=class))
    if(robust) {
        regControl <- getRegControl(regFun)
        call$regFun <- regControl$fun  # if possible, do not use formula interface
        call$useFormula <- regControl$useFormula
        call$regArgs <- regArgs
    }
    ## call function cvFit() to perform cross-validation
    out <- cvFit(call, x=x, y=y, cost=cost, K=K, R=R, foldType=foldType, 
        folds=folds, costArgs=list(...), envir=parent.frame(), seed=seed)
    out <- cvReshape(out, selectBest=selectBest, seFactor=seFactor)
    fits(out) <- as.numeric(as.character(fits(out)))
    class(out) <- c("cvSeqModel", class(out))
    out
}


#' Cross-validation along a robust least angle regression sequence
#' 
#' Estimate the prediction error of submodels along a robust least angle 
#' regression sequence via (repeated) \eqn{K}-fold cross-validation.
#' 
#' @param formula  a formula describing the full model.
#' @param data  an optional data frame, list or environment (or object coercible 
#' to a data frame by \code{\link{as.data.frame}}) containing the variables in 
#' the model.  If not found in data, the variables are taken from 
#' \code{environment(formula)}, typically the environment from which 
#' \code{cvRlars} is called.
#' @param x  a matrix or data frame containing the candidate predictors.
#' @param y  a numeric vector containing the response.
#' @param cost  a robust cost function measuring prediction loss.  It should 
#' expect vectors to be passed as its first two arguments, the first 
#' corresponding to the observed values of the response and the second to the 
#' predicted values, and must return a non-negative scalar value.  The default 
#' is to use the root trimmed mean squared prediction error (see 
#' \code{\link[cvTools]{cost}}).
#' @param K  an integer giving the number of groups into which the data should 
#' be split (the default is five).  Keep in mind that this should be chosen 
#' such that all groups are of approximately equal size.  Setting \code{K} 
#' equal to \code{n} yields leave-one-out cross-validation.
#' @param R  an integer giving the number of replications for repeated 
#' \eqn{K}-fold cross-validation.  This is ignored for for leave-one-out 
#' cross-validation and other non-random splits of the data.
#' @param foldType  a character string specifying the type of folds to be 
#' generated.  Possible values are \code{"random"} (the default), 
#' \code{"consecutive"} or \code{"interleaved"}.
#' @param folds  an object of class \code{"cvFolds"} giving the folds of the 
#' data for cross-validation (as returned by 
#' \code{\link[cvTools]{cvFolds}}).  If supplied, this is preferred over 
#' \code{K} and \code{R}.
#' @param selectBest  a character string specifying a criterion for selecting 
#' the best model.  Possible values are \code{"min"} (the default) or 
#' \code{"hastie"}.  The former selects the model with the smallest prediction 
#' error.  The latter selects the most parsimonious model whose prediction 
#' error is no larger than \code{seFactor} standard errors above the prediction 
#' error of the best overall model.
#' @param seFactor  a numeric value giving a multiplication factor of the 
#' standard error for the selection of the best model.  This is ignored if 
#' \code{selectBest} is \code{"min"}.
#' @param active  an integer vector containing the sequence of predictor 
#' groups (as returned by \code{\link{rlars}}).
#' @param s  an integer vector giving the steps of the submodels for which to 
#' estimate the prediction errors (the default is to use all steps along the 
#' sequence as long as there are twice as many observations as predictors).
#' @param regFun  a function to compute robust linear regressions (defaults to 
#' \code{\link[robustbase]{lmrob}}).
#' @param regArgs  a list of arguments to be passed to \code{regFun}.
#' @param seed  optional initial seed for the random number generator (see 
#' \code{\link{.Random.seed}}).
#' @param \dots  additional arguments to be passed to the prediction loss 
#' function \code{cost}.
#' 
#' @return An object of class \code{"cvSeqModel"} (which inherits from class 
#' \code{"cvSelect"}) with the following components:
#' @returnItem n  an integer giving the number of observations.
#' @returnItem K  an integer giving the number of folds used in 
#' cross-validation.
#' @returnItem R  an integer giving the number of replications used in 
#' cross-validation.
#' @returnItem best  an integer giving the index of the submodel with the best 
#' prediction performance.
#' @returnItem cv  a data frame containing the estimated prediction errors for 
#' the submodels.  For repeated cross-validation, those are average values over 
#' all replications.
#' @returnItem se  a data frame containing the estimated standard errors of the 
#' prediction loss for the submodels.
#' @returnItem selectBest  a character string specifying the criterion used for 
#' selecting the best model.
#' @returnItem seFactor  a numeric value giving the multiplication factor of 
#' the standard error used for the selection of the best model.
#' @returnItem reps  a data frame containing the estimated prediction errors 
#' for the submodels from all replications.  This is only returned for repeated 
#' cross-validation.
#' @returnItem call  the matched function call.
#' 
#' @author Andreas Alfons
#' 
#' @seealso \code{\link{repCV.rlars}}, \code{\link{rlars}}, 
#' \code{\link{predict.rlars}}, \code{\link[cvTools]{cvFolds}}, 
#' \code{\link[cvTools]{cost}}
#' 
#' @example inst/doc/examples/example-cvRlars.R
#' 
#' @keywords utilities robust
#' 
#' @export
#' @import cvTools

cvRlars <- function(x, ...) UseMethod("cvRlars")


#' @rdname cvRlars
#' @method cvRlars formula
#' @export

cvRlars.formula <- function(formula, data, ...) {
    ## initializations
    call <- match.call()  # get function call
    call[[1]] <- as.name("cvRlars")
    # prepare model frame
    mf <- match.call(expand.dots = FALSE)
    m <- match(c("formula", "data"), names(mf), 0)
    mf <- mf[c(1, m)]
    mf$drop.unused.levels <- TRUE
    mf[[1]] <- as.name("model.frame")
    mf <- eval(mf, parent.frame())
    mt <- attr(mf, "terms")
    if(is.empty.model(mt)) stop("empty model")
    # extract response and candidate predictors from model frame
    y <- model.response(mf, "numeric")
    x <- model.matrix(mt, mf)
    # remove first column for intercept, if existing
    if(attr(mt, "intercept")) x <- x[, -1, drop=FALSE]
    ## call default method
    out <- cvRlars.default(x, y, ...)
    out$call <- call  # add call to return object
    out
}

#' @rdname cvRlars
#' @method cvRlars default
#' @export

cvRlars.default <- function(x, y, cost = rtmspe, K = 5, R = 1, 
        foldType = c("random", "consecutive", "interleaved"), folds = NULL, 
        selectBest = c("min", "hastie"), seFactor = 1, active, s = NULL, 
        regFun = lmrob, regArgs = list(), seed = NULL, ...) {
    ## initializations
    matchedCall <- match.call()
    matchedCall[[1]] <- as.name("cvRlars")
    ## call workhorse function and modify return object
    out <- cvSeqModel(x, y, cost=cost, K=K, R=R, foldType=foldType, 
        folds=folds, selectBest=selectBest, seFactor=seFactor, active=active, 
        s=s, groupwise=FALSE, robust=TRUE, regFun=regFun, regArgs=regArgs, 
        class="rlars", seed=seed, ...)
    out$call <- matchedCall
    out
}


#' Cross-validation for sparse LTS regression models
#' 
#' Estimate the prediction error of sparse least trimmed squares regression 
#' models for various values of the penalty parameter via (repeated) 
#' \eqn{K}-fold cross-validation.
#' 
#' @aliases print.cvSparseLTS
#' 
#' @param formula  a formula describing the model.
#' @param data  an optional data frame, list or environment (or object 
#' coercible to a data frame by \code{\link{as.data.frame}}) containing the 
#' variables in the model.  If not found in data, the variables are taken 
#' from \code{environment(formula)}, typically the environment from which 
#' \code{cvSparseLTS} is called.
#' @param x  a numeric matrix containing the predictor variables.
#' @param y  a numeric vector containing the response variable.
#' @param cost  a robust cost function measuring prediction loss.  It should 
#' expect vectors to be passed as its first two arguments, the first 
#' corresponding to the observed values of the response and the second to the 
#' predicted values, and must return a non-negative scalar value.  The default 
#' is to use the root trimmed mean squared prediction error (see 
#' \code{\link[cvTools]{cost}}).
#' @param K  an integer giving the number of groups into which the data should 
#' be split (the default is five).  Keep in mind that this should be chosen 
#' such that all groups are of approximately equal size.  Setting \code{K} 
#' equal to \code{n} yields leave-one-out cross-validation.
#' @param R  an integer giving the number of replications for repeated 
#' \eqn{K}-fold cross-validation.  This is ignored for for leave-one-out 
#' cross-validation and other non-random splits of the data.
#' @param foldType  a character string specifying the type of folds to be 
#' generated.  Possible values are \code{"random"} (the default), 
#' \code{"consecutive"} or \code{"interleaved"}.
#' @param folds  an object of class \code{"cvFolds"} giving the folds of the 
#' data for cross-validation (as returned by 
#' \code{\link[cvTools]{cvFolds}}).  If supplied, this is preferred over 
#' \code{K} and \code{R}.
#' @param fit  a character string specifying for which fit to estimate the 
#' prediction error.  Possible values are \code{"reweighted"} (the default) for 
#' the prediction error of the reweighted fit, \code{"raw"} for the prediction 
#' error of the raw fit, or \code{"both"} for the prediction error of both 
#' fits.
#' @param selectBest  a character string specifying a criterion for selecting 
#' the best model.  Possible values are \code{"min"} (the default) or 
#' \code{"hastie"}.  The former selects the model with the smallest prediction 
#' error.  The latter selects the model with the largest penalty parameter 
#' whose prediction error is no larger than \code{seFactor} standard errors 
#' above the prediction error of the best overall model.
#' @param seFactor  a numeric value giving a multiplication factor of the 
#' standard error for the selection of the best model.  This is ignored if 
#' \code{selectBest} is \code{"min"}.
#' @param lambda  a numeric vector of non-negative values giving the penalty 
#' parameters for which to estimate the prediction error.
#' @param mode  a character string specifying the type of penalty parameter.  If 
#' \code{"lambda"}, this gives penalty parameter directly.  If 
#' \code{"fraction"}, the smallest value of the penalty parameter that sets all 
#' coefficients to 0 is first estimated based on bivariate winsorization, then 
#' \code{lambda} gives the fraction of that estimate to be used (hence 
#' \code{lambda} should be in the interval [0,1] in that case).
#' @param alpha  a numeric value giving the percentage of the residuals for 
#' which the \eqn{L_{1}}{L1} penalized sum of squares should be minimized (the 
#' default is 0.75).
#' @param intercept  a logical indicating whether a constant term should be 
#' included in the model (the default is \code{TRUE}).
#' @param nsamp  a numeric vector giving the number of subsamples to be used in 
#' the two phases of the algorithm.  The first element gives the number of 
#' initial subsamples to be used.  The second element gives the number of 
#' subsamples to keep after the first phase of \code{ncstep} C-steps.  For 
#' those remaining subsets, additional C-steps are performed until 
#' convergence.  The default is to first perform \code{ncstep} C-steps on 500 
#' initial subsamples, and then to keep the 10 subsamples with the lowest value 
#' of the objective function for additional C-steps until convergence.
#' @param initial  a character string specifying the type of initial subsamples 
#' to be used.  If \code{"sparse"}, the lasso fit given by three randomly 
#' selected data points is first computed.  The corresponding initial subsample 
#' is then formed by the fraction \code{alpha} of data points with the smallest 
#' squared residuals.  Note that this is optimal from a robustness point of 
#' view, as the probability of including an outlier in the initial lasso fit is 
#' minimized.  If \code{"hyperplane"}, a hyperplane through \eqn{p} randomly 
#' selected data points is first computed, where \eqn{p} denotes the number of 
#' variables.  The corresponding initial subsample is then again formed by the 
#' fraction \code{alpha} of data points with the smallest squared residuals.  
#' Note that this cannot be applied if \eqn{p} is larger than the number of 
#' observations.  Nevertheless, the probability of including an outlier 
#' increases with increasing dimension \eqn{p}.  If \code{"random"}, the 
#' initial subsamples are given by a fraction \code{alpha} of randomly 
#' selected data points.  Note that this leads to the largest probability of 
#' including an outlier.
#' @param ncstep  a positive integer giving the number of C-steps to perform on 
#' all subsamples in the first phase of the algorithm (the default is to 
#' perform two C-steps).
#' @param use.correction  currently ignored.  The consistency factor for the 
#' residual scale estimate is always applied.
#' @param tol  a small positive numeric value giving the tolerance for 
#' convergence.
#' @param eps  a small positive numeric value used to determine whether the 
#' variability within a variable is too small (an effective zero).
#' @param use.Gram  a logical indicating whether the Gram matrix of the 
#' explanatory variables should be precomputed in the lasso fits (the default 
#' is \code{TRUE}).  If the number of variables is large (e.g., larger than the 
#' number of observations), computation may be faster when this is set to 
#' \code{FALSE}.
#' @param centerFun  a function to compute a robust estimate for the center to 
#' be used for robust standardization (defaults to 
#' \code{\link[stats]{median}}).  Ignored if \code{standardized} is \code{TRUE}.
#' @param scaleFun  a function to compute a robust estimate for the scale to 
#' be used for robust standardization (defaults to \code{\link[stats]{mad}}).  
#' Ignored if \code{standardized} is \code{TRUE}.
#' @param const  numeric; tuning constant to be used in univariate 
#' winsorization (defaults to 2).
#' @param prob  numeric; probability for the quantile of the 
#' \eqn{\chi^{2}}{chi-squared} distribution to be used in multivariate
#' winsorization (defaults to 0.95).
#' @param fallback  a logical indicating whether standardization with 
#' \code{\link{mean}} and \code{\link[stats]{sd}} should be performed as a 
#' fallback mode for variables whose robust scale estimate is too small.  This 
#' is useful, e.g., for data containing dummy variables.
#' @param seed  optional initial seed for the random number generator (see 
#' \code{\link{.Random.seed}}).
#' @param \dots  additional arguments to be passed to the prediction loss 
#' function \code{cost}.
#' 
#' @return An object of class \code{"cvSparseLTS"} (which inherits from classes 
#' \code{"cvTuning"} and \code{"cvSelect"}) with the following components:
#' @returnItem n  an integer giving the number of observations.
#' @returnItem K  an integer giving the number of folds.
#' @returnItem R  an integer giving the number of replications.
#' @returnItem tuning  a data frame containing the values of the penalty 
#' parameter for which the prediction error was estimated.
#' @returnItem best  an integer vector giving the indices of the optimal 
#' penalty parameter for the requested model fits.
#' @returnItem cv  a data frame containing the estimated prediction errors of 
#' the requested model fits for all values of the penalty parameter.
#' @returnItem se  a data frame vector containing the estimated standard errors 
#' of the prediction loss for the requested model fits for all values of the 
#' penalty parameter.
#' @returnItem selectBest  a character string specifying the criterion used for 
#' selecting the best model.
#' @returnItem seFactor  a numeric value giving the multiplication factor of 
#' the standard error used for the selection of the best model.
#' @returnItem reps  a data frame containing the estimated prediction errors of 
#' the requested model fits from all replications and for all values of the 
#' penalty parameter.  This is only returned for repeated cross-validation.
#' @returnItem seed  the seed of the random number generator before 
#' cross-validation was performed.
#' @returnItem call  the matched function call.
#' 
#' @author Andreas Alfons
#' 
#' @seealso \code{\link{repCV.sparseLTS}}, \code{\link{sparseLTS}}, 
#' \code{\link{predict.sparseLTS}}, \code{\link[cvTools]{cvFolds}}, 
#' \code{\link[cvTools]{cost}}
#' 
#' @example inst/doc/examples/example-cvSparseLTS.R
#' 
#' @keywords utilities
#' 
#' @export
#' @import cvTools

cvSparseLTS <- function(x, ...) UseMethod("cvSparseLTS")


#' @rdname cvSparseLTS
#' @method cvSparseLTS formula
#' @export

cvSparseLTS.formula <- function(formula, data, ...) {
    # get function call
    matchedCall <- match.call()
    matchedCall[[1]] <- as.name("cvSparseLTS")
    # prepare model frame
    mf <- match.call(expand.dots = FALSE)
    m <- match(c("formula", "data"), names(mf), 0)
    mf <- mf[c(1, m)]
    mf$drop.unused.levels <- TRUE
    mf[[1]] <- as.name("model.frame")
    mf <- eval(mf, parent.frame())
    mt <- attr(mf, "terms")
    if(is.empty.model(mt)) stop("empty model")
    # extract response and candidate predictors from model frame
    y <- model.response(mf, "numeric")
    x <- model.matrix(mt, mf)
    # check if the specified model contains an intercept
    # if so, remove the column for intercept and use 'intercept=TRUE'
    # otherwise use 'intercept=FALSE'
    whichIntercept <- match("(Intercept)", colnames(x), nomatch = 0)
    intercept <- whichIntercept > 0
    if(intercept) x <- x[, -whichIntercept, drop = FALSE]
    # call default method
    fit <- cvSparseLTS(x, y, intercept=intercept, ...)
    fit$call <- matchedCall  # add call to return object
    fit
}


#' @rdname cvSparseLTS
#' @method cvSparseLTS default
#' @export

cvSparseLTS.default <- function(x, y, cost = rtmspe, K = 5, R = 1, 
        foldType = c("random", "consecutive", "interleaved"), folds = NULL, 
        fit = c("reweighted", "raw", "both"), selectBest = c("min", "hastie"), 
        seFactor = 1, lambda, mode = c("lambda", "fraction"), alpha = 0.75, 
        intercept = TRUE, nsamp = c(500, 10), 
        initial = c("sparse", "hyperplane", "random"), 
        ncstep = 2, use.correction = TRUE, tol = .Machine$double.eps^0.5, 
        eps = .Machine$double.eps, use.Gram = TRUE, centerFun = median, 
        scaleFun = mad, const = 2, prob = 0.95, fallback = FALSE, 
        seed = NULL, ...) {
    ## initializations
    matchedCall <- match.call()
    matchedCall[[1]] <- as.name("cvSparseLTS")
    n <- length(y)
    x <- as.matrix(x)
    if(!isTRUE(n == nrow(x))) stop(sprintf("'x' must have %d rows", n))
    if(missing(lambda)) {
        # if penalty parameter is not supplied, use a small fraction of a 
        # robust estimate of the smallest value that sets all coefficients 
        # to zero
        lower <- if(nrow(x) > ncol(x)) 0 else 0.1
        lambda <- seq(from=1, to=lower, by=-0.1)
        mode <- "fraction"
    } else {
        # otherwise check the supplied penalty parameter
        if(!is.numeric(lambda) || length(lambda) == 0 || any(!is.finite(lambda))) {
            stop("missing or invalid value of 'lambda'")
        }
        if(any(negative <- lambda < 0)) {
            lambda[negative] <- 0
            warning("negative value for 'lambda', using no penalization")
        }
        mode <- match.arg(mode)
        lambda <- sort(unique(lambda), decreasing=TRUE)
    }
    ## call function cvTuning() to perform cross-validation
    call <- call("sparseLTS", mode=mode, alpha=alpha, intercept=intercept, 
        nsamp=nsamp, initial=initial, ncstep=ncstep, 
        use.correction=use.correction, tol=tol, eps=eps, use.Gram=use.Gram, 
        model=TRUE, centerFun=centerFun, scaleFun=scaleFun, const=const, 
        prob=prob, fallback=fallback)
    out <- cvTuning(call, x=x, y=y, tuning=list(lambda=lambda), cost=cost, 
        K=K, R=R, foldType=foldType, folds=folds, predictArgs=list(fit=fit), 
        costArgs=list(...), selectBest=selectBest, seFactor=seFactor, seed=seed)
    if(mode == "fraction" && any(lambda > 0)) {
        # penalty parameters supplied as fractions, make sure that result 
        # contains absolute values
        out$tuning$lambda <- lambda * lambda0(x, y, intercept=intercept, 
            const=const, prob=prob, tol=tol, eps=eps, centerFun=centerFun, 
            scaleFun=scaleFun, fallback=fallback)
    }
    out$call <- matchedCall
    class(out) <- c("cvSparseLTS", class(out))
    out
}
