% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/quasiquotation.R
\name{quasiquotation}
\alias{quasiquotation}
\alias{UQ}
\alias{UQS}
\alias{!!}
\alias{!!!}
\alias{:=}
\alias{qq_show}
\title{Quasiquotation of an expression}
\usage{
qq_show(expr)
}
\arguments{
\item{expr}{An expression to be quasiquoted.}
}
\description{
Quasiquotation is the mechanism that makes it possible to program
flexibly with tidy evaluation grammars like dplyr. It is enabled in
all functions quoting their arguments with \code{enquo()}, \code{enexpr()},
or the plural variants.

Quasiquotation is the combination of quoting an expression while
allowing immediate evaluation (unquoting) of part of that
expression. We provide both syntactic operators and functional
forms for unquoting.
\itemize{
\item The \code{!!} operator unquotes its argument. It gets evaluated
immediately in the surrounding context.
\item The \code{!!!} operator unquotes and splices its argument. The
argument should represent a list or a vector. Each element will
be embedded in the surrounding call, i.e. each element is
inserted as an argument. If the vector is named, the names are
used as argument names.

If the vector is a classed object (like a factor), it is
converted to a list with \code{\link[base:as.list]{base::as.list()}} to ensure proper
dispatch. If it is an S4 objects, it is converted to a list with
\code{\link[methods:as]{methods::as()}}.
\item The \code{{{ }}} operator quotes and unquotes its argument in one
go, a pattern that we call \strong{interpolation}. It is an alias for
\code{!!enquo(arg)}.

Like \code{enquo()}, \code{{{ }}} is used in functions to capture an
argument as a quoted expression. This expression is immediately
unquoted in place.
}

Use \code{qq_show()} to experiment with quasiquotation or debug the
effect of unquoting operators. \code{qq_show()} quotes its input,
processes unquoted parts, and prints the result with
\code{\link[=expr_print]{expr_print()}}. This expression printer has a clearer output than
the base R printer (see the \link[=expr_print]{documentation topic}).
}
\section{Unquoting names}{


When a function takes multiple named arguments
(e.g. \code{dplyr::mutate()}), it is difficult to supply a variable as
name. Since the LHS of \code{=} is quoted, giving the name of a variable
results in the argument having the name of the variable rather than
the name stored in that variable. This problem is right up the
alley for the unquoting operator \code{!!}. If you were able to unquote
the variable when supplying the name, the argument would be named
after the content of that variable.

Unfortunately R is very strict about the kind of expressions
supported on the LHS of \code{=}. This is why we have made the more
flexible \code{:=} operator an alias of \code{=}. You can use it to supply
names, e.g. \code{a := b} is equivalent to \code{a = b}. Since its syntax is
more flexible you can unquote on the LHS:\preformatted{name <- "Jane"

list2(!!name := 1 + 2)
exprs(!!name := 1 + 2)
quos(!!name := 1 + 2)
}

Like \code{=}, the \code{:=} operator expects strings or symbols on its LHS.

Note that unquoting on the LHS of \code{:=} only works in top level
expressions. These are all valid:\preformatted{exprs(!!nm := x)
tibble(!!nm := x)
list2(!!nm := x)
}

But deep-unquoting names isn't supported:\preformatted{expr(foo(!!nm := x))
exprs(foo(!!nm := x))
}
}

\section{Theory}{


Formally, \code{quo()} and \code{expr()} are quasiquote functions, \code{!!} is
the unquote operator, and \code{!!!} is the unquote-splice operator.
These terms have a rich history in Lisp languages, and live on in
modern languages like
\href{https://docs.julialang.org/en/v1/manual/metaprogramming/}{Julia}
and
\href{https://docs.racket-lang.org/reference/quasiquote.html}{Racket}.
}

\section{Life cycle}{

\itemize{
\item Calling \code{UQ()} and \code{UQS()} with the rlang namespace qualifier is
deprecated as of rlang 0.3.0. Just use the unqualified forms
instead:\preformatted{# Bad
rlang::expr(mean(rlang::UQ(var) * 100))

# Ok
rlang::expr(mean(UQ(var) * 100))

# Good
rlang::expr(mean(!!var * 100))
}

Supporting namespace qualifiers complicates the implementation of
unquotation and is misleading as to the nature of unquoting
operators (which are syntactic operators that operates at
quotation-time rather than function calls at evaluation-time).
\item \code{UQ()} and \code{UQS()} were soft-deprecated in rlang 0.2.0 in order
to make the syntax of quasiquotation more consistent. The prefix
forms are now \code{`!!`()} and \code{`!!!`()} which is
consistent with other R operators (e.g. \code{`+`(a, b)} is the
prefix form of \code{a + b}).

Note that the prefix forms are not as relevant as before because
\code{!!} now has the right operator precedence, i.e. the same as
unary \code{-} or \code{+}. It is thus safe to mingle it with other
operators, e.g. \code{!!a + !!b} does the right thing. In addition the
parser now strips one level of parentheses around unquoted
expressions. This way \code{(!!"foo")(...)} expands to \code{foo(...)}.
These changes make the prefix forms less useful.

Finally, the named functional forms \code{UQ()} and \code{UQS()} were
misleading because they suggested that existing knowledge about
functions is applicable to quasiquotation. This was reinforced by
the visible definitions of these functions exported by rlang and
by the tidy eval parser interpreting \code{rlang::UQ()} as \code{!!}. In
reality unquoting is \emph{not} a function call, it is a syntactic
operation. The operator form makes it clearer that unquoting is
special.
}
}

\examples{
# Interpolation with {{  }} is the easiest way to forward
# arguments to tidy eval functions:
if (is_attached("package:dplyr")) {

# Forward all arguments involving data frame columns by
# interpolating them within other data masked arguments.
# Here we interpolate `arg` in a `summarise()` call:
my_function <- function(data, arg) {
  summarise(data, avg = mean({{ arg }}, na.rm = TRUE))
}

my_function(mtcars, cyl)
my_function(mtcars, cyl * 10)

# The  operator is just a shortcut for `!!enquo()`:
my_function <- function(data, arg) {
  summarise(data, avg = mean(!!enquo(arg), na.rm = TRUE))
}

my_function(mtcars, cyl)

}

# Quasiquotation functions quote expressions like base::quote()
quote(how_many(this))
expr(how_many(this))
quo(how_many(this))

# In addition, they support unquoting. Let's store symbols
# (i.e. object names) in variables:
this <- sym("apples")
that <- sym("oranges")

# With unquotation you can insert the contents of these variables
# inside the quoted expression:
expr(how_many(!!this))
expr(how_many(!!that))

# You can also insert values:
expr(how_many(!!(1 + 2)))
quo(how_many(!!(1 + 2)))


# Note that when you unquote complex objects into an expression,
# the base R printer may be a bit misleading. For instance compare
# the output of `expr()` and `quo()` (which uses a custom printer)
# when we unquote an integer vector:
expr(how_many(!!(1:10)))
quo(how_many(!!(1:10)))

# This is why it's often useful to use qq_show() to examine the
# result of unquotation operators. It uses the same printer as
# quosures but does not return anything:
qq_show(how_many(!!(1:10)))


# Use `!!!` to add multiple arguments to a function. Its argument
# should evaluate to a list or vector:
args <- list(1:3, na.rm = TRUE)
quo(mean(!!!args))

# You can combine the two
var <- quote(xyz)
extra_args <- list(trim = 0.9, na.rm = TRUE)
quo(mean(!!var , !!!extra_args))


# The plural versions have support for the `:=` operator.
# Like `=`, `:=` creates named arguments:
quos(mouse1 := bernard, mouse2 = bianca)

# The `:=` is mainly useful to unquote names. Unlike `=` it
# supports `!!` on its LHS:
var <- "unquote me!"
quos(!!var := bernard, mouse2 = bianca)


# All these features apply to dots captured by enquos():
fn <- function(...) enquos(...)
fn(!!!args, !!var := penny)


# Unquoting is especially useful for building an expression by
# expanding around a variable part (the unquoted part):
quo1 <- quo(toupper(foo))
quo1

quo2 <- quo(paste(!!quo1, bar))
quo2

quo3 <- quo(list(!!quo2, !!!syms(letters[1:5])))
quo3
}
