% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/comp_accu.R
\name{comp_accu_prob}
\alias{comp_accu_prob}
\title{Compute exact accuracy metrics based on probabilities.}
\usage{
comp_accu_prob(prev = prob$prev, sens = prob$sens, mirt = NA,
  spec = prob$spec, fart = NA, tol = 0.01, w = 0.5)
}
\arguments{
\item{prev}{The condition's prevalence \code{\link{prev}}
(i.e., the probability of condition being \code{TRUE}).}

\item{sens}{The decision's sensitivity \code{\link{sens}}
(i.e., the conditional probability of a positive decision
provided that the condition is \code{TRUE}).
\code{sens} is optional when its complement \code{mirt} is provided.}

\item{mirt}{The decision's miss rate \code{\link{mirt}}
(i.e., the conditional probability of a negative decision
provided that the condition is \code{TRUE}).
\code{mirt} is optional when its complement \code{sens} is provided.}

\item{spec}{The decision's specificity value \code{\link{spec}}
(i.e., the conditional probability
of a negative decision provided that the condition is \code{FALSE}).
\code{spec} is optional when its complement \code{fart} is provided.}

\item{fart}{The decision's false alarm rate \code{\link{fart}}
(i.e., the conditional probability
of a positive decision provided that the condition is \code{FALSE}).
\code{fart} is optional when its complement \code{spec} is provided.}

\item{tol}{A numeric tolerance value for \code{\link{is_complement}}.
Default: \code{tol = .01}.}

\item{w}{The weighting parameter \code{w} (from 0 to 1)
for computing weighted accuracy \code{wacc}.
Default: \code{w = .50} (i.e., yielding balanced accuracy \code{bacc}).

Notes:

\itemize{

   \item Accuracy metrics describe the \emph{correspondence} of decisions (or predictions) to actual conditions (or truth).

   There are several possible interpretations of accuracy:

   \enumerate{

     \item as \emph{probabilities} (i.e., \code{acc} being the proportion of correct classifications,
     or the ratio \code{\link{dec_cor}}/\code{\link{N}}),

     \item as \emph{frequencies} (e.g., as classifying a population of \code{\link{N}}
     individuals into cases of \code{\link{dec_cor}} vs. \code{\link{dec_err}}),

     \item as \emph{correlations} (e.g., see \code{mcc} in \code{\link{accu}}).

   }

   \item Computing exact accuracy values based on probabilities (by \code{\link{comp_accu_prob}}) may differ from
   accuracy values computed from (possibly rounded) frequencies (by \code{\link{comp_accu_freq}}).

   When frequencies are rounded to integers (see the default of \code{round = TRUE}
   in \code{\link{comp_freq}} and \code{\link{comp_freq_prob}}) the accuracy metrics computed by
   \code{comp_accu_freq} correspond to these rounded values.
   Use \code{\link{comp_accu_prob}} to obtain exact accuracy metrics from probabilities.

   }}
}
\value{
A list \code{\link{accu}} containing current accuracy metrics.
}
\description{
\code{comp_accu_prob} computes a list of exact accuracy metrics
from a sufficient and valid set of 3 essential probabilities
(\code{\link{prev}}, and
\code{\link{sens}} or its complement \code{\link{mirt}}, and
\code{\link{spec}} or its complement \code{\link{fart}}).
}
\details{
Currently computed accuracy metrics include:

\enumerate{

\item \code{acc}: Overall accuracy as the proportion (or probability)
   of correctly classifying cases or of \code{\link{dec_cor}} cases:

(a) from \code{\link{prob}}: \code{acc = (prev x sens) + [(1 - prev) x spec]}

(b) from \code{\link{freq}}: \code{acc = dec_cor/N = (hi + cr)/(hi + mi + fa + cr)}

When frequencies in \code{\link{freq}} are not rounded, (b) coincides with (a).

Values range from 0 (no correct prediction) to 1 (perfect prediction).

\item \code{wacc}: Weighted accuracy, as a weighted average of the
   sensitivity \code{\link{sens}} (aka. hit rate \code{\link{HR}}, \code{\link{TPR}},
   \code{\link{power}} or \code{\link{recall}})
   and the the specificity \code{\link{spec}} (aka. \code{\link{TNR}})
   in which \code{\link{sens}} is multiplied by a weighting parameter \code{w}
   (ranging from 0 to 1) and \code{\link{spec}} is multiplied by
   \code{w}'s complement \code{(1 - w)}:

\code{wacc = (w * sens) + ((1 - w) * spec)}

If \code{w = .50}, \code{wacc} becomes \emph{balanced} accuracy \code{bacc}.

\item \code{mcc}: The Matthews correlation coefficient (with values ranging from -1 to +1):

\code{mcc = ((hi * cr) - (fa * mi)) / sqrt((hi + fa) * (hi + mi) * (cr + fa) * (cr + mi))}

A value of \code{mcc = 0} implies random performance; \code{mcc = 1} implies perfect performance.

See \href{https://en.wikipedia.org/wiki/Matthews_correlation_coefficient}{Wikipedia: Matthews correlation coefficient}
   for additional information.

\item \code{f1s}: The harmonic mean of the positive predictive value \code{\link{PPV}}
   (aka. \code{\link{precision}})
   and the sensitivity \code{\link{sens}} (aka. hit rate \code{\link{HR}},
   \code{\link{TPR}}, \code{\link{power}} or \code{\link{recall}}):

\code{f1s =  2 * (PPV * sens) / (PPV + sens)}

See \href{https://en.wikipedia.org/wiki/F1_score}{Wikipedia: F1 score} for additional information.

}

Note that some accuracy metrics can be interpreted
as probabilities (e.g., \code{acc}) and some as correlations (e.g., \code{mcc}).

Also, accuracy can be viewed as a probability (e.g., the ratio of or link between
\code{\link{dec_cor}} and \code{\link{N}}) or as a frequency type
(containing \code{\link{dec_cor}} and \code{\link{dec_err}}).

\code{comp_accu_prob} computes exact accuracy metrics from probabilities.
When input frequencies were rounded (see the default of \code{round = TRUE}
in \code{\link{comp_freq}} and \code{\link{comp_freq_prob}}) the accuracy
metrics computed by \code{comp_accu} correspond these rounded values.
}
\examples{
comp_accu_prob()  # => accuracy metrics for prob of current scenario
comp_accu_prob(prev = .2, sens = .5, spec = .5)  # medium accuracy, but cr > hi.

# Extreme cases:
comp_accu_prob(prev = NaN, sens = NaN, spec = NaN)  # returns list of NA values
comp_accu_prob(prev = 0, sens = NaN, spec = 1)      # returns list of NA values
comp_accu_prob(prev = 0, sens = 0, spec = 1)     # perfect acc = 1, but f1s is NaN
comp_accu_prob(prev = .5, sens = .5, spec = .5)  # random performance
comp_accu_prob(prev = .5, sens = 1,  spec = 1)   # perfect accuracy
comp_accu_prob(prev = .5, sens = 0,  spec = 0)   # zero accuracy, but f1s is NaN
comp_accu_prob(prev = 1,  sens = 1,  spec = 0)   # perfect, but see wacc (0.5) and mcc (0)

# Effects of w:
comp_accu_prob(prev = .5, sens = .6, spec = .4, w = 1/2)  # equal weights to sens and spec
comp_accu_prob(prev = .5, sens = .6, spec = .4, w = 2/3)  # more weight on sens: wacc up
comp_accu_prob(prev = .5, sens = .6, spec = .4, w = 1/3)  # more weight on spec: wacc down

# Contrasting comp_accu_freq and comp_accu_prob:
# (a) comp_accu_freq (based on rounded frequencies):
freq1 <- comp_freq(N = 10, prev = 1/3, sens = 2/3, spec = 3/4)   # => rounded frequencies!
accu1 <- comp_accu_freq(freq1$hi, freq1$mi, freq1$fa, freq1$cr)  # => accu1 (based on rounded freq).
# accu1

# (b) comp_accu_prob (based on probabilities):
accu2 <- comp_accu_prob(prev = 1/3, sens = 2/3, spec = 3/4)      # => exact accu (based on prob).
# accu2
all.equal(accu1, accu2)  # => 4 differences!
#
# (c) comp_accu_freq (exact values, i.e., without rounding):
freq3 <- comp_freq(N = 10, prev = 1/3, sens = 2/3, spec = 3/4, round = FALSE)
accu3 <- comp_accu_freq(freq3$hi, freq3$mi, freq3$fa, freq3$cr)  # => accu3 (based on EXACT freq).
# accu3
all.equal(accu2, accu3)  # => TRUE (qed).


}
\references{
Consult \href{https://en.wikipedia.org/wiki/Confusion_matrix}{Wikipedia: Confusion matrix}
for additional information.
}
\seealso{
\code{\link{accu}} for all accuracy metrics;
\code{\link{comp_accu_freq}} computes accuracy metrics from frequencies;
\code{\link{num}} for basic numeric parameters;
\code{\link{freq}} for current frequency information;
\code{\link{txt}} for current text settings;
\code{\link{pal}} for current color settings;
\code{\link{popu}} for a table of the current population.

Other metrics: \code{\link{accu}}, \code{\link{acc}},
  \code{\link{comp_accu_freq}}, \code{\link{comp_acc}},
  \code{\link{comp_err}}, \code{\link{err}}

Other functions computing probabilities: \code{\link{comp_FDR}},
  \code{\link{comp_FOR}}, \code{\link{comp_NPV}},
  \code{\link{comp_PPV}}, \code{\link{comp_accu_freq}},
  \code{\link{comp_acc}}, \code{\link{comp_comp_pair}},
  \code{\link{comp_complement}},
  \code{\link{comp_complete_prob_set}},
  \code{\link{comp_err}}, \code{\link{comp_fart}},
  \code{\link{comp_mirt}}, \code{\link{comp_ppod}},
  \code{\link{comp_prob_freq}}, \code{\link{comp_prob}},
  \code{\link{comp_sens}}, \code{\link{comp_spec}}
}
\concept{functions computing probabilities}
\concept{metrics}
