% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/predict.CauseSpecificCox.R
\name{predict.CauseSpecificCox}
\alias{predict.CauseSpecificCox}
\alias{predictBig.CauseSpecificCox}
\title{Predicting Absolute Risk from Cause-Specific Cox Models}
\usage{
\method{predict}{CauseSpecificCox}(
  object,
  newdata,
  times,
  cause,
  type = "absRisk",
  landmark = NA,
  keep.times = 1L,
  keep.newdata = 1L,
  keep.strata = 1L,
  se = FALSE,
  band = FALSE,
  iid = FALSE,
  confint = (se + band) > 0,
  average.iid = FALSE,
  product.limit = TRUE,
  store.iid = "full",
  diag = FALSE,
  ...
)
}
\arguments{
\item{object}{The fitted cause specific Cox model}

\item{newdata}{[data.frame or data.table]  Contain the values of the predictor variables
defining subject specific predictions relative to each cause.
Should have the same structure as the data set used to fit the \code{object}.}

\item{times}{[numeric vector] Time points at which to return
the estimated absolute risk.}

\item{cause}{[integer/character] Identifies the cause of interest among the competing
events.}

\item{type}{[character] Can be changed to \code{"survival"} if the event free survival should be output instead of the absolute risk.}

\item{landmark}{[integer] The starting time for the computation of the cumulative risk.}

\item{keep.times}{[logical] If \code{TRUE} add the evaluation times to the output.}

\item{keep.newdata}{[logical] If \code{TRUE} add the value of the covariates used to make the prediction in the output list.}

\item{keep.strata}{[logical] If \code{TRUE} add the value of the strata used to make the prediction in the output list.}

\item{se}{[logical] If \code{TRUE} compute and add the standard errors to the output.}

\item{band}{[logical] If \code{TRUE} compute and add the quantiles for the confidence bands to the output.}

\item{iid}{[logical] If \code{TRUE} compute and add the influence function to the output.}

\item{confint}{[logical] If \code{TRUE} compute and add the confidence intervals/bands to the output.
They are computed applying the \code{confint} function to the output.}

\item{average.iid}{[logical]. If \code{TRUE} add the average of the influence function over \code{newdata} to the output.}

\item{product.limit}{[logical]. If \code{TRUE} the survival is computed using the product limit estimator.
Otherwise the exponential approximation is used (i.e. exp(-cumulative hazard)).}

\item{store.iid}{[character] Implementation used to estimate the influence function and the standard error.
Can be \code{"full"} or \code{"minimal"}.}

\item{diag}{[logical] when \code{FALSE} the absolute risk/survival for all observations at all times is computed,
otherwise it is only computed for the i-th observation at the i-th time.}

\item{...}{not used.}
}
\description{
Apply formula to combine two or more Cox models into absolute risk (cumulative incidence function).
}
\details{
This function computes the absolute risk as given by formula 2 of (Ozenne et al., 2017).
Confidence intervals and confidence bands can be computed using a first order von Mises expansion.
See the section "Construction of the confidence intervals" in (Ozenne et al., 2017).

A detailed explanation about the meaning of the argument \code{store.iid} can be found
in (Ozenne et al., 2017) Appendix B "Saving the influence functions".

Note: for Cox regression models with time varying
    covariates it does not make sense to use this function, because
    the predicted risk has to be a measurable function of the data
    available at the time origin.

The iid decomposition is output using an array containing the value of the influence
of each subject used to fit the object (dim 3),
for each subject in newdata (dim 1),
and each time (dim 2).
}
\examples{
library(survival)
library(prodlim)
#### generate data ####
set.seed(5)
d <- sampleData(80,outcome="comp") ## training dataset
nd <- sampleData(4,outcome="comp") ## validation dataset
d$time <- round(d$time,1) ## create tied events
ttt <- sort(sample(x = unique(d$time), size = 10))

## estimate a CSC model based on the coxph function
CSC.fit <- CSC(Hist(time,event)~ X3+X8, data=d, method = "breslow")

## compute the absolute risk of cause 1, in the validation dataset
## at time 1:10
CSC.risk <-  predict(CSC.fit, newdata=nd, times=1:10, cause=1)
CSC.risk

## compute absolute risks with CI for cause 2
## (without displaying the value of the covariates)
predict(CSC.fit,newdata=nd,times=1:10,cause=2,se=TRUE,
        keep.newdata = FALSE)

## other example
library(survival)
CSC.fit.s <- CSC(list(Hist(time,event)~ strata(X1)+X2+X9,
 Hist(time,event)~ X2+strata(X4)+X8+X7),data=d, method = "breslow")
predict(CSC.fit.s,cause=1,times=ttt,se=1L) ## note: absRisk>1 due to small number of observations

## using the cph function instead of coxph
CSC.cph <- CSC(Hist(time,event)~ X1+X2,data=d, method = "breslow", fitter = "cph")#' 
predict(CSC.cph, newdata = d, cause = 2, times = ttt)

## landmark analysis
T0 <- 1
predCSC_afterT0 <- predict(CSC.fit, newdata = d, cause = 2, times = ttt[ttt>T0], landmark = T0)
predCSC_afterT0
}
\references{
Brice Ozenne, Anne Lyngholm Sorensen, Thomas Scheike, Christian Torp-Pedersen and Thomas Alexander Gerds.
riskRegression: Predicting the Risk of an Event using Cox Regression Models.
The R Journal (2017) 9:2, pages 440-460.
}
\seealso{
\code{\link{confint.predictCSC}} to compute confidence intervals/bands.
\code{\link{autoplot.predictCSC}} to display the predictions.
}
\author{
Brice Ozenne broz@sund.ku.dk, Thomas A. Gerds
    tag@biostat.ku.dk
}
