% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/table_spec.r
\name{column_defaults}
\alias{column_defaults}
\title{Set default attributes for one or more columns}
\usage{
column_defaults(
  x,
  vars = NULL,
  from = NULL,
  to = NULL,
  label = NULL,
  format = NULL,
  align = NULL,
  label_align = NULL,
  width = NULL,
  n = NULL,
  standard_eval = FALSE
)
}
\arguments{
\item{x}{A table spec.}

\item{vars}{The variable name or names to define defaults for.  Variable
names may be quoted or unquoted.  The parameter will also accept 
integer column positions instead of names.  For multiple variables, 
pass the names or positions as a vector. If you want to pass an R variable 
of names,
set the \code{standard_eval} parameter to TRUE.
The \code{standard_eval} parameter is useful when writing functions that construct
reports dynamically.}

\item{from}{The variable name or position that starts a column range.
If passed as a variable name, it may be quoted or unquoted.}

\item{to}{The variable name or position that ends a column range. 
If passed as a variable name, it 
may be quoted or unquoted.}

\item{label}{The label to use for a column header.  This label will be 
applied to all variables assigned to the \code{column_defaults} function.}

\item{format}{The format to use for the column data.  The format can 
be a string format, a formatting function, a lookup list, a user-defined
format, or a formatting list. 
All formatting is performed by the \code{\link[fmtr]{fmtr}} package.  For 
additional information, see the help for that package.}

\item{align}{The column alignment.  Valid values are "left", "right", 
"center", and "centre".}

\item{label_align}{How to align the header labels for this column.
Valid values are "left", "right", "center", and "centre".}

\item{width}{The width of the column in the specified units of measure.
The units of measure are specified on the \code{units} parameter of the
\code{\link{create_report}} function.  If no width is supplied, the
\code{\link{write_report}} function will assign a default width based on the 
width of the column data and the label.  \code{write_report} will not set a 
column width less than the width of the largest word in the data or label.
In other words, \code{write_report} will not break words.}

\item{n}{The n value to place in the "N=" header label.  Formatting for
the n value will be performed by the formatting function assigned to the 
\code{n_format} parameter on \code{\link{create_table}}.}

\item{standard_eval}{A TRUE or FALSE value that indicates whether to
use standard or non-standard evaluation of the \code{vars}, \code{from},
and \code{to} parameters.  Set \code{standard_eval} to TRUE if you want
to pass the column names as variables. Default is FALSE, meaning it
will use non-standard (unquoted) evaluation.}
}
\value{
The modified table spec.
}
\description{
A function to set default attributes for columns on a table.  
The \code{column_defaults} function contains a subset of the parameters
on the \code{\link{define}} function that can be shared across variables.
Any attributes set by \code{column_defaults} can be overridden by 
the \code{define} function.  The overall purpose of the 
function is to minimize redundancy in column definitions.
}
\details{
Column defaults can be specified for multiple variables.  By default,
the function will apply to all variables.  Alternately, you can 
specify a vector of columns on the \code{vars} parameter, or a range of
columns using the \code{from} and \code{to} parameters.  Both the 
\code{vars} parameters and the \code{from} and \code{to} parameters
will accept column positions, quoted variable names, or unquoted variable 
names.

The parameters that can be set with the \code{column_defaults} 
include the formatting attributes 'width', 'justify', 'label', and 
'format'.  Any parameters set with \code{column_defaults} will override
any attributes set on the data frame.

Note that you may call the \code{column_defaults} function multiple times
on the same table specification.  Typically, multiple \code{column_defaults}
calls would be made with a different set or range of variables.
}
\examples{
library(reporter)
library(magrittr)
 
# Create temp file name
tmp <- file.path(tempdir(), "mtcars.txt")

# Prepare data
dat <- mtcars[1:10, ]
dat <- data.frame(vehicle = rownames(dat), dat, stringsAsFactors = FALSE)

# Define table
tbl <- create_table(dat, show_cols = 1:8) \%>\% 
  column_defaults(from = mpg, to = qsec, width = .5, format = "\%.1f") \%>\% 
  define(vehicle, label = "Vehicle", width = 1.5, align = "left") \%>\% 
  define(c(cyl, hp), format = "\%.0f") 

# Create the report
rpt <- create_report(tmp, orientation = "portrait") \%>\% 
  titles("Table 2.5", "MTCARS Sample Report") \%>\% 
  add_content(tbl) 

# Write the report
write_report(rpt)

# Send report to console for viewing
writeLines(readLines(tmp, encoding = "UTF-8"))

#                                 Table 2.5
#                            MTCARS Sample Report
# 
#    Vehicle               mpg    cyl   disp     hp   drat     wt   qsec
#    -------------------------------------------------------------------
#    Mazda RX4            21.0      6  160.0    110    3.9    2.6   16.5
#    Mazda RX4 Wag        21.0      6  160.0    110    3.9    2.9   17.0
#    Datsun 710           22.8      4  108.0     93    3.8    2.3   18.6
#    Hornet 4 Drive       21.4      6  258.0    110    3.1    3.2   19.4
#    Hornet Sportabout    18.7      8  360.0    175    3.1    3.4   17.0
#    Valiant              18.1      6  225.0    105    2.8    3.5   20.2
#    Duster 360           14.3      8  360.0    245    3.2    3.6   15.8
#    Merc 240D            24.4      4  146.7     62    3.7    3.2   20.0
#    Merc 230             22.8      4  140.8     95    3.9    3.1   22.9
#    Merc 280             19.2      6  167.6    123    3.9    3.4   18.3
#
}
\seealso{
Other table: 
\code{\link{create_table}()},
\code{\link{define}()},
\code{\link{print.table_spec}()},
\code{\link{spanning_header}()},
\code{\link{stub}()}
}
\concept{table}
