#' Read a character text file from a secure (https) site into R as a single
#' object.
#'
#' @param url The files's URL.
#' @param sha1 Character string of the file's SHA-1 hash, generated by
#' \code{source_data}. Note if you are using data stored using Git, this is not
#' the file's commit SHA-1 hash.
#' @return a charcter object of length 1
#' @source Originally based on source_url from the Hadley Wickham's devtools
#' package.
#'
#' @importFrom digest digest
#' @importFrom httr GET stop_for_status text_content content
#' @export

scan_https <- function(url, sha1 = NULL) {
    stopifnot(is.character(url), length(url) == 1)

    temp_file <- tempfile()
    on.exit(unlink(temp_file))

    request <- httr::GET(url)
    httr::stop_for_status(request)
    writeBin(httr::content(request, type = "raw"), temp_file)

    file_sha1 <- digest::digest(file = temp_file, algo = "sha1")

    if (is.null(sha1)) {
        message("SHA-1 hash of file is ", file_sha1)
    } else {
        if (nchar(sha1) < 6) {
            stop("Supplied SHA-1 hash is too short (must be at least 6 characters)")
        }

        # Truncate file_sha1 to length of sha1
        file_sha1 <- substr(file_sha1, 1, nchar(sha1))

        if (!identical(file_sha1, sha1)) {
            stop("SHA-1 hash of downloaded file (", file_sha1,
                 ")\n  does not match expected value (", sha1, ")",
                call. = FALSE)
        }
    }
    out <- suppressMessages(scan(temp_file, character(0), sep = "\n"))
    out <- paste(out, collapse = '\n')
    return(out)
}
