% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/compactness.R
\name{redist.compactness}
\alias{redist.compactness}
\title{Calculate compactness measures for a set of districts}
\usage{
redist.compactness(
  shp = NULL,
  district_membership,
  measure = c("PolsbyPopper"),
  population = NULL,
  adjacency = NULL,
  nloop = 1,
  ncores = 1
)
}
\arguments{
\item{shp}{A SpatialPolygonsDataFrame or sf object. Required unless "EdgesRemoved"
and "logSpanningTree" with adjacency provided.}

\item{district_membership}{A numeric vector (if only one map) or matrix with one row 
for each precinct and one column for each map. Required.}

\item{measure}{A vector with a string for each measure desired. "PolsbyPopper", 
"Schwartzberg", "LengthWidth", "ConvexHull", "Reock", "BoyceClark", "FryerHolden",  
"EdgesRemoved", and "logSpanningTree" are implemented. Defaults to "PolsbyPopper". Use "all" to
return all implemented measures.}

\item{population}{A numeric vector with the population for every observation. Is
only necessary when "FryerHolden" is used for measure. Defaults to NULL.}

\item{adjacency}{A zero-indexed adjacency list. Only used for "EdgesRemoved" and "logSpanningTree".
Created with \code{redist.adjacency} if not supplied and needed. Default is NULL.}

\item{nloop}{A numeric to specify loop number. Defaults to 1 if only one map provided 
and the column number if multiple maps given.}

\item{ncores}{Number of cores to use for parallel computing. Default is 1.}
}
\value{
A tibble with a column that specifies the district, a column for 
each specified measure, and a column that specifies the map number.
}
\description{
\code{redist.compactness} is used to compute different compactness statistics for a 
shapefile. It currently computes the Polsby-Popper, Schwartzberg score, Length-Width Ratio,
Convex Hull score, Reock score, Boyce Clark Index, Fryer Holden score, Edges Removed number, 
and the log of the Spanning Trees.
}
\details{
This function computes specified compactness scores for a map.  If 
there is more than one shape specified for a single district, it combines 
them, if necessary, and computes one score for each district.

Polsby-Popper is computed as \deqn{\frac{4*\pi*A(d)}{P(d)^2}} where A is the area 
function, the district is d, and P is the perimeter function.

Schwartzberg is computed as \deqn{\frac{P(d)}{2*\pi*\sqrt{\frac{A(d)}{\pi}}}}
where A is the area function, the district is d, and P is the perimeter function.

The Length Width ratio is computed as \deqn{\frac{length}{width}} where length
is the shorter of the maximum x distance and the maximum y distance. Width is 
the longer of the two values.

The Reock score is computed as \deqn{\frac{A(d)}{A(CVH)}} where A is the area
function, d is the district, and CVH is the convex hull of the district.

The Boyce Clark Index is computed as \deqn{1 - \sum_{1}^{16}\{\frac{|\frac{r_i}{\sum_ir_i}*100-6.25 |\}}{200}}.
The \eqn{r_i} are the distances of the 16 radii computed from the geometric 
centroid of the shape to the most outward point of the shape that intersects
the radii, if the centroid is contained within the shape.  If the centroid
lies outside of the shape, a point on the surface is used, which will naturally
incur a penalty to the score.

The Fryer Holden score for each district is computed with \deqn{Pop\odot D(precinct)^2},
where \eqn{Pop} is the population product matrix.  Each element is the 
product of the ith and jth precinct's populations.  D represents the distance, 
where the matrix is the distance between each precinct.  To fully compute this 
index, for any map, the sum of these values should be used as the numerator. 
The denominator can be calculated from the full enumeration of districts as the
smallest calculated numerator.

The log spanning tree measure is the log number of spanning trees.

The edges removed measure is number of egdes removed from the underlying adjacency graph.
}
\examples{
\dontrun{
library(sf)
library(lwgeom)
library(redist)
library(tidyverse)
#Create (or load) a shapefile, in this case the unit square 
box <-  rbind(c(0,0), c(1,0), c(1,1), c(0,1), c(0,0)) \%>\% list() \%>\%  
st_polygon() \%>\% st_sfc() \%>\% st_as_sf()
# Index the congressional districts
box <- box \%>\% mutate(cds = 1, pop = 10)
# Run redist.compactness
redist.compactness(box, "cds", "pop")
}

}
\references{
Boyce, R., & Clark, W. 1964. The Concept of Shape in Geography. 
Geographical Review, 54(4), 561-572.

Cox, E. 1927. A Method of Assigning Numerical and Percentage Values to the 
Degree of Roundness of Sand Grains. Journal of Paleontology, 1(3), 179-183.

Fryer R, Holden R. 2011. Measuring the Compactness of Political Districting Plans. 
Journal of Law and Economics.  

Harris, Curtis C. 1964. “A scientific method of districting”. 
Behavioral Science 3(9), 219–225.

Maceachren, A. 1985. Compactness of Geographic Shape: Comparison and 
Evaluation of Measures. Geografiska Annaler. Series B, Human Geography, 67(1), 
53-67.

Polsby, Daniel D., and Robert D. Popper. 1991. “The Third Criterion: 
Compactness as a procedural safeguard against partisan gerrymandering.” 
Yale Law & Policy Review 9 (2): 301–353.
 
Reock, E. 1961. A Note: Measuring Compactness as a Requirement of Legislative 
Apportionment. Midwest Journal of Political Science, 5(1), 70-74.

Schwartzberg, Joseph E. 1966. Reapportionment, Gerrymanders, and the Notion 
of Compactness. Minnesota Law Review. 1701.
}
