#' @name importFiles
#' @title Imports a File to REDCap to Attach to a Record
#' 
#' @description A single file may be attached to a single record.  The 
#'   behavior of this function is consistent with the
#'   behavior of the API, which only allows one file to be uploaded at a time
#'   
#' @param rcon A REDCap connection object as generated by \code{redcapConnection}
#' @param file Character string giving the file path to the file to be imported.
#' @param record The record ID in which the desired file is stored. Must be length 1.
#' @param field The field name in which the file is stored. Must be length 1.
#' @param event The event name for the file.  Must be length 1.  This applies 
#'   only to longitudinal projects.  If the event is not supplied for a 
#'   longitudinal project, the API will return an error
#' @param overwrite Logical.  When \code{FALSE}, the function checks if a 
#'   file already exists for that record.  If a file exists, the function 
#'   terminates to prevent overwriting.  When \code{TRUE}, no additional 
#'   check is performed.
#' @param repeat_instance The repeat instance number of the repeating
#'   event or the repeating instrument. When available in your instance
#'   of REDCap, and passed as NULL, the API will assume a value of 1.
#' @param ... Arguments to be passed to other methods
#' @param error_handling An option for how to handle errors returned by the API.
#'   see \code{\link{redcap_error}}
#' @param config \code{list} Additional configuration parameters to pass to 
#'   \code{\link[httr]{POST}}. These are appended to any parameters in 
#'   \code{rcon$config}.
#' @param api_param \code{list} Additional API parameters to pass into the
#'   body of the API call. This provides users to execute calls with options
#'   that may not otherwise be supported by \code{redcapAPI}.
#' 
#' @details The function may only import a single file
#' 
#' @author Benjamin Nutter
#' 
#' @export

importFiles <- function(rcon, 
                        file, 
                        record, 
                        field, 
                        event, 
                        overwrite       = TRUE,
                        repeat_instance = NULL, 
                        ...){
  UseMethod("importFiles")
}

#' @rdname importFiles
#' @export

importFiles.redcapApiConnection <- function(rcon, 
                                            file, 
                                            record, 
                                            field, 
                                            event           = NULL, 
                                            overwrite       = TRUE,
                                            repeat_instance = NULL, 
                                            ...,
                                            error_handling  = getOption("redcap_error_handling"),
                                            config          = list(), 
                                            api_param       = list()){
  
  if (is.numeric(record)) record <- as.character(record)
  
   ##################################################################
  # Argumetn Validation
  
  coll <- checkmate::makeAssertCollection()
  
  checkmate::assert_class(x = rcon, 
                          classes = "redcapApiConnection", 
                          add = coll)
  
  checkmate::assert_character(x = file, 
                              len = 1, 
                              any.missing = FALSE,
                              add = coll)
  
  checkmate::assert_character(x = record, 
                              len = 1, 
                              any.missing = FALSE, 
                              add = coll)
  
  checkmate::assert_character(x = field, 
                              len = 1, 
                              any.missing = FALSE, 
                              add = coll)
  
  checkmate::assert_character(x = event, 
                              len = 1, 
                              any.missing = FALSE, 
                              null.ok = TRUE,
                              add = coll)
  
  checkmate::assert_logical(x = overwrite, 
                            len = 1, 
                            any.missing = FALSE, 
                            add = coll)
  
  checkmate::assert_integerish(x = repeat_instance,
                               len = 1,
                               any.missing = FALSE,
                               null.ok = TRUE,
                               add = coll)
  
  error_handling <- checkmate::matchArg(x = error_handling,
                                        choices = c("null", "error"),
                                        .var.name = "error_handling",
                                        add = coll)
  
  checkmate::assert_list(x = config, 
                         names = "named", 
                         add = coll)
  
  checkmate::assert_list(x = api_param, 
                         names = "named", 
                         add = coll)
  
  checkmate::reportAssertions(coll)
  
  checkmate::assert_file_exists(x = file,
                                add = coll)
  
  # make sure 'field' exists in the project and are 'file' fields
  MetaData <- rcon$metadata()

  if (!field %in% MetaData$field_name) 
    coll$push(paste("'", field, "' does not exist in the project.", sep=""))
  
  if (MetaData$field_type[MetaData$field_name == field] != "file")
    coll$push(paste0("'", field, "' is not of field type 'file'"))
  
  # make sure 'event' exists in the project
  
  is_project_longitudinal <- as.logical(rcon$projectInformation()$is_longitudinal)

  if (is_project_longitudinal)
  {
    EventsList <- rcon$events()
    
    if (nrow(EventsList) == 0)
    {
      message("No events defined in this project. Ignoring the 'event' argument.")
      event <- NULL
    } else {
      checkmate::assert_subset(x = event, 
                               choices = EventsList$unique_event_name, 
                               add = coll)
      checkmate::reportAssertions(coll)
    }
  } else {
    event <- NULL
  }
  
  if (!overwrite)
  {
    fileThere <- exportRecordsTyped(rcon, 
                                    records = record, 
                                    fields = c(MetaData$field_name[1], field), 
                                    events = event)
    if (nrow(fileThere) > 0 && length(repeat_instance) > 0){
      fileThere <- fileThere[fileThere$redcap_repeat_instance %in% repeat_instance, ]
    }
    
    if (nrow(fileThere) > 0 && !is.na(fileThere[[field]])) 
      coll$push("A file exists and overwrite = FALSE")
  }
  
  checkmate::reportAssertions(coll)
  
   ###########################################################################
  # Build the body list
  
  body <- list(content = 'file',
               action = 'import', 
               record = record,
               field = field, 
               file = httr::upload_file(file), 
               returnFormat = 'csv', 
               event = event, 
               repeat_instance = repeat_instance)
  
  body <- body[lengths(body) > 0]
  
   ###########################################################################
  # Make the API Call
  
  response <- makeApiCall(rcon, 
                          body = c(body, 
                                   api_param), 
                          config = config)
  
  if (response$status_code != "200") 
    redcap_error(response, error_handling)
  else 
    message("The file was successfully uploaded")
}