#' @name exportFieldNames
#' @title Export the Export Field Names for a Project
#' 
#' @description Retrieve a data frame giving the original (as defined in REDCap)
#'   field name, choice values (for checkboxes), and the export field name.
#'
#' @param rcon A REDCap connection object as generated by \code{redcapConnection}.
#' @param fields \code{character} with maximum length of 1. Field name to be returned.  By 
#'   default, all fields are returned.
#' @param bundle A \code{redcapProject} object as created by \code{redcapProjectInfo}.
#' @param ... Arguments to be passed to other methods.
#' @param error_handling An option for how to handle errors returned by the API.
#'   see \code{\link{redcap_error}}
#' @param config \code{list} Additional configuration parameters to pass to 
#'   \code{\link[httr]{POST}}. These are appended to any parameters in 
#'   \code{rcon$config}.
#' @param api_param \code{list} Additional API parameters to pass into the
#'   body of the API call. This provides users to execute calls with options
#'   that may not otherwise be supported by \code{redcapAPI}.
#'
#' 
#' @section REDCap API Documentation:
#' This function returns a list of the export/import-specific version of field names for 
#' all fields (or for one field, if desired) in a project. This is mostly used for 
#' checkbox fields because during data exports and data imports, 
#' checkbox fields have a different variable name used than the exact one 
#' defined for them in the Online Designer and Data Dictionary, in which 
#' *each checkbox option* gets represented as its own export field name in the 
#' following format: field_name + triple underscore + converted coded value for the 
#' choice. For non-checkbox fields, the export field name will be exactly the same 
#' as the original field name. Note: The following field types will be automatically 
#' removed from the list returned by this method since they cannot be utilized during 
#' the data import process: "calc", "file", and "descriptive".
#'
#' The list that is returned will contain the three following attributes for each 
#' field/choice: "original_field_name", "choice_value", and "export_field_name". 
#' The choice_value attribute represents the raw coded value for a checkbox choice. 
#' For non-checkbox fields, the choice_value attribute will always be blank/empty. 
#' The export_field_name attribute represents the export/import-specific version of 
#' that field name.
#' 
#' @section REDCap Version:
#' 6.5.0+ (perhaps earlier; need to confirm its introduction)
#' 
#' @section Known REDCap Limitations: 
#' In 6.5.0, it has been observed that "slider" fields are not returned.  
#' 
#' Signature fields are also not included, but these are effectively the same as 
#' "file" fields.  This isn't a true limitation, but is documented here just to
#' avoid confusion.
#' 
#' @return
#' A data frame containing three fields: 
#' \itemize{
#'   \item{\code{original_field_name} }{The field name as recorded in the 
#'        data dictionary}
#'   \item{\code{choice_value} }{represents the raw coded value for a checkbox 
#'        choice. For non-checkbox fields, this will always be \code{NA}.}
#'   \item{\code{export_field_name} }{The field name specific to the field.
#'        For non-checkbox fields, this is the same as \code{original_field_name}.
#'        For checkbox fields, it is the field name appended with 
#'        \code{___[choice_value]}.}
#' }
#'
#' @author Stephen Lane
#'
#' @references
#' Please refer to your institution's API documentation
#' (https://YOUR_REDCAP_URL/redcap/api/help)
#'
#' Additional details on API parameters are found on the package wiki at
#' \url{https://github.com/vubiostat/redcapAPI/wiki/REDCap-API-Parameters}
#'
#' @export


exportFieldNames <- function(rcon, 
                             ...,
                             error_handling = getOption("redcap_error_handling")){
  UseMethod("exportFieldNames")
}

#' @rdname exportFieldNames
#' @export

exportFieldNames.redcapApiConnection <- function(rcon, 
                                                 fields = character(0), 
                                                 bundle = NULL, ...,
                                                 error_handling = getOption("redcap_error_handling"), 
                                                 config = list(), 
                                                 api_param = list()){
  if (!is.na(match("proj", names(list(...)))))
  {
    message("The 'proj' argument is deprecated.  Please use 'bundle' instead")
    bundle <- list(...)[["proj"]]
  }
 
  # Argument validation ---------------------------------------------
  coll <- checkmate::makeAssertCollection()
  
  checkmate::assert_class(x = rcon, 
                          classes = "redcapApiConnection", 
                          add = coll)
  
  checkmate::assert_character(x = fields,
                              max.len = 1,
                              add = coll)
  
  checkmate::assert_class(x = bundle, 
                          classes = "redcapBundle", 
                          null.ok = TRUE, 
                          add = coll)
  
  error_handling <- checkmate::matchArg(x = error_handling,
                                        choices = c("null", "error"),
                                        add = coll)
  
  checkmate::assert_list(x = config, 
                         names = "named", 
                         add = coll)
  
  checkmate::assert_list(x = api_param, 
                         names = "named", 
                         add = coll)
  
  checkmate::reportAssertions(coll)
  
  if (length(fields) > 0){
    .exportFieldNames_validateFieldName(fields = fields, 
                                        rcon = rcon, 
                                        bundle = bundle, 
                                        coll = coll)
  }

  # Build the Body List ---------------------------------------------
  body <- list(token = rcon$token, 
               content = 'exportFieldNames', 
               format = 'csv',
               returnFormat = 'csv', 
               field = fields)
  
  body <- body[lengths(body) > 0]
  
  # Make the API Call -----------------------------------------------
  response <- makeApiCall(rcon, 
                          body = c(body, api_param), 
                          config = config)
  
  if (response$status_code != 200){
    redcap_error(response, 
                 error_handling = error_handling)

  }
  
  utils::read.csv(text = as.character(response), 
                  stringsAsFactors = FALSE,
                  na.strings = "")
}

# Unexported --------------------------------------------------------

.exportFieldNames_validateFieldName <- function(fields, rcon, bundle, coll){
  # Get project metadata
  meta_data <- 
    if(is.null(bundle$meta_data)){
      exportMetaData(rcon) 
    } else {
      bundle$meta_data
    }
  
  if (!all(fields %in% meta_data$field_name)){
    coll$push(sprintf("Field does not exist in the database: %s", 
                      fields))
    checkmate::reportAssertions(coll)
  }
}
