#' @name deleteRecords
#' @title Delete Records From a Project
#' 
#' @description Delete records from a project.  This is a destructive action that
#'   will result in deletion of any events and data associated with the arm.
#'   Data loss is non-reversible.  The user must have 'Delete Record' privileges
#'   in the database.
#'   
#' @param rcon A REDCap connection object as generated by 
#'   \code{redcapConnection}.
#' @param records a vector of record names to be deleted.
#' @param arms \code{integerish}, the arm number of the arm in which the 
#'   record(s) should be deleted. (This can only be used if the project is 
#'   longitudinal with more than one arm.) NOTE: If the arm parameter is not 
#'   provided, the specified records will be deleted from all arms in which 
#'   they exist. Whereas, if arm is provided, they will only be deleted from 
#'   the specified arm.
#' @param error_handling An option for how to handle errors returned by the API.
#'   see \code{\link{redcap_error}}
#' @param ... Additional arguments to pass to other methods.
#' 
#' @section REDCap API Documentation:
#' This method allows you to delete one or more records from a project in a 
#' single API request.
#' 
#' @section REDCap Version:
#' At least 8.1.17+ (and likely some earlier versions)
#' 
#' @return 
#' The number of deleted records.
#' 
#' @references
#' Please refer to your institution's API documentation.
#' 
#' @export

deleteRecords <- function(rcon, records, arms = NULL, ...){
  UseMethod("deleteRecords")
}

#' @rdname deleteRecords
#' @export

deleteRecords.redcapDbConnection <- function(rcon, records, arms = NULL, ...){
  message("Please accept my apologies.  The deleteRecords method for redcapDbConnection objects\n",
          "has not yet been written.  Please consider using the API.")
}

#' @rdname deleteRecords
#' @export

deleteRecords.redcapApiConnection <- function(rcon, records, arms = NULL, ...,
                                           error_handling = getOption("redcap_error_handling")){

  coll <- checkmate::makeAssertCollection()
  
  checkmate::assert_atomic(x = records,
                           any.missing = FALSE,
                           min.len = 1,
                           add = coll)
  
  checkmate::assert_integerish(arms,
                               null.ok = TRUE,
                               add = coll)
  
  checkmate::reportAssertions(coll)
  
  if (!is.character(records)) records <- as.character(records)
  
  body <- list(token = rcon$token,
               content = "record",
               action = "delete")
  
  records <- lapply(records, 
                      identity)
  names(records) <- sprintf("records[%s]", records)
  
  body <- c(body, records)

  if (!is.null(arms))
    body[["arms"]] <- paste0(arms, collapse = ",")
  
  x <- httr::POST(url = rcon$url, 
                  body = body, 
                  config = rcon$config)
  
  if (x$status_code != 200) return(redcap_error(x, error_handling))
  
  as.character(x)
}
