#' Estimating the tree-topology, and outputting it into Newick format from the splits of the tree-topology
#'
#' \code{topology_to_newick()} inputs a matrix of with P columns, where each row represents a split. In a given row, a k-way split is represented by assigning each set of taxa that is descendent of a split as an identifying number from 1 to k, and numbering the position corresponding to each taxa that is part of such a set by that identifying number. The other positions are set to zero. The function outputs an estimated tree-topology in Newick format.
#'
#' @param label Generated by \code{row.names(trans_mat_allele_freq)}, which are row names of the input \code{mat_allele_freq}.
#' @param new_split_set_indicator A matrix, where each row represents a split. In a given row, a k-way split is represented by assigning each set of taxa that is descendent of a split as an identifying number from 1 to k, and numbering the position corresponding to each taxa that is part of such a set by that identifying number. The other positions are set to zero.
#'
#' @return An estimated tree-topology in Newick format.
#' @noRd
#'
#' @import stats
#' @import phangorn
#' @import ape
#'
#' @examples
#'
#' # load example data from rapidphylo package
#' data("Human_Allele_Frequencies")
#' mat_allele_freq <- Human_Allele_Frequencies
#' # perform logistic transformation
#' mat_allele_freq[mat_allele_freq==1] <- 0.99
#' mat_allele_freq[mat_allele_freq==0] <- 0.01
#' trans_mat_allele_freq <- log(mat_allele_freq/(1-mat_allele_freq))
#' # convert type of object into data frame
#' trans_mat_allele_freq <- as.data.frame(trans_mat_allele_freq)
#' outgroup <- 'Han'
#' names<-row.names(trans_mat_allele_freq)
#' # use the population names as the row names of your transformed allele frequency matrix")
#' if (is.character(outgroup)){
#'   index<-which(names==outgroup)
#'   }else {
#'     index<-outgroup
#'     }
#' trans_mat_allele_freq<-rbind(trans_mat_allele_freq[-index, ],trans_mat_allele_freq[index,])
#' label<-row.names(trans_mat_allele_freq)
#' array_zero_ID<-BMaf_to_zeroRDD(trans_mat_allele_freq,use="pairwise.complete.obs")
#' base_tree<-zeroRDD_to_splits(array_zero_ID)
#' # run function
#' rd_tre<-topology_to_newick(base_tree,label=label)
#' rd_tre
#' plot(rd_tre, use.edge.length = FALSE, cex = 0.5)


topology_to_newick<-function(new_split_set_indicator,label){
  mp2<-multiphylo(new_split_set_indicator,label=label)
  tre1<-superTree(mp2, rooted = TRUE)
  return(tre1)
}






