% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fractional.computations.R
\name{rSPDE.construct.matern.loglike}
\alias{rSPDE.construct.matern.loglike}
\title{Constructor of Matern loglikelihood functions.}
\usage{
rSPDE.construct.matern.loglike(
  object,
  Y,
  A,
  sigma.e = NULL,
  mu = 0,
  user_nu = NULL,
  user_kappa = NULL,
  user_sigma = NULL,
  user_m = NULL,
  log_scale = TRUE,
  return_negative_likelihood = TRUE,
  pivot = TRUE
)
}
\arguments{
\item{object}{The rational SPDE approximation,
computed using \code{\link[=matern.operators]{matern.operators()}}}

\item{Y}{The observations, either a vector or a matrix where
the columns correspond to independent replicates of observations.}

\item{A}{An observation matrix that links the measurement location to the
finite element basis.}

\item{sigma.e}{IF non-null, the standard deviation of the measurement noise will be kept fixed in
the returned likelihood.}

\item{mu}{Expectation vector of the latent field (default = 0).}

\item{user_nu}{If non-null, the shape parameter will be kept fixed in the returned likelihood.}

\item{user_kappa}{If non-null, the range parameter will be kept fixed in the returned likelihood.}

\item{user_sigma}{If non-null, the standard deviation will be kept fixed in the returned likelihood.}

\item{user_m}{If non-null, update the order of the rational approximation,
which needs to be a positive integer.}

\item{log_scale}{Should the parameters be evaluated in log-scale?}

\item{return_negative_likelihood}{Return minus the likelihood to turn the maximization into a minimization?}

\item{pivot}{Should pivoting be used for the Cholesky decompositions? Default
is TRUE}
}
\value{
The log-likelihood function. The parameters of the returned function
are given in the order sigma, kappa, nu, sigma.e, whenever they are available.
}
\description{
This function returns a log-likelihood function for a
Gaussian process with a Matern covariance
function, that is observed under Gaussian measurement noise:
\eqn{Y_i = u(s_i) + \epsilon_i}{Y_i = u(s_i) + \epsilon_i}, where
\eqn{\epsilon_i}{\epsilon_i} are
iid mean-zero Gaussian variables. The latent model is approximated using
the a rational approximation
of the fractional SPDE model corresponding to the Gaussian process.
}
\examples{
# this example illustrates how the function can be used for maximum
# likelihood estimation

set.seed(123)
# Sample a Gaussian Matern process on R using a rational approximation
nu <- 0.8
kappa <- 5
sigma <- 1
sigma.e <- 0.1
n.rep <- 10
n.obs <- 100
n.x <- 51

# create mass and stiffness matrices for a FEM discretization
x <- seq(from = 0, to = 1, length.out = n.x)
fem <- rSPDE.fem1d(x)

tau <- sqrt(gamma(nu) / (sigma^2 * kappa^(2 * nu) *
(4 * pi)^(1 / 2) * gamma(nu + 1 / 2)))

# Compute the covariance-based rational approximation
op_cov <- matern.operators(
  C = fem$C, G = fem$G, nu = nu,
  kappa = kappa, sigma = sigma, d = 1, m = 2
)

# Sample the model
u <- simulate(op_cov, n.rep)

# Create some data
obs.loc <- runif(n = n.obs, min = 0, max = 1)
A <- rSPDE.A1d(x, obs.loc)
noise <- rnorm(n.obs * n.rep)
dim(noise) <- c(n.obs, n.rep)
Y <- as.matrix(A \%*\% u + sigma.e * noise)

# Define the negative likelihood function for optimization
# using CBrSPDE.matern.loglike

# Notice that we are also using sigma instead of tau, so it can be compared
# to matern.loglike()
loglike <- rSPDE.construct.matern.loglike(op_cov, Y, A) 

# The parameters can now be estimated by minimizing mlik with optim
\donttest{
# Choose some reasonable starting values depending on the size of the domain
theta0 <- log(c(sqrt(8), 1 / sqrt(var(c(Y))), 0.9, 0.01))

# run estimation and display the results
theta <- optim(theta0, loglike,
  method = "L-BFGS-B"
)

print(data.frame(
  kappa = c(kappa, exp(theta$par[1])), sigma = c(sigma, exp(theta$par[2])),
  nu = c(nu, exp(theta$par[3])), sigma.e = c(sigma.e, exp(theta$par[4])),
  row.names = c("Truth", "Estimates")
))
}

}
\seealso{
\code{\link[=matern.operators]{matern.operators()}}, \code{\link[=predict.CBrSPDEobj]{predict.CBrSPDEobj()}}
}
