% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/C_simulateFullWaveform.R
\name{gediWFSimulator}
\alias{gediWFSimulator}
\title{GEDI full waveform data simulation}
\usage{
gediWFSimulator(input, output, waveID = NULL, coords = NULL,
  listCoord = NULL, gridBound = NULL, gridStep = 30, pSigma = -1,
  pFWHM = 15, readPulse = NULL, fSigma = 5.5, wavefront = NULL,
  res = 0.15, topHat = FALSE, sideLobe = FALSE, lobeAng = 0,
  checkCover = FALSE, maxScanAng = 1e+06, decimate = 1,
  pBuff = as.integer(2e+08), maxBins = as.integer(1024),
  countOnly = FALSE, pulseAfter = FALSE, pulseBefore = TRUE,
  noNorm = FALSE, noOctree = FALSE, octLevels = as.integer(0),
  nOctPix = as.integer(40), keepOld = FALSE, useShadow = FALSE,
  polyGround = FALSE)
}
\arguments{
\item{input}{character vector. lasfile input filename}

\item{output}{character. output filename}

\item{waveID}{id. supply a waveID to pass to the output (only for single footprints)
Single footprint, list of footprints, or grid of footprints}

\item{coords}{lon lat numeric vector. footprint coordinate in same system as lasfile}

\item{listCoord}{name. Text file with list of coordinates. Pattern: X Y `[waveID]` `[geoCoordsX]` `[geoCoordsY]`. `[]` are optional, separated by spaces.}

\item{gridBound}{minX maxX minY maxY numeric vector. make a grid of waveforms in this box}

\item{gridStep}{res. grid step size
Lidar characteristics. Defaults are expected GEDI values.}

\item{pSigma}{pSigmasig. set Gaussian pulse width as 1 sigma}

\item{pFWHM}{fhwm. set Gaussian pulse width as FWHM in ns}

\item{readPulse}{file. read pulse shape and width from a file insteda of making Gaussian}

\item{fSigma}{sig. set footprint width}

\item{wavefront}{file. read wavefront shape from file instead of setting Gaussian. Note that footprint width is still set by fSigma}

\item{res}{res. range resolution of waveform digitisation to output, in units of ALS data}

\item{topHat}{use a top hat wavefront}

\item{sideLobe}{use side lobes}

\item{lobeAng}{ang. lobe axis azimuth
Input data quality filters}

\item{checkCover}{check that the footprint is covered by ALS data. Do not output if not}

\item{maxScanAng}{ang. maximum scan angle, degrees}

\item{decimate}{x. probability of accepting an ALS beam
Computational speed options}

\item{pBuff}{s. point reading buffer size in Gbytes}

\item{maxBins}{for HDF5, limiting number of bins to save trimming.}

\item{countOnly}{only use count method}

\item{pulseAfter}{apply the pulse smoothing after binning for computational speed, at the risk of aliasing (default)}

\item{pulseBefore}{apply the pulse smoothing before binning to avoid the risk of aliasing, at the expense of computational speed}

\item{noNorm}{don't normalise for ALS density
Octree}

\item{noOctree}{do not use an octree}

\item{octLevels}{n. number of octree levels to use}

\item{nOctPix}{n. number of octree pixels along a side for the top level
Using full waveform input data (not tested)}

\item{keepOld}{do not overwrite old files, if they exist}

\item{useShadow}{account for shadowing in discrete return data through voxelization}

\item{polyGround}{find mean ground elevation and slope through fitting a polynomial

#'}
}
\value{
Returns an S4 object of class \code{\link[hdf5r:H5File-class]{hdf5r::H5File}} form the \emph{hdf5r} package
containing the simulated GEDI full-waveform.
}
\description{
Simulate GEDI full waveform data from Airborne Laser Scanning (ALS) 3D point cloud

Input and output filenames, and formats
}
\examples{
\dontshow{
rm(list=ls())
}
libsAvailable = require(lidR) && require(plot3D)
if (libsAvailable) {
outdir=tempdir()

# specify the path to ALS data (zip)
alsfile_Amazon_zip <- system.file("extdata", "Amazon.zip", package="rGEDI")
alsfile_Savanna_zip <- system.file("extdata", "Savanna.zip", package="rGEDI")

# Unzipping ALS data
alsfile_Amazon_filepath <- unzip(alsfile_Amazon_zip,exdir = outdir)
alsfile_Savanna_filepath <- unzip(alsfile_Savanna_zip,exdir = outdir)

# Reading and plot ALS file (las file)
als_Amazon<-readLAS(alsfile_Amazon_filepath)
als_Savanna<-readLAS(alsfile_Savanna_filepath)

# Extracting plot center geolocations
xcenter_Amazon = mean(als_Amazon@bbox[1,])
ycenter_Amazon = mean(als_Amazon@bbox[2,])
xcenter_Savanna = mean(als_Savanna@bbox[1,])
ycenter_Savanna = mean(als_Savanna@bbox[2,])

# Simulating GEDI full waveform
wf_Amazon<-gediWFSimulator(input=alsfile_Amazon_filepath,
                          output=file.path(outdir,"gediWF_amazon_simulation.h5"),
                          coords = c(xcenter_Amazon, ycenter_Amazon))

wf_Savanna<-gediWFSimulator(input=alsfile_Savanna_filepath,
                           output=file.path(outdir,"gediWF_Savanna_simulation.h5"),
                           coords = c(xcenter_Savanna, ycenter_Savanna))

# Plot ALS and GEDI simulated full waveform

oldpar<-par()
par(mfrow=c(2,2), mar=c(4,4,0,0), oma=c(0,0,1,1),cex.axis = 1.2)
scatter3D(
         als_Amazon@data$X,als_Amazon@data$Y,als_Amazon@data$Z,
         pch = 16,colkey = FALSE, main="",
         cex = 0.5,bty = "u",col.panel ="gray90",
         phi = 30,alpha=1,theta=45,col.grid = "gray50",
         xlab="UTM Easting (m)", ylab="UTM Northing (m)", zlab="Elevation (m)")

plot(wf_Amazon, relative=TRUE, polygon=TRUE, type="l", lwd=2, col="forestgreen",
    xlab="", ylab="Elevation (m)", ylim=c(90,140))
grid()
scatter3D(
         als_Savanna@data$X,als_Savanna@data$Y,als_Savanna@data$Z,
         pch = 16,colkey = FALSE, main="",
         cex = 0.5,bty = "u",col.panel ="gray90",
         phi = 30,alpha=1,theta=45,col.grid = "gray50",
         xlab="UTM Easting (m)", ylab="UTM Northing (m)", zlab="Elevation (m)")

plot(wf_Savanna, relative=TRUE, polygon=TRUE, type="l", lwd=2, col="green",
    xlab="Waveform Amplitude (\%)", ylab="Elevation (m)", ylim=c(815,835))
grid()

par(oldpar)

close(wf_Amazon)
close(wf_Savanna)
}

}
\seealso{
i) Hancock, S., Armston, J., Hofton, M., Sun, X., Tang, H., Duncanson, L.I., Kellner,
J.R. and Dubayah, R., 2019. The GEDI simulator: A large-footprint waveform lidar simulator
for calibration and validation of spaceborne missions. Earth and Space Science.
https://doi.org/10.1029/2018EA000506

ii) gediSimulator: https://bitbucket.org/StevenHancock/gedisimulator/src/master/
}
