/*
 * Copyright (c) 2016 Villu Ruusmann
 *
 * This file is part of JPMML-R
 *
 * JPMML-R is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * JPMML-R is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with JPMML-R.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.jpmml.rexp;

import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.dmg.pmml.DataField;
import org.dmg.pmml.DataType;
import org.dmg.pmml.DerivedField;
import org.dmg.pmml.Field;
import org.dmg.pmml.Model;
import org.dmg.pmml.OpType;
import org.dmg.pmml.PMML;
import org.jpmml.converter.CategoricalFeature;
import org.jpmml.converter.ContinuousFeature;
import org.jpmml.converter.Feature;
import org.jpmml.converter.Label;
import org.jpmml.converter.ModelEncoder;
import org.jpmml.converter.ScalarLabelUtil;
import org.jpmml.converter.Schema;
import org.jpmml.converter.TypeUtil;
import org.jpmml.model.visitors.FieldRenamer;

public class RExpEncoder extends ModelEncoder {

	private Label label = null;

	private List<Feature> features = new ArrayList<>();

	private Map<String, String> renamedFields = new LinkedHashMap<>();


	@Override
	public PMML encodePMML(Model model){
		PMML pmml = super.encodePMML(model);

		Collection<Map.Entry<String, String>> entries = this.renamedFields.entrySet();
		for(Map.Entry<String, String> entry : entries){
			FieldRenamer renamer = new FieldRenamer(entry.getKey(), entry.getValue());

			renamer.applyTo(pmml);
		}

		return pmml;
	}

	public void addFields(RExpEncoder encoder){
		Map<String, DataField> dataFields = encoder.getDataFields();
		Map<String, DerivedField> derivedFields = encoder.getDerivedFields();

		for(String name : dataFields.keySet()){
			DataField dataField = getDataField(name);

			if(dataField == null){
				dataField = dataFields.get(name);

				addDataField(dataField);
			}
		}

		for(String name : derivedFields.keySet()){
			DerivedField derivedField = getDerivedField(name);

			if(derivedField == null){
				derivedField = derivedFields.get(name);

				addDerivedField(derivedField);
			}
		}
	}

	@Override
	public DataField createDataField(String name, OpType opType, DataType dataType, List<?> values){

		if(dataType == null){
			dataType = TypeUtil.getDataType(values);
		}

		return super.createDataField(name, opType, dataType, values);
	}

	public Schema createSchema(){
		return new Schema(this, getLabel(), getFeatures());
	}

	public void setLabel(DataField dataField){
		setLabel(ScalarLabelUtil.createScalarLabel(dataField));
	}

	public void addFeature(Field<?> field){
		Feature feature;

		OpType opType = field.requireOpType();
		switch(opType){
			case CONTINUOUS:
				feature = new ContinuousFeature(this, field);
				break;
			case CATEGORICAL:
				feature = new CategoricalFeature(this, (DataField)field);
				break;
			default:
				throw new IllegalArgumentException();
		}

		addFeature(feature);
	}

	public void addFeature(Feature feature){
		List<Feature> features = getFeatures();

		features.add(feature);
	}

	public Label getLabel(){
		return this.label;
	}

	public void setLabel(Label label){
		this.label = label;
	}

	public List<Feature> getFeatures(){
		return this.features;
	}

	public void renameField(String name, String renamedName){
		this.renamedFields.put(name, renamedName);
	}
}