% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/scan1perm.R
\name{scan1perm}
\alias{scan1perm}
\title{Permutation test for genome scan with a single-QTL model}
\usage{
scan1perm(
  genoprobs,
  pheno,
  kinship = NULL,
  addcovar = NULL,
  Xcovar = NULL,
  intcovar = NULL,
  weights = NULL,
  reml = TRUE,
  model = c("normal", "binary"),
  n_perm = 1,
  perm_Xsp = FALSE,
  perm_strata = NULL,
  chr_lengths = NULL,
  cores = 1,
  ...
)
}
\arguments{
\item{genoprobs}{Genotype probabilities as calculated by
\code{\link[=calc_genoprob]{calc_genoprob()}}.}

\item{pheno}{A numeric matrix of phenotypes, individuals x phenotypes.}

\item{kinship}{Optional kinship matrix, or a list of kinship matrices (one
per chromosome), in order to use the LOCO (leave one chromosome
out) method.}

\item{addcovar}{An optional numeric matrix of additive covariates.}

\item{Xcovar}{An optional numeric matrix with additional additive covariates used for
null hypothesis when scanning the X chromosome.}

\item{intcovar}{An optional numeric matrix of interactive covariates.}

\item{weights}{An optional numeric vector of positive weights for the
individuals. As with the other inputs, it must have \code{names}
for individual identifiers.}

\item{reml}{If \code{kinship} provided: if \code{reml=TRUE}, use
REML; otherwise maximum likelihood.}

\item{model}{Indicates whether to use a normal model (least
squares) or binary model (logistic regression) for the phenotype.
If \code{model="binary"}, the phenotypes must have values in \eqn{[0, 1]}.}

\item{n_perm}{Number of permutation replicates.}

\item{perm_Xsp}{If TRUE, do separate permutations for the autosomes
and the X chromosome.}

\item{perm_strata}{Vector of strata, for a stratified permutation
test. Should be named in the same way as the rows of
\code{pheno}. The unique values define the strata.}

\item{chr_lengths}{Lengths of the chromosomes; needed only if
\code{perm_Xsp=TRUE}. See \code{\link[=chr_lengths]{chr_lengths()}}.}

\item{cores}{Number of CPU cores to use, for parallel calculations.
(If \code{0}, use \code{\link[parallel:detectCores]{parallel::detectCores()}}.)
Alternatively, this can be links to a set of cluster sockets, as
produced by \code{\link[parallel:makeCluster]{parallel::makeCluster()}}.}

\item{...}{Additional control parameters; see Details.}
}
\value{
If \code{perm_Xsp=FALSE}, the result is matrix of
genome-wide maximum LOD scores, permutation replicates x
phenotypes. If \code{perm_Xsp=TRUE}, the result is a list of
two matrices, one for the autosomes and one for the X
chromosome. The object is given class \code{"scan1perm"}.
}
\description{
Permutation test for a enome scan with a single-QTL model by
Haley-Knott regression or a linear mixed model, with possible
allowance for covariates.
}
\details{
If \code{kinship} is not provided, so that analysis proceeds by
Haley-Knott regression, we permute the rows of the phenotype data;
the same permutations are also applied to the rows of the
covariates (\code{addcovar}, \code{Xcovar}, and \code{intcovar})
are permuted.

If \code{kinship} is provided, we instead permute the rows of the
genotype data and fit an LMM with the same residual heritability
(estimated under the null hypothesis of no QTL).

If \code{Xcovar} is provided and \code{perm_strata=NULL}, we do a
stratified permutation test with the strata defined by the rows of
\code{Xcovar}. If a simple permutation test is desired, provide
\code{perm_strata} that is a vector containing a single repeated
value.

The \code{...} argument can contain several additional control
parameters; suspended for simplicity (or confusion, depending on
your point of view). \code{tol} is used as a tolerance value for linear
regression by QR decomposition (in determining whether columns are
linearly dependent on others and should be omitted); default
\code{1e-12}. \code{maxit} is the maximum number of iterations for
converence of the iterative algorithm used when \code{model=binary}.
\code{bintol} is used as a tolerance for converence for the iterative
algorithm used when \code{model=binary}. \code{eta_max} is the maximum value
for the "linear predictor" in the case \code{model="binary"} (a bit of a
technicality to avoid fitted values exactly at 0 or 1).
}
\examples{
# read data
iron <- read_cross2(system.file("extdata", "iron.zip", package="qtl2"))
\dontshow{iron <- iron[,c(10,18,"X")]}

# insert pseudomarkers into map
map <- insert_pseudomarkers(iron$gmap, step=1)

# calculate genotype probabilities
probs <- calc_genoprob(iron, map, error_prob=0.002)

# grab phenotypes and covariates; ensure that covariates have names attribute
pheno <- iron$pheno
covar <- match(iron$covar$sex, c("f", "m")) # make numeric
names(covar) <- rownames(iron$covar)
Xcovar <- get_x_covar(iron)

# strata for permutations
perm_strata <- mat2strata(Xcovar)

# permutations with genome scan (just 3 replicates, for illustration)
operm <- scan1perm(probs, pheno, addcovar=covar, Xcovar=Xcovar,
                   n_perm=3, perm_strata=perm_strata)
summary(operm)

# leave-one-chromosome-out kinship matrices
kinship <- calc_kinship(probs, "loco")

# permutations of genome scan with a linear mixed model
\donttest{
operm_lmm <- scan1perm(probs, pheno, kinship, covar, Xcovar, n_perm=3,
                       perm_Xsp=TRUE, perm_strata=perm_strata,
                       chr_lengths=chr_lengths(map))
summary(operm_lmm)
}

}
\references{
Churchill GA, Doerge RW (1994) Empirical threshold
values for quantitative trait mapping. Genetics 138:963--971.

Manichaikul A, Palmer AA, Sen S, Broman KW (2007) Significance
thresholds for quantitative trait locus mapping under selective
genotyping. Genetics 177:1963--1966.

Haley CS, Knott SA (1992) A simple regression method for mapping
quantitative trait loci in line crosses using flanking markers.
Heredity 69:315--324.

Kang HM, Zaitlen NA, Wade CM, Kirby A, Heckerman D, Daly MJ, Eskin
E (2008) Efficient control of population structure in model
organism association mapping. Genetics 178:1709--1723.
}
\seealso{
\code{\link[=scan1]{scan1()}}, \code{\link[=chr_lengths]{chr_lengths()}}, \code{\link[=mat2strata]{mat2strata()}}
}
