\name{ratiocalc}
\alias{ratiocalc}

\title{Calculation of ratios from qPCR runs with/without reference genes}

\description{For multiple qPCR data from type 'pcrbatch', this function calculates ratios between two samples (control/treatment) of a gene-of-interest, using normalization against a reference gene, if supplied. The input can be single qPCR data or (more likely) data containing replicates. Errors and confidence intervals for the obtained ratios can be calculated by Monte-Carlo simulation, a permutation approach similar to the popular REST software and by error propagation. Statistical significance for the ratios is calculated by a permutation approach of randomly reallocated vs. non-reallocated data. See 'Details'.
}

\usage{
ratiocalc(data, group = NULL, which.eff = c("sig", "sli", "exp", "mak"), 
          type.eff = c("individual", "mean.single", "median.single", 
                       "mean.pair", "median.pair"),
          which.cp = c("cpD2", "cpD1", "cpE", "cpR", "cpT", "Cy0"),
          ...)
}

\arguments{
  \item{data}{multiple qPCR data generated by \code{\link{pcrbatch}}.}
  \item{group}{a character vector defining the replicates (if any) of control/treatment samples and reference genes/genes-of-interest. See 'Details'}.
  \item{which.eff}{efficiency calculated by which method. Defaults to sigmoidal fit. See output of \code{\link{pcrbatch}}. Alternatively, a fixed numeric value between 1 and 2 for all runs or a vector of external efficiencies with one element per run.}
  \item{type.eff}{type of efficiency pre-processing prior to error analysis. See 'Details'.}
  \item{which.cp}{type of threshold cycles to be used for the analysis. See output of \code{\link{efficiency}}. Alternatively, a vector of external threshold cycles with one element per run.}   
  \item{...}{other parameters to be passed to \code{\link{propagate}}.}
}

\details{
The replicates for the data columns are to be defined as a character vector with the following abbreviations:\cr

"gs":   gene-of-interest in treatment sample\cr
"gc":   gene-of-interest in control sample\cr
"rs":   reference gene in treatment sample\cr
"rc":   reference gene in control sample\cr

There is no distinction between the different runs of the same sample, so that three different runs of a gene-of-interest in a treatment sample are defined as c("gs", "gs", "gs"). The error analysis calculates statistics from ALL replicates, so that a further sub-categorization of runs is superfluous. NOTE: If the setup consists of different sample or gene combinations, use \code{\link{ratiobatch}}!

Examples:\cr
No replicates: NULL.\cr
2 runs with 2 replicates each, no reference gene: c("gs", "gs", "gs", "gs", "gc", "gc", "gc", "gc").\cr
1 run with two replicates each and reference gene:  c("gs", "gs", "gc", "gc", "rs", "rs", "rc", "rc").\cr

\code{type.eff} defines the pre-processing of the efficiencies before being transferred to \code{\link{propagate}}. The qPCR community sometimes uses single efficiencies, or averaged over replicates etc., so that different settings were implemented. In detail, these are the following:\cr

"individual":     The individual efficiencies from each run are used.\cr
"mean.single":    Efficiencies are averaged over all replicates.\cr
"median.single":  Same as above but median instead of mean.\cr 
"mean.pair":      Efficiencies are averaged from all replicates of treatment sample AND control.\cr
"median.pair":    Same as above but median instead of mean.\cr

Efficiencies can be taken from the individual curves or averaged from the replicates as described in the documentation to \code{\link{ratiocalc}}. The different combinations of \code{type.eff}, \code{which.eff} and \code{which.cp} can yield very different results in ratio calculation. We observed a relatively stable setup which minimizes the overall variance using the combination
  
\code{type.eff = "mean.single"}     # averaging efficiency across replicates\cr
\code{which.eff = "sli"}            # taking efficiency from the sliding window method\cr
\code{which.cp = "sig"}             # using the second derivative maximum of a sigmoidal fit\cr 

The ratios are calculated according to the following formulas:\cr
Without reference gene: \deqn{\frac{E_{gs}^{cp_{gs}}}{E_{gc}^{cp_{gc}}}}
With reference gene: \deqn{\frac{E_{gs}^{cp_{gs}}}{E_{gc}^{cp_{gc}}}/\frac{E_{rs}^{cp_{rs}}}{E_{rc}^{cp_{rc}}}}  

The permutation approach permutates threshold cycles and efficiency replicates within treatment and control samples. The treatment/control samples (and their respective efficiencies) are tied together, which is similar to the popular REST software approach ("pairwise-reallocation test"). Ratios are calculated for each permutation and compared to ratios obtained if samples were randomly reallocated from the treatment to the control group. Three p-values are calculated from all permutations that gave a higher/equal/lower ratio than the original data. The resulting p-values are thus an indication for the significance in any direction AGAINST the null hypothesis that ratios calculated by permutation are just by chance.

If the mechanistic \code{mak3/mak3n/chag} models are used in \code{\link{pcrbatch}}, this is automatically recognized and the ratio calculation is conducted from the \eqn{D0} values of the model:\cr
Without reference gene: \deqn{\frac{D0_{gs}}{D0_{gc}}}
With reference gene: \deqn{\frac{D0_{gs}}{D0_{gc}}/\frac{D0_{rs}}{D0_{rc}}}

Confidence values are returned for all three methods (Monte Carlo, permutation, error propagation) as follows:\cr
Monte-Carlo:  From the evaluations of the Monte-Carlo simulated data.\cr
Permutation:  From the evaluations of the within-group permutated data.\cr
Propagation:  From the propagated error, assuming normality.
}

\value{
A list with the following components:\cr
\code{data}: the data that was transferred to \code{\link{propagate}} for the error analysis.\cr
\code{data.Sim, data.Perm, data.Prop, derivs, covMat}: The complete output from \code{\link{propagate}}.\cr
\code{summary}: a summary of the results obtained from the Monte Carlo simulation, permutation and error propagation. 
}

\author{
Andrej-Nikolai Spiess
}

\note{
The error calculated from qPCR data by \code{\link{propagate}} often seems quite high. This largely depends on the error of the exponent (i.e. threshold cycles) of the exponential function. The error usually decreases when setting \code{use.cov = TRUE} in the \code{...} part of the function. It can be debated anyhow, if the variables 'efficiency' and 'threshold cycles' have a covariance structure. As the efficiency is deduced at the second derivative maximum of the sigmoidal curve, variance in the second should show an effect on the first, such that the use of a var-covar matrix might be feasible. It is also commonly encountered that the propagated error is much higher when using reference genes, as the number of partial derivative functions increases.
}

\references{
Analysis of relative gene expression data using real-time quantitative PCR and the 2(-Delta Delta C(T)) method.\cr
Livak KJ & Schmittgen TD.\cr
\emph{Methods} (2001), \bold{25}: 402-428.\cr

Standardized determination of real-time PCR efficiency from a single reaction set-up.\cr
Tichopad A, Dilger M, Schwarz G, Pfaffl MW.\cr
\emph{Nucleic Acids Res} (2003), \bold{31}: e122.\cr

Validation of a quantitative method for real time PCR kinetics.\cr
Liu W & Saint DA.\cr
\emph{Biochem Biophys Res Commun} (2002), \bold{294}: 347-53.\cr

Relative expression software tool (REST) for group-wise comparison and statistical analysis of relative expression results in real-time PCR.\cr
Pfaffl MW, Horgan GW, Dempfle L.\cr
\emph{Nucl Acids Res} (2002), \bold{30}: e36. 
}


\examples{
## Only treatment sample and control,
## no reference gene, 4 replicates each.
## Individual efficiencies for error calculation. 
DAT1 <- pcrbatch(reps, fluo = 2:9, model = l4)
GROUP1 <- c("gs", "gs", "gs", "gs", "gc", "gc", "gc", "gc")
RES1 <- ratiocalc(DAT1, group = GROUP1, which.eff = "sli",
                 type.eff = "individual", which.cp = "cpD2")       
RES1$summary

## Gets even better using averaged efficiencies 
## over all replicates.
## p-value indicates significant upregulation in 
## comparison to randomly reallocated 
## samples (similar to REST software)
RES2 <- ratiocalc(DAT1, GROUP1, which.eff = "sli", 
                  type.eff = "mean.single", which.cp = "cpD2")                 
RES2$summary    

## failed fits and sigmoidal outliers 
## using 'testdat' set.
## GROUP is automatically reduced to
## runs that passed checking.
DAT3 <- pcrbatch(testdat[, 1:9], fluo = 2:9, model = l5)
GROUP3 <- c("gs", "gs", "gs", "gs", "gc", "gc", "gc", "gc")
RES3 <- ratiocalc(DAT3, GROUP3, which.eff = "sli",
                 type.eff = "individual", which.cp = "cpD2")       
RES3$summary

\dontrun{
## Using reference data.
## Toy example is same data as above
## but replicated as reference such
## that the ratio should be 1.
DAT4 <- pcrbatch(reps, fluo = c(2:9, 2:9), model = l4)
GROUP4 <- c("gs", "gs", "gs", "gs", 
            "gc", "gc", "gc", "gc",
            "rs", "rs", "rs", "rs",
            "rc", "rc", "rc", "rc")
RES4 <- ratiocalc(DAT4, GROUP4, which.eff = "sli", 
                  type.eff = "mean.single", which.cp = "cpD2")                  
RES4$summary

## Using one of the mechanistic models
## => ratios are calculated from the replicate
## D0 values. Without/with reference gene.
DAT5 <- pcrbatch(reps, fluo = 2:9, do.mak = "mak3")
GROUP5 <- c("gs", "gs", "gs", "gs", "gc", "gc", "gc", "gc")
RES5 <- ratiocalc(DAT5, GROUP5, which.eff = "mak")
RES5$summary

## Example without replicates 
## => no Monte-Carlo simulations
## and no plots.
DAT6 <- pcrbatch(reps, fluo = 2:5, model = l4)
GROUP6 <- c("gs", "gc", "rs", "rc")
RES6 <- ratiocalc(DAT6, GROUP6, which.eff = "sli", 
                  type.eff = "individual", which.cp = "cpD2")
RES6$summary

## Using external efficiencies
DAT7 <- pcrbatch(reps, fluo = 2:9, model = l5)
GROUP7 <- c("gs", "gs", "gs", "gs", "gc", "gc", "gc", "gc")
EFF7 <- rep(c(1.72, 1.76), c(4, 4)) 
RES7 <- ratiocalc(DAT7, GROUP7, which.eff = EFF7, 
                 type.eff = "individual", which.cp = "cpD2")       
RES7$summary

## Using external efficiencies AND
## external threshold cycles
DAT8 <- pcrbatch(reps, fluo = 2:9, model = l5)
GROUP8 <- c("gs", "gs", "gs", "gs", "gc", "gc", "gc", "gc")
EFF8 <- rep(c(1.72, 1.76), c(4, 4))
CP8 <- c(15.44, 15.33, 14.84, 15.34, 18.89, 18.71, 18.13, 17.22)
RES8 <- ratiocalc(DAT8, GROUP8, which.eff = EFF8, 
                 type.eff = "individual", which.cp = CP8)       
RES8$summary

## Compare 'propagate' to REST software
## using the data from the REST 2008
## manual (http://rest.gene-quantification.info/),
## Have to create dataframe with values as we do
## not use 'pcrbatch', but external cp's & eff's!
## Ties define random reallocation of crossing points
## keeping controls and samples together.
## See help for 'propagate'.
EXPR <- expression((2.01^(cp.gc - cp.gs)/1.97^(cp.rc - cp.rs)))
cp.rc <- c(26.74, 26.85, 26.83, 26.68, 27.39, 27.03, 26.78, 27.32)
cp.rs <- c(26.77, 26.47, 27.03, 26.92, 26.97, 26.97, 26.07, 26.3, 26.14, 26.81)
cp.gc <- c(27.57, 27.61, 27.82, 27.12, 27.76, 27.74, 26.91, 27.49)
cp.gs <- c(24.54, 24.95, 24.57, 24.63, 24.66, 24.89, 24.71, 24.9, 24.26, 24.44)
DAT9 <- qpcR:::cbind.na(cp.rc, cp.rs, cp.gc, cp.gs)
RES9 <- propagate(EXPR, DAT9, do.sim = TRUE, do.perm = TRUE, 
                  ties = c(1, 2, 1, 2), verbose = TRUE)
RES9$summary
}
}




\keyword{nonlinear}

