% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/setup.R
\name{cohend}
\alias{cohend}
\title{Use Cohen's d as the effect size in \code{study_parameters}}
\usage{
cohend(ES, standardizer = "pretest_SD", treatment = "control")
}
\arguments{
\item{ES}{\code{numeric}; value of the standardized effect size. Can be a vector.}

\item{standardizer}{\code{character}; the standardizer (denominator) used to calculate
Cohen's d. Allows options are: "pretest_SD", "posttest_SD", or "slope_SD".
See Details from more information.}

\item{treatment}{\code{character}; indicates if the \code{standardizer} should
be based on the "treatment" or "control" group---this only matters for 3-level partially
nested designs.}
}
\value{
A \code{list} of the same length as \code{ES}. Each element is a named list
of class \code{plcp_cohend}, with the elements:
\itemize{
\item \code{set}: A helper \code{function} that converts the standardized ES to raw values.
Accepts a \code{study_parameters} objects, and returns a \code{numeric} indicating the
raw difference between the treatment at posttest.
\item \code{get}: contains a list with the original call: "ES", "standardizer", and "treatment".
}
}
\description{
This function is used as input to the \code{effect_size} argument in \code{study_parameters},
if standardized effect sizes should be used. The choice of the denominator differs between fields,
and this function supports the common ones: pre- or posttest SD, or the random slope SD.
}
\details{
\strong{Standardizing using the \code{pretest_SD} or \code{posttest_SD}}

For these effect sizes, ES indicates the standardized difference between
the treatment groups at posttest (\code{T_end}), standardized by using
either the implied standard deviation at pretest or posttest. Thus, the actual
raw differences in average slopes between the treatments are,

\code{slope_diff = (ES * SD)/T_end}.

\strong{\code{slope_SD}: standardizing using the random slopes}

This standardization is quite different from using the pretest or posttest SD.
Here the average slope difference is standardized using the total SD of the random slopes.
This is done by e.g. Raudenbush and Liu (2001). \strong{NB}, for this effect size
\code{ES} indicates the difference in change per unit time, and not at posttest. Thus, the raw
difference in average slopes is,

\code{slope_diff = ES * slope_SD}.

For a 3-level model, \code{slope_SD = sqrt(sigma_subject_slope^2 + sigma_cluster_slope^2)}.
}
\examples{

# Pretest SD
p <- study_parameters(n1 = 11,
                      n2 = 20,
                      icc_pre_subject = 0.5,
                      cor_subject = -0.4,
                      var_ratio = 0.03,
                      effect_size = cohend(0.4, standardizer = "pretest_SD"))

get_slope_diff(p)

# using posttest SD,
# due to random slope SD will be larger at posttest
# thus ES = 0.4 indicate larger raw slope difference
# using posttest SD
p <- update(p, effect_size = cohend(0.4,
                                    standardizer = "posttest_SD"))
get_slope_diff(p)


# Random slope SD
p <- study_parameters(n1 = 11,
                      n2 = 20,
                      icc_pre_subject = 0.5,
                      cor_subject = -0.4,
                      var_ratio = 0.03,
                      effect_size = cohend(0.4, standardizer = "slope_SD"))

# Partially nested ----------------------------------------------------------
p <- study_parameters(n1 = 11,
                      n2 = 20,
                      n3 = 4,
                      icc_pre_subject = 0.5,
                      icc_pre_cluster = 0.25,
                      cor_subject = -0.4,
                      var_ratio = 0.03,
                      partially_nested = TRUE,
                      effect_size = cohend(0.4, standardizer = "pretest_SD")
                      )
# Default is to use control groups SD
get_slope_diff(p)

# Treatment group's SD also include cluster-level intercept variance.
# Thus, ES of 0.4 will indicate a larger raw difference
# using the treatment group's SD
p <- update(p, effect_size = cohend(0.4,
                                    standardizer = "pretest_SD",
                                    treatment = "treatment"))
get_slope_diff(p)

## Combine multiple values, and raw and standardized effects ----------------
p <- study_parameters(n1 = 11,
                      n2 = 20,
                      icc_pre_subject = 0.5,
                      cor_subject = -0.4,
                      var_ratio = 0.03,
                      effect_size = c(-5, 9,
                                      cohend(c(0.5, 0.8), standardizer = "pretest_SD"),
                                      cohend(c(0.5, 0.8), standardizer = "posttest_SD")))


## Recreate results in Raudenbush & Liu 2001 --------------------------------
rauden_liu <- function(D, f, n = 238) {
    n1 <- f * D + 1
    p <- study_parameters(n1 = n1,
                          n2 = n/2,
                          T_end = D,
                          sigma_subject_intercept = sqrt(0.0333),
                          sigma_subject_slope = sqrt(0.0030),
                          sigma_error = sqrt(0.0262),
                          effect_size = cohend(0.4, standardizer = "slope_SD"))
    x <- get_power(p)
    round(x$power, 2)
}

## Table 1 in Raudenbush & Liu 2001
## NB, it looks like they made an error in column 1.
g <- expand.grid(D = 2:8,
                 f = c(0.5, 1:6))
g$power <- mapply(rauden_liu, D = g$D, f = g$f)
tidyr::spread(g, f, power)


## Table 3 Table 1 in Raudenbush & Liu 2001
g <- expand.grid(n = seq(100, 800, by = 100),
                 D = 4,
                 f = c(0.5, 1:6))
g$power <- mapply(rauden_liu, n = g$n, f = g$f, D = g$D)
tidyr::spread(g, n, power)

}
\references{
Raudenbush, S. W., & Liu, X. F. (2001). Effects of study duration,
frequency of observation, and sample size on power in studies of group differences
in polynomial change. \emph{Psychological methods}, 6(4), 387.
}
\seealso{
\code{\link{study_parameters}}
}
