% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ex_to_lifetab.R
\name{ex_to_lifetab_brass}
\alias{ex_to_lifetab_brass}
\title{Derive Life Tables that Match Life Expectancies,
using a Brass Logit Model}
\usage{
ex_to_lifetab_brass(
  target,
  standard,
  infant = c("constant", "linear", "CD", "AK"),
  child = c("constant", "linear", "CD"),
  closed = c("constant", "linear"),
  open = "constant",
  radix = 1e+05,
  suffix = NULL
)
}
\arguments{
\item{target}{A data frame containing a variable called
\code{"ex"}, and possibly others. See Details.}

\item{standard}{A data frame containing variables
called \code{age} and \code{lx}, and possibly others.
See details.}

\item{infant, child, closed, open}{Methods used to
calculate life expectancy. See \code{\link[=lifetab]{lifetab()}} for details.}

\item{radix}{Initial population for the
\code{lx} column in the derived life table(s).
Default is \code{100000}.}

\item{suffix}{Optional suffix added to life table columns.}
}
\value{
A data frame containing one or more life tables.
}
\description{
Turn life expectancies at birth into full life tables,
using the Brass logit model. The method is simple
and is designed for simulations or for
settings with little or no data on age-specific
mortality rates. In settings where data
on age-specific mortality is available,
other methods might be more appropriate.
}
\section{Method}{


The method implemented by \code{ex_to_lifetab_brass()} is
based on the observation that, if populations A and B
are demographically similar, then, in many cases,

\deqn{\text{logit}(l_x^{\text{B}}) \approx \alpha + \beta \text{logit}(l_x^{\text{A}})}

where \eqn{l_x} is the "survivorship probability" quantity
from a life table. When populations are
similar, \eqn{beta} is often close to 1.

Given (i) target life expectancy,
(ii) a set of \eqn{l_x^{\text{A}}}),
(referred to as a "standard"), and
(iii) a value for \eqn{\beta},
\code{ex_to_lifetab_brass()} finds
a value for \eqn{\alpha} that yields a set of
\eqn{l_x^{\text{B}}}) with the required life expectancy.
}

\section{\code{target} argument}{


\code{target} is a data frame specifying
life expectancies for each population being modelled,
and, possibly, inputs to the calculations, and
index variables. Values in \code{target} are not age-specific.
\itemize{
\item A variable called \code{"ex"}, with life expectancy at birth
must be included in \code{target}.
\item A variable called \code{"beta"} with values
for \code{beta} can be included in \code{target}.
This variable can be an \link[rvec:rvec]{rvec}.
If no \code{"beta"} variable is included in \code{target},
then \code{ex_to_lifetab_brass()} assumes that
\eqn{beta \equiv 1}.
\item A variable called \code{"sex"}. If the \code{infant}
argument to \code{ex_to_lifetab_brass()} is is \code{"CD"} or \code{"AK"},
or if the \code{child} argument is \code{"CD"},
\code{target} must include a \verb{"sex" variable, and the labels for this variable must be interpretable by function [format_sex()]. Otherwise, the }"sex"` variable  is optional,
and there is no restriction on labels.
\item Other variables used to distinguish between
life expectancies, such as time, region,
or model variant.
}
}

\section{\code{standard} argument}{


\code{standard} is a data frame specifying
the \eqn{l_x} to be used with each life expectancy
in \code{ex}, and, optionally, values the average age
person-years lived by people who die in each group,
\eqn{_na_x}. Values in \code{standard} are age-specific.
\itemize{
\item A variable called \code{"age"}, with labels that
can be parsed by \code{\link[=reformat_age]{reformat_age()}}.
\item A variable called \code{"lx"}.
Internally each set of \eqn{l_x} is are standardized
so that the value for age 0 equals 1.
Within each set, values must be non-increasing.
Cannot be an rvec.
\item Additional variables used to match rows in \code{standard}
to rows in \code{target}.
}

Internally, \code{standard} is merged with
\code{target} using a left join from \code{target},
on any variables that \code{target}
and \code{standard} have in common.
}

\examples{
## create new life tables based on level-1
## 'West' model life tables, but with lower
## life expectancy

library(dplyr, warn.conflicts = FALSE)

target <- data.frame(sex = c("Female", "Male"), 
                     ex = c(17.5, 15.6))

standard <- west_lifetab |>
    filter(level == 1) |>
    select(sex, age, lx)
    
ex_to_lifetab_brass(target = target,
                    standard = standard,
                    infant = "CD",
                    child = "CD")
}
\references{
Brass W, Coale AJ. 1968. “Methods of analysis and estimation,”
in Brass, W,  Coale AJ, Demeny P, Heisel DF, et al. (eds).
The Demography of Tropical Africa. Princeton NJ:
Princeton University Press, pp. 88–139.

Moultrie TA, Timæus IM. 2013. Introduction to Model Life Tables.
In Moultrie T, Dorrington R, Hill A, Hill K, Timæus I, Zaba B.
(eds). Tools for Demographic Estimation.
Paris: International Union for the Scientific Study of Population.
\href{https://demographicestimation.iussp.org/content/using-models-derive-life-tables-incomplete-data}{online version}.
}
\seealso{
\itemize{
\item \code{\link[=logit]{logit()}}, \code{\link[=invlogit]{invlogit()}} Logit function
\item \code{\link[=lifeexp]{lifeexp()}} Calculate life expectancy from detailed inputs
}
}
