% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/round_robin.R
\name{pgen}
\alias{pgen}
\title{Genotype Probability}
\usage{
pgen(gid, pop = NULL, by_pop = TRUE, log = TRUE, freq = NULL)
}
\arguments{
\item{gid}{a genind or genclone object.}

\item{pop}{either a formula to set the population factor from the
\code{\link{strata}} slot or a vector specifying the population factor for
each sample. Defaults to \code{NULL}.}

\item{by_pop}{When this is \code{TRUE} (default), the calculation will be
done by population.}

\item{log}{a \code{logical} if \code{log =TRUE} (default), the values
returned will be log(Pgen). If \code{log = FALSE}, the values returned will
be Pgen.}

\item{freq}{a vector or matrix of allele frequencies. This defaults to 
\code{NULL}, indicating that the frequencies will be determined via 
round-robin approach in \code{\link{rraf}}. \strong{If \code{by_pop =
TRUE}, and this matrix or vector is not provided, zero-value allele 
frequencies will automatically be corrected by 1/n.}}
}
\value{
A vector containing Pgen values per locus for each genotype in the 
  object.
}
\description{
Genotype Probability
}
\details{
Pgen is the probability of a given genotype occuring in a population
  assuming HWE. Thus, the value for diploids is 
  
  \deqn{P_{gen} = \left(\prod_{i=1}^m p_i\right)2^h}{pgen = prod(p_i)*(2^h)} 
  
  where \eqn{p_i} are the allele frequencies and \emph{h} is the count of the
  number of heterozygous sites in the sample (Arnaud-Haond et al. 2007; Parks
  and Werth, 1993). The allele frequencies, by default, are calculated using
  a round-robin approach where allele frequencies at a particular locus are 
  calculated on the clone-censored genotypes without that locus.
  
  To avoid issues with numerical precision of small numbers, this function 
  calculates pgen per locus by adding up log-transformed values of allele 
  frequencies. These can easily be transformed to return the true value (see
  examples).
}
\note{
Any zero-value allele frequencies will be corrected by 1/n. This is to
  avoid any zero-probability genotypes. To counteract this, you can supply
  your own allele frequencies with the \code{freq} argument (see example).
  For haploids, Pgen at a particular locus is the allele frequency. This 
  function cannot handle polyploids. Additionally, when the argument 
  \code{pop} is not \code{NULL}, \code{by_pop} is automatically \code{TRUE}.
}
\examples{
data(Pram)
head(pgen(Pram, log = FALSE))

\dontrun{
# You can get the Pgen values over all loci by summing over the logged results:
exp(rowSums(pgen(Pram, log = TRUE, na.rm = TRUE)))

# You can also take the product of the non-logged results:
apply(pgen(Pram, log = FALSE), 1, prod, na.rm = TRUE)

## Dealing with zero-frequency allele correction
# By default, allele frequencies are calculated with rraf with 
# correction = TRUE. This is normally benign when analyzing large populations,
# but it can have a great effect on small populations. Here's a way to supply
# your own correction.

# First, calculate round robin allele frequencies by population with no 
# correction. There are many zero values.
(my_rraf <- rraf(Pram, by_pop = TRUE, correction = FALSE))

# When you run pgen with these zero value allele frequencies, the
# probabilities of some genotypes crash to zero.
head(pgen(Pram, log = FALSE, freq = my_rraf))

# One solution: set the allele frequencies to 1/[samples in data]:
my_rraf[my_rraf == 0] <- 1/nInd(Pram)

# Now we don't have genotype probabilites of zero.
head(pgen(Pram, log = FALSE, freq = my_rraf))
}
}
\author{
Zhian N. Kamvar, Jonah Brooks, Stacy A. Krueger-Hadfield, Erik Sotka
}
\references{
Arnaud-Haond, S., Duarte, C. M., Alberto, F., & Serrão, E. A. 2007.
Standardizing methods to address clonality in population studies.
\emph{Molecular Ecology}, 16(24), 5115-5139.

Parks, J. C., & Werth, C. R. 1993. A study of spatial features of clones in a
population of bracken fern, \emph{Pteridium aquilinum} (Dennstaedtiaceae).
\emph{American Journal of Botany}, 537-544.
}
\seealso{
\code{\link{psex}}, \code{\link{rraf}}, \code{\link{rrmlg}}
}

