% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/kwic.R
\docType{methods}
\name{kwic}
\alias{kwic}
\alias{kwic,context-method}
\alias{kwic,slice-method}
\alias{kwic,partition-method}
\alias{kwic,subcorpus-method}
\alias{kwic,corpus-method}
\alias{kwic,character-method}
\alias{kwic,remote_corpus-method}
\alias{kwic,remote_partition-method}
\alias{kwic,remote_subcorpus-method}
\alias{kwic,partition_bundle-method}
\alias{kwic,subcorpus_bundle-method}
\title{Perform keyword-in-context (KWIC) analysis.}
\usage{
kwic(.Object, ...)

\S4method{kwic}{context}(
  .Object,
  s_attributes = getOption("polmineR.meta"),
  cpos = TRUE,
  verbose = FALSE
)

\S4method{kwic}{slice}(
  .Object,
  query,
  cqp = is.cqp,
  left = getOption("polmineR.left"),
  right = getOption("polmineR.right"),
  s_attributes = getOption("polmineR.meta"),
  region = NULL,
  p_attribute = "word",
  boundary = NULL,
  cpos = TRUE,
  stoplist = NULL,
  positivelist = NULL,
  regex = FALSE,
  verbose = TRUE,
  ...
)

\S4method{kwic}{partition}(
  .Object,
  query,
  cqp = is.cqp,
  left = getOption("polmineR.left"),
  right = getOption("polmineR.right"),
  s_attributes = getOption("polmineR.meta"),
  p_attribute = "word",
  region = NULL,
  boundary = NULL,
  cpos = TRUE,
  stoplist = NULL,
  positivelist = NULL,
  regex = FALSE,
  verbose = TRUE,
  ...
)

\S4method{kwic}{subcorpus}(
  .Object,
  query,
  cqp = is.cqp,
  left = getOption("polmineR.left"),
  right = getOption("polmineR.right"),
  s_attributes = getOption("polmineR.meta"),
  p_attribute = "word",
  region = NULL,
  boundary = NULL,
  cpos = TRUE,
  stoplist = NULL,
  positivelist = NULL,
  regex = FALSE,
  verbose = TRUE,
  ...
)

\S4method{kwic}{corpus}(
  .Object,
  query,
  cqp = is.cqp,
  check = TRUE,
  left = as.integer(getOption("polmineR.left")),
  right = as.integer(getOption("polmineR.right")),
  s_attributes = getOption("polmineR.meta"),
  p_attribute = "word",
  region = NULL,
  boundary = NULL,
  cpos = TRUE,
  stoplist = NULL,
  positivelist = NULL,
  regex = FALSE,
  verbose = TRUE,
  ...
)

\S4method{kwic}{character}(
  .Object,
  query,
  cqp = is.cqp,
  check = TRUE,
  left = as.integer(getOption("polmineR.left")),
  right = as.integer(getOption("polmineR.right")),
  s_attributes = getOption("polmineR.meta"),
  p_attribute = "word",
  region = NULL,
  boundary = NULL,
  cpos = TRUE,
  stoplist = NULL,
  positivelist = NULL,
  regex = FALSE,
  verbose = TRUE,
  ...
)

\S4method{kwic}{remote_corpus}(.Object, ...)

\S4method{kwic}{remote_partition}(.Object, ...)

\S4method{kwic}{remote_subcorpus}(.Object, ...)

\S4method{kwic}{partition_bundle}(.Object, ..., verbose = FALSE)

\S4method{kwic}{subcorpus_bundle}(.Object, ...)
}
\arguments{
\item{.Object}{A (length-one) \code{character} vector with the name of a CWB
corpus, a \code{partition} or \code{context} object.}

\item{...}{Further arguments, used to ensure backwards compatibility. If
\code{.Object} is a \code{remote_corpus} of \code{remote_partition} object,
the three dots (\code{...}) are used to pass arguments. Hence, it is
necessary to state the names of all arguments to be passed explicity.}

\item{s_attributes}{Structural attributes (s-attributes) to include into
output table as metainformation.}

\item{cpos}{Logical, if \code{TRUE}, a \code{data.table} with the corpus
positions ("cpos") of the hits and their surrounding context will be
assigned to the slot "cpos" of the \code{kwic}-object that is returned.
Defaults to \code{TRUE}, as the availability of the cpos-\code{data.table}
will often be a prerequisite for further operations on the \code{kwic}
object. Omitting the table may however be useful to minimize memory
consumption.}

\item{verbose}{A \code{logical} value, whether to print messages.}

\item{query}{A query, CQP-syntax can be used.}

\item{cqp}{Either a logical value (\code{TRUE} if \code{query} is a CQP
query), or a function to check whether query is a CQP query or not
(defaults to auxiliary function \code{is.query}).}

\item{left}{A single \code{integer} value defining the number of tokens to the
left of the query match to include in the context. Advanced usage: (a) If
\code{left} is a length-one \code{character} vector stating an s-attribute, the
context will be expanded to the (left) boundary of the region where the
match occurs. (b) If \code{left} is a named length-one \code{integer} vector, this
value is the number regions of the structural attribute referred to by the
vector's name  to the left of the query match that are included in the
context.}

\item{right}{A single \code{integer} value, a length-one \code{character} vector or a
named length-one \code{integer} value, with equivalent effects to argument
\code{left}.}

\item{region}{An s-attribute, given by a length-one \code{character} vector.
The context of query matches will be expanded to the left and right
boundary of the region where the match is located. If arguments \code{left} and
\code{right} are > 1, the left and right boundaries of the respective number of
regions will be identified.}

\item{p_attribute}{The p-attribute, defaults to 'word'.}

\item{boundary}{If provided, a length-one character vector stating an
s-attribute that will be used to check the boundaries of the text.}

\item{stoplist}{Terms or ids to prevent a concordance from occurring in
results.}

\item{positivelist}{Terms or ids required for a concordance to occurr in
results}

\item{regex}{Logical, whether \code{stoplist}/\code{positivelist} is
interpreted as regular expression.}

\item{check}{A \code{logical} value, whether to check validity of CQP query
using \code{check_cqp_query}.}
}
\value{
If there are no matches, or if all (initial) matches are dropped due to the
application of a positivelist, a \code{NULL} is returned.
}
\description{
Get concordances for the matches for a query / perform keyword-in-context
(kwic) analysis.
}
\details{
The method works with a whole CWB corpus defined by a  character vector, and
can be applied on a \code{partition}- or a \code{context} object.

If \code{query} produces a lot of matches, the \code{DT::datatable()} function used to
produce output in the Viewer pane of RStudio may issue a warning. Usually,
this warning is harmless and can be ignored. Use
\code{options("polmineR.warn.size" = FALSE)} for turning off this warning.

If a \code{positivelist} is supplied, only those concordances will be kept that
have one of the terms from the \code{positivelist} occurr in the context of
the query match. Use argument \code{regex} if the positivelist should be
interpreted as regular expressions. Tokens from the positivelist will be
highlighted in the output table.

If a \code{negativelist} is supplied, concordances are removed if any of the
tokens of the \code{negativelist} occurrs in the context of the query match.

Applying the \code{kwic}-method on a \code{partition_bundle} or
\code{subcorpus_bundle} will return a single \code{kwic} object that
includes a column 'subcorpus_name' with the name of the \code{subcorpus}
(or \code{partition}) in the input object where the match for a concordance
occurs.
}
\examples{
use("polmineR")

# basic usage
K <- kwic("GERMAPARLMINI", "Integration")
if (interactive()) show(K)
oil <- corpus("REUTERS") \%>\% kwic(query = "oil")
if (interactive()) show(oil)
oil <- corpus("REUTERS") \%>\%
  kwic(query = "oil") \%>\%
  highlight(yellow = "crude")
if (interactive()) show(oil)

# increase left and right context and display metadata
K <- kwic(
  "GERMAPARLMINI",
  "Integration", left = 20, right = 20,
  s_attributes = c("date", "speaker", "party")
)
if (interactive()) show(K)

# use CQP syntax for matching
K <- kwic(
  "GERMAPARLMINI",
  '"Integration" [] "(Menschen|Migrant.*|Personen)"', cqp = TRUE,
  left = 20, right = 20,
  s_attributes = c("date", "speaker", "party")
)
if (interactive()) show(K)

# check that boundary of region is not transgressed
K <- kwic(
  "GERMAPARLMINI",
  '"Sehr" "geehrte"', cqp = TRUE,
  left = 100, right = 100,
  boundary = "date"
)
if (interactive()) show(K)

# use positivelist and highlight matches in context
K <- kwic("GERMAPARLMINI", query = "Integration", positivelist = "[Ee]urop.*", regex = TRUE)
K <- highlight(K, yellow = "[Ee]urop.*", regex = TRUE)

# Apply kwic on partition_bundle/subcorpus_bundle
gparl_2009_11_10_speeches <- corpus("GERMAPARLMINI") \%>\%
  subset(date == "2009-11-10") \%>\%
  as.speeches(s_attribute_name = "speaker", progress = FALSE, verbose = FALSE)
k <- kwic(gparl_2009_11_10_speeches, query = "Integration")
}
\references{
Baker, Paul (2006): \emph{Using Corpora in Discourse Analysis}. London: continuum, pp. 71-93 (ch. 4).

Jockers, Matthew L. (2014): \emph{Text Analysis with R for Students of Literature}.
Cham et al: Springer, pp. 73-87 (chs. 8 & 9).
}
\seealso{
The return value is a \code{\link{kwic-class}} object; the
documentation for the class explains the standard generic methods
applicable to \code{\link{kwic-class}} objects. It is possible to read the
whole text where a query match occurs, see the \code{\link{read}}-method.
To highlight terms in the context of a query match, see the
\code{\link{highlight}}-method.
}
