% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/async-http.R
\name{download_one_of}
\alias{download_one_of}
\title{Download a files from multiple candidate URLs}
\usage{
download_one_of(urls, destfile, etag_file = NULL,
  headers = character(), ...)
}
\arguments{
\item{urls}{A non-empty character vector of alternative URLs to try.}

\item{destfile}{Destination file.}

\item{etag_file}{If not \code{NULL} then the path to a file that may contain
the ETag of a previous request to this URL. If \code{destfile} exists, and
\code{etag_file} exists and it is not empty, then the \code{If-None-Match} HTTP
header is used with this ETag to avoid downloading the file if it has
not changed. If the file at \code{url} has changed, then it is downloaded,
and the the new ETag is stored in \code{etag_file}.}

\item{headers}{HTTP headers to add to the request, a named character
vector.}

\item{...}{Additional arguments are passed to \code{\link[=http_get]{http_get()}}.}
}
\value{
A \link{deferred} object. It resolves to a list with entries:
\itemize{
\item \code{url}: The URL in the request.
\item \code{destfile}: The destination file.
\item \code{response}: The response object from the curl package.
\item \code{etag}: The ETag of the response, of \code{NULL} if missing.
\item \code{etag_file}: The file the ETag was written to, or \code{NULL} otherwise
}
}
\description{
Uses \code{\link[=download_if_newer]{download_if_newer()}} to starts downloads in parallel, and the
download that completes first is kept. (The others will be cancelled.)
Download errors are ignored, as long as at least one download completes
successfully.
}
\details{
It also uses ETags, so if the destination file already exists, and one
of the URLs contain the same file (and this request completes first),
the file is not downloaded again.

If all URLs fail, then \code{download_one_of} throws an error of class
\code{download_one_of_error}. The error object contains all errors from
the underlying \code{\link[=download_if_newer]{download_if_newer()}} calls, in a list, in the
\code{errors} member.
}
\section{Examples}{
\preformatted{dest <- tempfile()
## The first URL answers after a 1s delay,
## the second after a 10s delay,
## the third throws an error immediately, so it will be ignored.
## Once the first URL responds, the second is cancelled, so the call
## will finish well before the 10s are over.
dl <- function() {
  download_one_of(
    c("https://httpbin.org/delay/1",
      "https://httpbin.org/delay/10",
      "https://httpbin.org/status/404"),
    dest)
}
system.time(res <- synchronise(dl()))
file.exists(dest)
readLines(dest)

## Which URL responded
res$response$url

## If all URLs fail
dl2 <- function() {
  download_one_of(
    c("https://httpbin.org/status/418",
      "https://httpbin.org/status/401"),
    tempfile()
  )
}
res <- tryCatch(synchronise(dl2()), error = function(e) e)
res
res$errors
cat(rawToChar(res$errors[[1]]$response$content))
}
}

\seealso{
Other async HTTP tools: \code{\link{download_file}},
  \code{\link{download_if_newer}}
}
\concept{async HTTP tools}
