% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cima-.r
\name{cima}
\alias{cima}
\title{Calculating Confidence Intervals}
\usage{
cima(y, se, v = NULL, alpha = 0.05, method = c("boot", "DL", "HK",
  "SJ", "KR", "APX", "PL", "BC"), B = 25000, parallel = FALSE,
  seed = NULL, maxit1 = 1e+05, eps = 10^(-10), lower = 0,
  upper = 1000, maxit2 = 1000, tol = .Machine$double.eps^0.25,
  rnd = NULL, maxiter = 100)
}
\arguments{
\item{y}{the effect size estimates vector}

\item{se}{the within studies standard errors vector}

\item{v}{the within studies variance estimates vector}

\item{alpha}{the alpha level of the prediction interval}

\item{method}{the calculation method for the pretiction interval (default = "boot").
\itemize{
\item \code{boot}: A parametric bootstrap confidence interval
           (Nagashima et al., 2018).
\item \code{DL}: A Wald-type t-distribution confidence interval
           (the DerSimonian & Laird estimator for \eqn{\tau^2} with
           an approximate variance estimator for the average effect,
           \eqn{(1/\sum{\hat{w}_i})^{-1}}, \eqn{df=K-1}).
\item \code{HK}: A Wald-type t-distribution confidence interval
           (the REML estimator for \eqn{\tau^2} with
           the Hartung (1999)'s varance estimator [the Hartung and
           Knapp (2001)'s estimator] for the average effect,
           \eqn{df=K-1}).
\item \code{SJ}: A Wald-type t-distribution confidence interval
           (the REML estimator for \eqn{\tau^2} with
           the Sidik and Jonkman (2006)'s bias coreccted SE estimator
           for the average effect, \eqn{df=K-1}).
\item \code{KR}: Partlett--Riley (2017) confidence interval /
           (the REML estimator for \eqn{\tau^2} with
           the Kenward and Roger (1997)'s approach
           for the average effect, \eqn{df=\nu}).
\item \code{APX}: A Wald-type t-distribution confidence interval /
           (the REML estimator for \eqn{\tau^2} with
           an approximate variance estimator for the average
           effect, \eqn{df=K-1}).
\item \code{PL}: Profile likelihood confidence interval
           (Hardy & Thompson, 1996).
\item \code{BC}: Profile likelihood confidence interval with
           Bartlett-type correction (Noma, 2011).            
}}

\item{B}{the number of bootstrap samples}

\item{parallel}{the number of threads used in parallel computing, or FALSE that means single threading}

\item{seed}{set the value of random seed}

\item{maxit1}{the maximum number of iteration for the exact distribution function of \eqn{Q}}

\item{eps}{the desired level of accuracy for the exact distribution function of \eqn{Q}}

\item{lower}{the lower limit of random numbers of \eqn{\tau^2}}

\item{upper}{the lower upper of random numbers of \eqn{\tau^2}}

\item{maxit2}{the maximum number of iteration for numerical inversions}

\item{tol}{the desired level of accuracy for numerical inversions}

\item{rnd}{a vector of random numbers from the exact distribution of \eqn{\tau^2}}

\item{maxiter}{the maximum number of iteration for REML estimation}
}
\value{
\itemize{
\item \code{K}: the number of studies.
\item \code{muhat}: the average treatment effect estimate \eqn{\hat{\mu}}.
\item \code{lci}, \code{uci}: the lower and upper confidence limits \eqn{\hat{\mu}_l} and \eqn{\hat{\mu}_u}.
\item \code{tau2h}: the estimate for \eqn{\tau^2}.
\item \code{i2h}: the estimate for \eqn{I^2}.
\item \code{nuc}: degrees of freedom for the confidence interval.
\item \code{vmuhat}: the variance estimate for \eqn{\hat{\mu}}.
}
}
\description{
This function calculates confidence intervals.
}
\details{
Excellent reviews of heterogeneity variance estimation
have been published (e.g., Veroniki, et al., 2018).
}
\examples{
data(sbp, package = "pimeta")
set.seed(20161102)

# Nagashima-Noma-Furukawa confidence interval
\donttest{pimeta::cima(sbp$y, sbp$sigmak, seed = 3141592)}

# A Wald-type t-distribution confidence interval
# An approximate variance estimator & DerSimonian-Laird estimator for tau^2
pimeta::cima(sbp$y, sbp$sigmak, method = "DL")

# A Wald-type t-distribution confidence interval
# The Hartung variance estimator & REML estimator for tau^2
pimeta::cima(sbp$y, sbp$sigmak, method = "HK")

# A Wald-type t-distribution confidence interval
# The Sidik-Jonkman variance estimator & REML estimator for tau^2
pimeta::cima(sbp$y, sbp$sigmak, method = "SJ")

# A Wald-type t-distribution confidence interval
# The Kenward-Roger approach & REML estimator for tau^2
pimeta::cima(sbp$y, sbp$sigmak, method = "KR")

# A Wald-type t-distribution confidence interval
# An approximate variance estimator & REML estimator for tau^2
pimeta::cima(sbp$y, sbp$sigmak, method = "APX")

# Profile likelihood confidence interval
# Maximum likelihood estimators of variance for the average effect & tau^2
pimeta::cima(sbp$y, sbp$sigmak, method = "PL")

# Profile likelihood confidence interval with a Bartlett-type correction
# Maximum likelihood estimators of variance for the average effect & tau^2
pimeta::cima(sbp$y, sbp$sigmak, method = "BC")
}
\references{
Veroniki, A. A., Jackson, D., Bender, R., Kuss, O.,
Langan, D., Higgins, J. P. T., Knapp, G.,  and Salanti, J. (2016).
Methods to calculate uncertainty in the estimated overall effect
size from a random-effects meta-analysis
\emph{Res Syn Meth.}
\emph{In press}.
\url{https://doi.org/10.1002/jrsm.1319}.

Nagashima, K., Noma, H., and Furukawa, T. A. (2018).
Prediction intervals for random-effects meta-analysis:
a confidence distribution approach.
\emph{Stat Methods Med Res}.
\emph{In press}.
\url{https://doi.org/10.1177/0962280218773520}.

Higgins, J. P. T, Thompson, S. G., Spiegelhalter, D. J. (2009).
A re-evaluation of random-effects meta-analysis.
\emph{J R Stat Soc Ser A Stat Soc.}
\strong{172}(1): 137-159.
\url{https://doi.org/10.1111/j.1467-985X.2008.00552.x}
 
Partlett, C, and Riley, R. D. (2017).
Random effects meta-analysis: Coverage performance of 95%
confidence and prediction intervals following REML estimation.
\emph{Stat Med.}
\strong{36}(2): 301-317.
\url{https://doi.org/10.1002/sim.7140}

Hartung, J., and Knapp, G. (2001).
On tests of the overall treatment effect in meta-analysis with
normally distributed responses.
\emph{Stat Med.}
\strong{20}(12): 1771-1782.
\url{https://doi.org/10.1002/sim.791}

Sidik, K., and Jonkman, J. N. (2006).
Robust variance estimation for random effects meta-analysis.
\emph{Comput Stat Data Anal.}
\strong{50}(12): 3681-3701.
\url{https://doi.org/10.1016/j.csda.2005.07.019}

Noma H. (2011)
Confidence intervals for a random-effects meta-analysis
based on Bartlett-type corrections.
\emph{Stat Med.}
\strong{30}(28): 3304-3312.
\url{https://doi.org/10.1002/sim.4350}
}
\seealso{
\code{\link[=pima]{pima}}
}
