% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fit_r_light.R
\name{fit_r_light2}
\alias{fit_r_light2}
\title{Fit models to estimate light respiration (\eqn{R_\mathrm{d}})}
\usage{
fit_r_light2(
  .data,
  .model = "default",
  .method = "ls",
  Q_lower = NA,
  Q_upper = NA,
  Q_levels = NULL,
  C_upper = NA,
  quiet = FALSE,
  brm_options = NULL
)
}
\arguments{
\item{.data}{A data frame containing plant ecophysiological data. See \code{\link[=required_variables]{required_variables()}} for the variables required for each model.}

\item{.model}{A character string of model name to use. See \code{\link[=get_all_models]{get_all_models()}}.}

\item{.method}{A character string of the statistical method to use: 'ls' for least-squares and 'brms' for Bayesian model using \code{\link[brms:brm]{brms::brm()}}. Default is 'ls'.}

\item{Q_lower}{Lower light intensity limit for estimating Rd using \code{kok_1956} and \code{yin_etal_2011} models.}

\item{Q_upper}{Upper light intensity limit for estimating Rd using \code{kok_1956} and \code{yin_etal_2011} models}

\item{Q_levels}{A numeric vector of light intensity levels (\eqn{\mu}mol / mol) for estimating \eqn{R_\mathrm{d}} from the linear region of the A-C curve using the \code{walker_ort_2015} model.}

\item{C_upper}{Upper C (\eqn{\mu}mol / mol) limit for estimating \eqn{R_\mathrm{d}} from the linear region of the A-C curve using the \code{walker_ort_2015} model.}

\item{quiet}{Flag. Should messages be suppressed? Default is FALSE.}

\item{brm_options}{A list of options passed to \code{\link[brms:brm]{brms::brm()}} if \code{.method = "brms"}. Default is NULL.}
}
\value{
\itemize{
\item If \code{.method = 'ls'}: an \code{\link[stats:nls]{stats::nls()}} or \code{\link[stats:lm]{stats::lm()}} object.
\item If \code{.method = 'brms'}: a \code{\link[brms:brmsfit-class]{brms::brmsfit()}} object.
}
}
\description{
We recommend using \code{\link[=fit_photosynthesis]{fit_photosynthesis()}} with argument \code{.photo_fun = "r_light"} rather than calling this function directly.
}
\note{
Confusingly, \eqn{R_\mathrm{d}} typically denotes respiration in the light, but you might see \eqn{R_\mathrm{day}} or \eqn{R_\mathrm{light}}.

\strong{Models}

\emph{Kok (1956)}

The \code{kok_1956} model estimates light respiration using the Kok method
(Kok, 1956). The Kok method involves looking for a breakpoint in the
light response of net CO2 assimilation at very low light intensities
and extrapolating from data above the breakpoint to estimate light
respiration as the y-intercept. Rd value should be negative,
denoting an efflux of CO2.

\emph{Yin et al. (2011)}

The \code{yin_etal_2011} model estimates light respiration according
to the Yin \emph{et al.} (2009, 2011) modifications of the Kok
method. The modification uses fluorescence data to get a
better estimate of light respiration. Rd values should be negative here to
denote an efflux of CO2.

\emph{Walker & Ort (2015)}

The \code{walker_ort_2015} model estimates light respiration and
\eqn{\Gamma*} according to Walker & Ort (2015) using a slope-
intercept regression method to find the intercept of multiple
A-C curves run at multiple light intensities. The method estimates
\eqn{\Gamma*} and \eqn{R_\mathrm{d}}. If estimated  \eqn{R_\mathrm{d}} is
positive this could indicate issues (i.e. leaks) in the gas exchange
measurements. \eqn{\Gamma*} is in units of umol / mol and \eqn{R_\mathrm{d}}
is in units of \eqn{\mu}mol m\eqn{^{-2}} s\eqn{^{-1}} of respiratory flux.
If using \eqn{C_\mathrm{i}}, the estimated value is technically \eqn{C_\mathrm{i}}*.
You need to use \eqn{C_\mathrm{c}} to get \eqn{\Gamma*} Also note, however,
that the convention in the field is to completely ignore this note.
}
\examples{
\donttest{

# Walker & Ort (2015) model

library(broom)
library(dplyr)
library(photosynthesis)

acq_data = system.file("extdata", "A_Ci_Q_data_1.csv", package = "photosynthesis") |> 
  read.csv()

fit = fit_photosynthesis(
  .data = acq_data,
  .photo_fun = "r_light",
  .model = "walker_ort_2015",
  .vars = list(.A = A, .Q = Qin, .C = Ci),
  C_upper = 300,
  # Irradiance levels used in experiment
  Q_levels =  c(1500, 750, 375, 125, 100, 75, 50, 25),
)

# The 'fit' object inherits class 'lm' and many methods can be used

## Model summary:
summary(fit)

## Estimated parameters:
coef(fit)

## 95\% confidence intervals:
## n.b. these confidence intervals are not correct because the regression is fit 
## sequentially. It ignores the underlying data and uncertainty in estimates of 
## slopes and intercepts with each A-C curve. Use '.method = "brms"' to properly 
## calculate uncertainty. 
confint(fit)

## Tidy summary table using 'broom::tidy()'
tidy(fit, conf.int = TRUE, conf.level = 0.95)

## Calculate residual sum-of-squares
sum(resid(fit)^2)

# Yin et al. (2011) model

fit = fit_photosynthesis(
  .data = acq_data,
  .photo_fun = "r_light",
  .model = "yin_etal_2011",
  .vars = list(.A = A, .phiPSII = PhiPS2, .Q = Qin),
  Q_lower = 20,
  Q_upper = 250
)

# The 'fit' object inherits class 'lm' and many methods can be used

## Model summary:
summary(fit)

## Estimated parameters:
coef(fit)

## 95\% confidence intervals:
confint(fit)

## Tidy summary table using 'broom::tidy()'
tidy(fit, conf.int = TRUE, conf.level = 0.95)

## Calculate residual sum-of-squares
sum(resid(fit)^2)

# Kok (1956) model

fit = fit_photosynthesis(
  .data = acq_data,
  .photo_fun = "r_light",
  .model = "kok_1956",
  .vars = list(.A = A, .Q = Qin),
  Q_lower = 20,
  Q_upper = 150
)

# The 'fit' object inherits class 'lm' and many methods can be used

## Model summary:
summary(fit)

## Estimated parameters:
coef(fit)

## 95\% confidence intervals:
confint(fit)

## Tidy summary table using 'broom::tidy()'
tidy(fit, conf.int = TRUE, conf.level = 0.95)

## Calculate residual sum-of-squares
sum(resid(fit)^2)

}
}
\references{
Kok B. 1956. On the inhibition of photosynthesis by intense light.
Biochimica et Biophysica Acta 21: 234–244

Walker BJ, Ort DR. 2015. Improved method for measuring the apparent
CO2 photocompensation point resolves the impact of multiple internal
conductances to CO2 to net gas exchange. Plant Cell Environ 38:2462-
2474

Yin X, Struik PC, Romero P, Harbinson J, Evers JB, van der Putten
PEL, Vos J. 2009. Using combined measurements of gas exchange and
chlorophyll fluorescence to estimate parameters of a biochemical C3
photosynthesis model: a critical appraisal and a new integrated
approach applied to leaves in a wheat (Triticum aestivum) canopy.
Plant Cell Environ 32:448-464

Yin X, Sun Z, Struik PC, Gu J. 2011. Evaluating a new method to
estimate the rate of leaf respiration in the light by analysis of
combined gas exchange and chlorophyll fluorescence measurements.
Journal of Experimental Botany 62: 3489–3499
}
