#' summary.phenology prints the information from a result object.
#' @title Print the result information from a result object.
#' @author Marc Girondot
#' @return A list with four objects: synthesis is a data.frame with global estimate of nesting.\cr
#' details_MCMC, details_ML and details_mean are lists with day by day information for each series.
#' @param object A result file generated by fit_phenology
#' @param resultmcmc A mcmc object
#' @param series Names of the series to be analyzed or "all"
#' @param chain The number of chain to be used in resultmcmc
#' @param replicate.CI.mcmc Number of iterations to be used or "all"
#' @param replicate.CI Number of replicates for ML resampling
#' @param level Level to estimate confidence interval or credibility interval
#' @param print Should information be shown
#' @param ... Not used
#' @description The function summary.phenology shows result and estimates confidence interval.
#' @examples
#' \dontrun{
#' library(phenology)
#' # Read a file with data
#' Gratiot<-read.delim("http://max2.ese.u-psud.fr/epc/conservation/BI/Complete.txt", header=FALSE)
#' data(Gratiot)
#' # Generate a formatted list nammed data_Gratiot 
#' data_Gratiot<-add_phenology(Gratiot, name="Complete", 
#' 		reference=as.Date("2001-01-01"), format="%d/%m/%Y")
#' # Generate initial points for the optimisation
#' parg<-par_init(data_Gratiot, fixed.parameters=NULL)
#' # Run the optimisation
#' result_Gratiot<-fit_phenology(data=data_Gratiot, 
#' 		fitted.parameters=parg, fixed.parameters=NULL)
#' data(result_Gratiot)
#' # Display information from the result
#' summary(result_Gratiot)
#' # Using mcmc
#' summary(result_Gratiot, resultmcmc=result_Gratiot_mcmc)
#' }
#' @method summary phenology
#' @export



summary.phenology <- function(object, 
                              resultmcmc = NULL, 
                              chain = 1, 
                              series = "all", 
                              replicate.CI.mcmc = "all", 
                              replicate.CI = 10000,
                              level= 0.95,
                              print = TRUE, 
                              ...) {

  # object=result_Gratiot; resultmcmc=NULL; level=0.95;chain=1; series="all"; replicate.CI.mcmc = "all"; replicate.CI = 10000; print=TRUE
  # # object=result_Gratiot; resultmcmc=result_Gratiot_mcmc
  
  formatpar <- getFromNamespace(".format_par", ns="phenology")
  dailycount <- getFromNamespace(".daily_count", ns="phenology")
  
  
	if (print) {
  cat(paste("Number of timeseries: ", length(object$data), "\n", sep=""))
	for (i in 1:length(object$data)) {
		cat(paste(names(object$data[i]), "\n", sep=""))
	}
	cat(paste("Date uncertainty management: ", object$method_incertitude, "\n", sep=""))
	cat(paste("Managment of zero counts: ", object$zero_counts, "\n", sep=""))
	cat("Fitted parameters:\n")
	for (i in 1:length(object$par)) {
		cat(paste(names(object$par[i]), "=", object$par[i], " SE ", object$se[i], "\n", sep=""))
	}
	if (length(object$fixed.parameters)>0) {
	cat("Fixed parameters:\n")
	for (i in 1:length(object$fixed.parameters)) {
		cat(paste(names(object$fixed.parameters[i]), "=", object$fixed.parameters[i], "\n", sep=""))
	}
	}
	cat(paste("Ln L: ", object$value, "\n", sep=""))
	cat(paste("Parameter number: ", length(object$par), "\n", sep=""))
	cat(paste("AIC: ", 2*object$value+2*length(object$par), "\n", sep=""))
	}
	
  # chain <- 1
  # Name of the series for which we want an estimate
  if (is.numeric(series)) series <- names(object$data)[series]
  if (any(series == "all")) series <- names(object$data)
  nseries <- length(series)
  rna <- rep(NA, nseries)
  
  probs <- c((1-level)/2, 0.5, 1-(1-level)/2)
  
	retdf <- data.frame(series=series, 
	                    "without_obs_Mean"=rna,
	                    "with_obs_Mean"=rna,
	                    "without_obs_Low_ML"=rna, 
	                    "without_obs_Median_ML"=rna, 
	                    "without_obs_High_ML"=rna, 
	                    "with_obs_Low_ML"=rna, 
	                    "with_obs_Median_ML"=rna, 
	                    "with_obs_High_ML"=rna, 
	                    "without_obs_Low_MCMC"=rna, 
	                    "without_obs_Median_MCMC"=rna, 
	                    "without_obs_High_MCMC"=rna, 
	                    "with_obs_Low_MCMC"=rna, 
	                    "with_obs_Median_MCMC"=rna, 
	                    "with_obs_High_MCMC"=rna, 
	                    stringsAsFactors = FALSE)
	rownames(retdf) <- series
	  klist_mcmc <- list()
	  klist_ML <- list()
	  klist_Mean <- list()
	  
	  # nmser <- series[1]
	  for (nmser in series) {
	    
	    if (print) {
	      tx <- paste0("Timeseries: ", nmser)
	      cat(paste0(rep("-", nchar(tx)), collapse=""), "\n")
	      cat(tx, "\n")
	      cat(paste0(rep("-", nchar(tx)), collapse=""), "\n")
	    }

	    dref <- object$Dates[[nmser]]["reference"]
	    nday <- ifelse(as.POSIXlt(dref+365)$mday==as.POSIXlt(dref)$mday, 365, 366)
	    
	    # Observed counts
	    observedPontes <- data.frame(ordinal=object$data[[nmser]][, "ordinal"], 
	                                 observed=object$data[[nmser]][, "nombre"])
	    # je mets des 0 à toutes les dates supplémentaires de la série
	    if (any(!is.na(object$data[[nmser]][, "ordinal2"]))) {
	      for (i in which(!is.na(object$data[[nmser]][, "ordinal2"]))) {
	        rnge <- (object$data[[nmser]][i, "ordinal"]+1):(object$data[[nmser]][i, "ordinal2"])
	        observedPontes <- rbind(observedPontes, 
	                                data.frame(ordinal= rnge, 
	                                           observed=rep(0, length(rnge))))
	      }
	    }
	    
	    
	    parg <- formatpar(c(object$par, object$fixed.parameters), nmser)
	    dc_mean <- dailycount(1:nday, parg)
	    retdf[nmser, "without_obs_Mean"] <- sum(dc_mean)
	    
	    
	    if (print) {
	      cat("Total estimate not taking into account the observations: ")
	      cat(paste0("Mean=", retdf[nmser, "without_obs_Mean"], "\n"))
	    }
	    
	    SDMin <- NULL
	    SDMax <- NULL
	    for (mu in dc_mean) {
	      qnb <- qnbinom(p = c(probs[1], probs[3]), 
	                     size=c(object$par, object$fixed.parameters)["Theta"], 
	                     mu=mu)
	      SDMin <- c(SDMin, qnb[1])
	      SDMax <- c(SDMax, qnb[2])
	    }
	    
	    dc_mean <- matrix(data = c(1:nday, dc_mean, SDMin, SDMax), nrow=4, byrow = TRUE)
	    rownames(dc_mean) <- c("Ordinal", "Mean", "SD.Low", "SD.High")
	    
	    k <- list(dc_mean)
	    names(k) <- nmser
	    
	    klist_Mean <- c(klist_Mean, k)
	    
	    dc_mean["Mean", observedPontes[, "ordinal"]] <- observedPontes[, "observed"]
	    retdf[nmser, "with_obs_Mean"] <- sum(dc_mean["Mean", ])
	    
	    if (print) {
	      cat("Total estimate taking into account the observations: ")
	      cat(paste0("Mean=", retdf[nmser, "with_obs_Mean"], "\n"))
	    }
	    
	    pfixed <- object$fixed.parameters
	    sepfixed <- pfixed[strtrim(names(pfixed), 3)=="sd#"]
	    pfixed <- pfixed[strtrim(names(pfixed), 3) != "sd#"]
	    if (!is.null(sepfixed)) names(sepfixed) <- substring(names(sepfixed), 4)
	    
	    # J'ai un sd sur des paramètres fixés
	    # if (!is.null(sepfixed) & (!identical(unname(sepfixed), numeric(0)))) {
	      
	      pfixed.df <- data.frame()
	      pfixed.df.mcmc <- data.frame()
	      
	      replicate.CI.mcmc.x <- NULL
	      if (!is.null(resultmcmc)) {
	        if (replicate.CI.mcmc == "all") {
	          replicate.CI.mcmc.x <- nrow(resultmcmc$resultMCMC[[chain]])
	        } else {
	          replicate.CI.mcmc.x <- replicate.CI.mcmc
	        }
	      }
	      
	      if (!is.null(pfixed)) {
	        for (i in seq_along(pfixed)) {
	          dfadd <- data.frame()
	          dfadd.mcmc <- data.frame()
	          
	          if (!is.na(sepfixed[names(pfixed[i])])) {
	            if (!is.null(replicate.CI)) {
	              dfadd <- data.frame(rnorm(n=replicate.CI, mean=unname(pfixed[i]), sd=sepfixed[names(pfixed[i])]))
	              colnames(dfadd) <- names(pfixed[i])
	            }
	            if (!is.null(replicate.CI.mcmc.x)) {
	              dfadd.mcmc <- data.frame(rnorm(n=replicate.CI.mcmc.x, mean=unname(pfixed[i]), sd=sepfixed[names(pfixed[i])]))
	              colnames(dfadd.mcmc) <- names(pfixed[i])
	            }
	          } else {
	            if (!is.null(replicate.CI)) {
	              dfadd <- data.frame(rep(unname(pfixed[i]), replicate.CI))
	              colnames(dfadd) <- names(pfixed[i])
	            }
	            if (!is.null(replicate.CI.mcmc.x)) {
	              dfadd.mcmc <- data.frame(rep(unname(pfixed[i]), replicate.CI.mcmc.x))
	              colnames(dfadd.mcmc) <- names(pfixed[i])
	            }
	          }
	          if (ncol(pfixed.df.mcmc) ==0 ) {
	            pfixed.df.mcmc <- dfadd.mcmc
	          } else {
	            pfixed.df.mcmc <- cbind(pfixed.df.mcmc, dfadd.mcmc)
	          }
	          if (ncol(pfixed.df) ==0 ) {
	            pfixed.df <- dfadd
	          } else {
	            pfixed.df <- cbind(pfixed.df, dfadd)
	          }
	          
	        }
	      }
	      
	      pfixed.df <- as.matrix(pfixed.df)
	      pfixed.df.mcmc <- as.matrix(pfixed.df.mcmc)
	    
	    
	    lnday <- 1:nday
	    opord <- observedPontes[, "ordinal"]
	    opnumb <- observedPontes[, "observed"]
	    
      if (!is.null(resultmcmc)) {
        lmcmc <- nrow(resultmcmc$resultMCMC[[chain]])
        mcmctobeused <- 1:lmcmc
        if (replicate.CI.mcmc != "all") {
          repl <- ifelse(nrow(resultmcmc$resultMCMC[[chain]]) <= replicate.CI.mcmc, TRUE, FALSE)
          mcmctobeused <- sample(x=mcmctobeused, 
	                             size=replicate.CI.mcmc, 
	                             replace = repl)
        } else {
          replicate.CI.mcmc <- nrow(resultmcmc$resultMCMC[[chain]])
        }
        
	      
        if (ncol(pfixed.df.mcmc) != 0) {
          dailydata <- sapply(X = 1:replicate.CI.mcmc, FUN=function(xxx) {
            px <- c(resultmcmc$resultMCMC[[chain]][mcmctobeused[xxx], ], pfixed.df.mcmc[xxx, ])
            xparec <- formatpar(px, nmser)
            dailycount(lnday, xparec, print=FALSE)
          })
        } else {
          dailydata <- sapply(X = 1:replicate.CI.mcmc, FUN=function(xxx) {
            px <- c(resultmcmc$resultMCMC[[chain]][mcmctobeused[xxx], ])
            xparec <- formatpar(px, nmser)
            dailycount(lnday, xparec, print=FALSE)
          })
        }
	    
	    synthesisPontes <- apply(X = dailydata, MARGIN = 2, FUN=sum)
	    
	    synthesisPontes_withObs <- apply(X = dailydata, MARGIN = 2, FUN=function(xxx) {
	      xxx[opord] <- opnumb
	      sum(xxx)
	    })
	    
	    k <- apply(X = dailydata, MARGIN=1, 
	                    FUN = function(x) {quantile(x, probs=probs)})
	    k <- list(rbind(Ordinal=lnday, k))

	    names(k) <- nmser

	    klist_mcmc <- c(klist_mcmc, k)
	    
	    k <- unname(quantile(synthesisPontes, probs=probs))
	    retdf[nmser, c("without_obs_Low_MCMC", "without_obs_Median_MCMC", "without_obs_High_MCMC")] <- k
	    
	    if (print) {
	      cat("Total estimate not taking into account the observations MCMC-based:\n")
	      cat(paste0("Low=", k[1], "   Median=", k[2], "   High=", k[3], "\n"))
	    }
	    
	    k <- unname(quantile(synthesisPontes_withObs, probs=probs))
	  retdf[nmser, c("with_obs_Low_MCMC", "with_obs_Median_MCMC", "with_obs_High_MCMC")] <- k
	  
	  if (print) {
	    cat("Total estimate taking into account the observations MCMC-based:\n")
	    cat(paste0("Low=", k[1], "   Median=", k[2], "   High=", k[3], "\n"))
	  }
	  
    } else {
        k <- list(NA)
        names(k) <- nmser
        klist_mcmc <- c(klist_mcmc, k)
    }
	    # fin du mcmc analysis
	    
	    
	    
	    
	    
	    
	    # Maintenant en prenant en compte la matrice hessienne
	    
	    if (!is.null(object$hessian)) {
	      
	      vcov <- solve(object$hessian)
	      # dg <- diag(vcov)
	      parg <- object$par
	      
	      par2 <- rmnorm(n = replicate.CI, mean = parg, vcov)
	      colnames(par2) <- names(parg)
	      
	      
	      if (ncol(pfixed.df) != 0) {
	        dailydata <- sapply(1:replicate.CI, FUN=function(xxx) {
	          dailycount(lnday, formatpar(c(par2[xxx, ], pfixed.df[xxx, ]), nmser), print=FALSE)
	        })
	      } else {
	        dailydata <- sapply(1:replicate.CI, FUN=function(xxx) {
	          dailycount(lnday, formatpar(c(par2[xxx, ]), nmser), print=FALSE)
	        })
	        
	      }
	      
	      k <- apply(X = dailydata, MARGIN=1, 
	                      FUN = function(x) {quantile(x, probs=probs)})
	      k <- list(rbind(Ordinal=lnday, k))

	      names(k) <- nmser
	      
	      klist_ML <- c(klist_ML, k)
	      
	      synthesisPontes <- apply(X = dailydata, MARGIN = 2, FUN=sum)
	      
	      synthesisPontes_withObs <- apply(X = dailydata, MARGIN = 2, FUN=function(xxx) {
	        xxx[opord] <- opnumb
	        sum(xxx)
	      })
	      
	      k <- unname(quantile(synthesisPontes, probs=probs))
	      retdf[nmser, c("without_obs_Low_ML", "without_obs_Median_ML", "without_obs_High_ML")] <- k
	      
	      if (print) {
	        cat("Total estimate not taking into account the observations ML-based:\n")
	        cat(paste0("Low=", k[1], "   Median=", k[2], "   High=", k[3], "\n"))
	      }
	      
	      k <- unname(quantile(synthesisPontes_withObs, probs=probs))
	      retdf[nmser, c("with_obs_Low_ML", "with_obs_Median_ML", "with_obs_High_ML")] <- k
	      
	      if (print) {
	        cat("Total estimate taking into account the observations ML-based:\n")
	        cat(paste0("Low=", k[1], "   Median=", k[2], "   High=", k[3], "\n"))
	      }
	      
	    } else {
	      k <- list(NA)
	      names(k) <- nmser
	      klist_ML <- c(klist_ML, k)
	    }
	    
	    
	    

# fin de la boucle des séries
	  }
	  
	  rout <- list(synthesis=retdf, details_mcmc=klist_mcmc, 
	               details_ML=klist_ML, details_Mean=klist_Mean)
	  class(rout) <- "phenologyout"
  return(invisible(rout))
}
