% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pm_match.R
\name{validate_peru_mammals}
\alias{validate_peru_mammals}
\title{Match Species Names Against Peru Mammals Database}
\usage{
validate_peru_mammals(splist, quiet = TRUE)
}
\arguments{
\item{splist}{A character vector containing the species names to be matched.
Names can be in any format (uppercase, lowercase, with underscores, etc.).
Duplicate names are preserved in the output.}

\item{quiet}{Logical, default TRUE. If FALSE, prints informative messages
about the matching progress.}
}
\value{
A tibble with the following columns:
\describe{
\item{sorter}{Integer. Original position in input vector}
\item{Orig.Name}{Character. Original input name (standardized)}
\item{Matched.Name}{Character. Matched name from database or "---"}
\item{Match.Level}{Character. Quality of match ("Exact rank", "No match", etc.)}
\item{matched}{Logical. Whether a match was found}
\item{Rank}{Integer. Input taxonomic rank (1 or 2)}
\item{Matched.Rank}{Integer. Matched taxonomic rank (1 or 2)}
\item{Comp.Rank}{Logical. Whether ranks match exactly}
\item{valid_rank}{Logical. Whether match is valid at correct rank}
\item{Orig.Genus}{Character. Input genus (uppercase)}
\item{Orig.Species}{Character. Input species (uppercase)}
\item{Author}{Character. Taxonomic authority if provided}
\item{Matched.Genus}{Character. Matched genus (uppercase)}
\item{Matched.Species}{Character. Matched species (uppercase)}
\item{genus_dist}{Integer. Edit distance for genus (0=exact, >0=fuzzy, NA=no match)}
\item{species_dist}{Integer. Edit distance for species (0=exact, >0=fuzzy, NA=no match or genus-only)}
\item{scientific_name}{Character. Scientific name from peru_mammals}
\item{common_name}{Character. Common name in Spanish}
\item{family}{Character. Family}
\item{order}{Character. Order}
\item{endemic}{Logical. Endemic to Peru?}
}

\strong{Attributes:}
The output includes metadata accessible via \code{attr()}:
\itemize{
\item \code{target_database}: "peru_mammals"
\item \code{matching_date}: Date of matching
\item \code{n_input}: Number of input names
\item \code{n_matched}: Number of successful matches
\item \code{match_rate}: Percentage of successful matches
\item \code{n_fuzzy_genus}: Number of fuzzy genus matches
\item \code{n_fuzzy_species}: Number of fuzzy species matches
\item \code{ambiguous_genera}: Ambiguous genus matches (if any)
\item \code{ambiguous_species}: Ambiguous species matches (if any)
}
}
\description{
Matches given species names against the official list of mammal species
of Peru (Pacheco et al. 2021). Uses a hierarchical matching strategy that
includes direct matching, genus-level matching, and fuzzy matching to
maximize successful matches while maintaining accuracy.

\strong{Peru Mammals Database:}
\itemize{
\item 575 mammal species
\item Binomial nomenclature only (no infraspecific taxa)
\item Includes 6 undescribed species ("sp." cases)
\item Fields: genus, species, scientific_name, common_name, family, order, endemic
}
}
\details{
\strong{Matching Strategy:}
The function implements a hierarchical matching pipeline:
\enumerate{
\item \strong{Node 1 - Direct Match:} Exact matching of binomial names (genus + species)
\item \strong{Node 2 - Genus Match:} Exact matching at genus level
\item \strong{Node 3 - Fuzzy Genus:} Fuzzy matching for genus with typos (max distance = 1)
\item \strong{Node 4 - Fuzzy Species:} Fuzzy matching for species within matched genus
}

\strong{Special Cases:}
\itemize{
\item Handles "sp." cases: "Akodon sp. Ancash", "Oligoryzomys sp. B", etc.
\item Case-insensitive matching
\item Removes common qualifiers (CF., AFF.)
\item Standardizes spacing and formatting
}

\strong{Rank System:}
\itemize{
\item \strong{Rank 1:} Genus level only (e.g., "Panthera")
\item \strong{Rank 2:} Binomial (genus + species, e.g., "Panthera onca")
}

\strong{Ambiguous Matches:}
When multiple candidates have identical fuzzy match scores, a warning is
issued and the first match is selected. Use \code{get_ambiguous_matches()}
to examine these cases.

\strong{Input Requirements:}

Species names must be provided as binomials (Genus species) WITHOUT:
\itemize{
\item Author information: Panthera onca Linnaeus"
\item Infraspecific taxa: "Panthera onca onca"
\item Parenthetical authors: "Panthera onca (Linnaeus, 1758)"
}

Valid formats:
\itemize{
\item Standard binomial: "Panthera onca"
\item Undescribed species: "Akodon sp. Ancash"
\item Case-insensitive: "PANTHERA ONCA" or "panthera onca"
}

Names with 3+ elements will be automatically rejected with a warning.
}
\examples{

# Basic usage
species_list <- c("Panthera onca", "Tremarctos ornatus", "Puma concolor")
results <- validate_peru_mammals(species_list)

# Check results
table(results$matched)
table(results$Match.Level)

# View matched species
results |>
  dplyr::filter(matched) |>
  dplyr::select(Orig.Name, Matched.Name, common_name, endemic)

# With typos (fuzzy matching)
typos <- c("Pumma concolor", "Tremarctos ornatu")  # Spelling errors
results_fuzzy <- validate_peru_mammals(typos, quiet = FALSE)

# Check for ambiguous matches
get_ambiguous_matches(results_fuzzy, type = "genus")

# Access metadata
attr(results, "match_rate")
attr(results, "n_fuzzy_genus")

# With special "sp." cases
sp_cases <- c("Akodon sp. Ancash", "Oligoryzomys sp. B")
results_sp <- validate_peru_mammals(sp_cases)
# Should match exactly

}
\seealso{
\code{\link{get_ambiguous_matches}} to retrieve ambiguous match details
}
