\name{SimpleRedistributions}
\alias{SimpleRedistributions}

\alias{as.block}
\alias{as.rowblock}
\alias{as.colblock}
\alias{as.rowcyclic}
\alias{as.colcyclic}
\alias{as.blockcyclic}

\title{ Distribute/Redistribute matrices across the process grid }
\description{
  Takes either an R matrix and distributes it as a distributed matrix,
  or takes a distributed matrix and redistributes it across a (possibly)
  new BLACS context, using a (possibly) new blocking dimension.
}
\usage{
  as.block(dx, square.bldim = TRUE)
  as.rowblock(dx)
  as.colblock(dx)
  as.rowcyclic(dx, bldim = .BLDIM)
  as.colcyclic(dx, bldim = .BLDIM)
  as.blockcyclic(dx, bldim = .BLDIM)
}
\arguments{
  \item{dx}{numeric distributed matrix}
  \item{square.bldim}{logical.  Determines whether or not the blocking
  factor for the resulting redistributed matrix will be square or not.}
  \item{bldim}{the blocking dimension for block-cyclically distributing 
  the matrix across the process grid. }
}
\details{
  These functions are simple wrappers of the very general \code{redistribute()}
  funciton (see \link{Distribute}).  Different distributed matrix distributions
  of note can be classified into three categories:  block, cyclic, and
  block-cyclic.
  
  \code{as.block()} will convert \code{ddmatrix} into one which is merely
  "block" distributed, i.e., the blocking factor is chosen in such a way
  that there will be no cycling.  By default, this new blocking factor
  will be square.  This can result in some raggedness (some processors
  owning less than others --- or nothing) if the matrix is far from square
  itself.  However, the methods of factoring \code{ddmatrix} objects, and
  therefore anything that relies on (distributed) matrix factorizations
  such as computing an inverse, least squares solution, etc., require that
  blocking factors be square.  The matrix will not change BLACS contexts.
  
  \code{as.rowblock()} will convert a distributed matrix into one which
  is distributed by row into a block distributed matrix.  That is, the 
  rows are stored contiguously, and different processors will own different
  rows, but with no cycling.  In other words, it block redistributes the data 
  across context 2.
  
  \code{as.colblock()} is the column-wise analogue of \code{as.rowblock()}.
  In other words, it block redistributes the data across context 1.
  
  \code{as.rowcyclic()} is a slightly more general version of \code{as.rowblock()},
  in that the data will be distributed row-wise, but with the possibility
  of cycling, as determined by the blocking factor.  In other words it
  block-cyclically redistributes the data across context 2.
  
  \code{as.colcyclic()} is a the column-wise analogue of \code{as.rowcyclic()}.
  In other words, it block-cyclically redistributes the data across context 1.
  
  \code{as.blockcyclic()} moves the distributed matrix into a general block-cyclic
  distribution across a 2-dimensional process grid.  In other words, it
  block-cyclically redistributes the data across context 0.
}
\value{
  Returns a distributed matrix.
}
\seealso{
  \code{\link{as.ddmatrix}, \link{Distribute}, \link{BLACS}}
}
\examples{
\dontrun{
# Save code in a file "demo.r" and run with 2 processors by
# > mpiexec -np 2 Rscript demo.r

library(pbdDMAT, quiet = TRUE)
init.grid()

dx <- ddmatrix(1:30, nrow=10)

x <- as.block(dx)
x

x <- as.rowblock(dx)
x

x <- as.colblock(dx)
x

x <- as.rowcyclic(dx)
x

x <- as.colcyclic(dx)
x

x <- as.blockcyclic(dx)
x

finalize()
}
}
\keyword{BLACS}
\keyword{Distributing Data}
