#' @title Print summary information from Outlier Tree model
#' @description Displays general statistics from a fitted Outlier Tree model (same as `summary`).
#' For printing the outliers discovered, use function `print` on the returned outliers
#' (e.g. from `predict`), not on the model object iself.
#' @param x An Outlier Tree model as produced by function `outlier.tree`.
#' @param ... Not used.
#' @details Note that after loading a serialized object from `outlier.tree` through `readRDS` or `load`,
#' it will only de-serialize the underlying C++ object upon running `predict` or `print`, so the first run will
#' be slower, while subsequent runs will be faster as the C++ object will already be in-memory.
#' @export
print.outliertree <- function(x, ...) {
    check.is.model.obj(x)
    if (check_null_ptr_model(x$obj_from_cpp$ptr_model)) {
        ptr_new <- deserialize_OutlierTree(x$obj_from_cpp$serialized_obj)
        eval.parent(substitute(x$obj_from_cpp$ptr_model <- ptr_new))
        x$obj_from_cpp$ptr_model <- ptr_new
    }
    cat("Outlier Tree model\n")
    if (NROW(x$cols_num))  cat(sprintf("\tNumeric variables: %d\n",     NROW(x$cols_num)))
    if (NROW(x$cols_date)) cat(sprintf("\tDate variables: %d\n",        NROW(x$cols_date)))
    if (NROW(x$cols_ts))   cat(sprintf("\tTimestamp variables: %d\n",   NROW(x$cols_ts)))
    if (NROW(x$cols_cat))  cat(sprintf("\tCategorical variables: %d\n", NROW(x$cols_cat)))
    if (NROW(x$cols_bool)) cat(sprintf("\tBoolean variables: %d\n",     NROW(x$cols_bool)))
    if (NROW(x$cols_ord))  cat(sprintf("\tOrdinal variables: %d\n",     NROW(x$cols_ord)))
    if (!is.null(x$outliers_data)) cat(sprintf("Contains saved outlier info for %d rows\n", NROW(x$outliers_data)))
    cat("\n")
    cat(sprintf("Consists of %d clusters, spread across %d tree branches\n",
                x$obj_from_cpp$nclust, x$obj_from_cpp$ntrees))
}

#' @title Print summary information from Outlier Tree model
#' @description Displays general statistics from a fitted Outlier Tree model (same as `print`).
#' For printing the outliers discovered, use function `print` on the returned outliers
#' (e.g. from `predict`), not on the model object iself.
#' @param object An Outlier Tree model as produced by function `outlier.tree`.
#' @param ... Not used.
#' @seealso \link{print.outliertree}
#' @export
summary.outliertree <- function(object, ...) {
    print.outliertree(object)
}

#' @title Print outliers in human-readable format
#' @description Pretty-prints outliers as output by the `predict` function from an Outlier Tree
#' model (as generated by function `outlier.tree`), or by `extract.training.outliers`.
#' Same as function `print` (see the documentation of `print` for more information about
#' the parameters).
#' @param object Outliers as returned by predict method on an object from `outlier.tree`.
#' @param outliers_print Maximum number of outliers to print.
#' @param ... Not used.
#' @seealso \link{print.outlieroutputs}
#' @export 
summary.outlieroutputs <- function(object, outliers_print = 15, ...) {
    cat(sprintf("Outlier outputs from %d rows [%d of which are outliers]\n\n\n",
                NROW(object), sum(!is.na(sapply(object, function(x) x$tree_depth)))))
    print.outlieroutputs(object, outliers_print)
}
