#' add_osm_suface
#'
#' Adds a colour-coded surface of spatial objects (polygons, lines, or points
#' generated by extract_osm_objects ()) to a graphics object initialised with
#' plot_osm_basemap(). The surface is spatially interpolated between the values
#' given in 'dat', which has to be a matrix or data.frame of 3 colums (x, y, z),
#' where (x,y) are (longitude, latitude), and z are the values to be
#' interpolated. Interpolation uses spatstat::Smoothing.ppp, which applies a
#' Gaussian kernel smoother optimised to the given data, and is effectively
#' non-parametric.
#'
#' @param obj an sp SPDF or SLDF (list of polygons or lines) returned by
#' extract_osm_objects ()
#' @param dat A matrix or data frame of 3 columns (x, y, z), where (x, y) are
#' (longitude, latitude), and z are the values to be interpolated
#' @param method Either 'idw' (Inverse Distance Weighting as spatstat::idw;
#' default), otherwise uses 'Gaussian' for kernel smoothing (as
#' spatstat::Smooth.ppp)
#' @param bg If specified, OSM objects outside the convex hull surrounding 'dat'
#' are plotted in this colour, otherwise they are included in the interpolation
#' (which will generally be inaccurate for peripheral values)
#' @param cols Vector of colours for shading z-values (for example,
#' 'terrain.colors (30)')
#' @param border Plot border? (For SpatialPolygons only)
#' @param verbose If TRUE, provides notification of progress
#' @param ... other parameters to be passed to polygons, lines (such as lwd,
#' lty), or points (such as pch, cex)
#' @return Range of interpolated values (which may differ from range of 'dat$z'
#' as submitted, and can be used to scale 'add_colourbar()')
#'
#' @note 
#' Spatial smoothing is *interpolative* and so values beyond the bounding
#' polygon of 'dat' will generally be unreliable. Points beyond the bounding
#' polygon are only included in the interpolation if 'bg' is NA or NULL.
#'
#' @export
#'
#' @examples
#' plot_osm_basemap (bbox=get_bbox (c (-0.15, 51.5, -0.1, 51.52)), col="gray20")
#' add_osm_objects (london$dat_BNR, col="gray40") # non-residential buildings
#' bbox <- get_bbox (c (-0.15, 51.5, -0.1, 51.52))
#' x <- seq (bbox [1,1], bbox [1,2], length.out=dim (volcano)[1])
#' y <- seq (bbox [2,1], bbox [2,2], length.out=dim (volcano)[2])
#' xy <- cbind (rep (x, dim (volcano) [2]), rep (y, each=dim (volcano) [1]))
#' z <- as.numeric (volcano)
#' plot_osm_basemap (bbox=bbox, bg="gray20")
#' # uncomment to run: can be time-consuming
#' # zl <- add_osm_surface (london$dat_BNR, dat=cbind (xy, z), method="idw")
#' cols <- adjust_colours (terrain.colors (30), -0.2) # Darken by ~20%
#' # zl <- add_osm_surface (london$dat_H, dat=cbind (xy, z), cols=cols)
#' # zl <- add_osm_surface (london$dat_HP, dat=cbind (xy, z), cols=cols, lwd=2)
#' # add_colourbar (cols=terrain.colors (30), side=4, zlims=zl)

add_osm_surface <- function (obj=obj, dat=NULL, method="idw", bg=NULL,
                             cols=terrain.colors (30), border=FALSE, 
                             verbose=FALSE, ...)
{
    if (is.null (dev.list ()))
        stop ('add_osm_surface can only be called after plot_osm_basemap')
    if (is.null (dat))
        stop ('data must be supplied to add_osm_surface ()')
    if (!dim (dat)[2] >= 3)
        stop ('data must have at least 3 columns')

    # Spatial interpolation.
    usr <- par ("usr") # used below
    din <- floor (par ("din") * 72)
    x <- dat [,1]
    y <- dat [,2]
    marks <- dat [,3]
    xy <- spatstat::ppp (x, y, xrange=range (x), yrange=range(y), marks=marks)
    if (verbose) cat ("1/3: Interpolating surface ... ")
    if (method == 'idw')
        z <- spatstat::idw (xy, at="pixels", dimyx=din)$v
    else
        z <- spatstat::Smooth (xy, at="pixels", dimyx=din, diggle=TRUE)$v
    if (verbose) cat ("\t\tdone\n")
    # Then set all z-values beyond the convex hull of xy to NA
    if (!is.null (bg))
        if (is.na (bg))
            bg <- NULL
    if (!is.null (bg))
    {
        if (verbose) cat ("2/3: identifying background objects ... ")
        xyh <- spatstat::ppp (x, y, xrange=range (x), yrange=range (y))
        ch <- spatstat::convexhull (xyh)
        bdry <- cbind (ch$bdry[[1]]$x, ch$bdry[[1]]$y)
        xall <- rep (seq (usr [1], usr [2], length.out=din [1]), din [2])
        yall <- rep (seq (usr [3], usr [4], length.out=din [2]), each=din [1])
        indx <- apply (cbind (xall, yall), 1, function (x)
                       sp::point.in.polygon (x [1], x [2], 
                                             bdry [,1], bdry [,2]))
        x1 <- rep (seq (din [1]), din [2])
        y1 <- rep (seq (din [2]), each=din [1])
        z [which (indx == 0)] <- NA
        if (verbose) cat ("done\n")
    }


    getColour <- function (xyi, z, bg, cols)
    {
        xi <- din [1] * (mean (xyi [,1]) - usr [1]) / (usr [2] - usr [1])
        # mean coordinates can extend beyond boundaries
        xi <- min (din [1], max (1, xi))
        yi <- din [2] * (mean (xyi [,2]) - usr [3]) / (usr [4] - usr [3])
        yi <- min (din [2], max (1, yi))
        zi <- z [ceiling (xi), ceiling (yi)]
        if (!is.na (zi))
        {
            ci <- length (cols) * (zi - min (z, na.rm=TRUE)) / 
                        diff (range (z, na.rm=TRUE))
            ci <- cols [ceiling (ci)]
        } else
            ci <- bg
        return (ci)
    }

    if (verbose) cat ("3/3: Drawing objects on map ... ")
    if (class (obj) == 'SpatialPolygonsDataFrame')
    {
        plotfunPtsColour <- function (i, dx=dx, dy=dy, z=z, border=border, 
                                      cols=cols, ...) 
        {
            xyi <- slot (slot (i, 'Polygons') [[1]], 'coords')
            if (diff (range (xyi [,1])) > dx | diff (range (xyi [,2])) > dy)
            {
                ci <- getColour (xyi, z=z, bg=bg, cols=cols)
                if (border)
                    polypath (xyi, col=ci, border=ci, ...)
                else
                    polypath (xyi, col=ci, border=NA, ...)
            }
        }
        # Find out which objects are < 1 pixel in size. NOTE this presumes
        # standard device resolution of 72dpi. 
        # TODO#1: Find out how to read dpi from open devices and modify
        dx <- diff (usr [1:2]) / din [1]
        dy <- diff (usr [3:4]) / din [2]
        # NOTE dy=dx only if figures are sized automatically
        junk <- lapply (slot (obj, 'polygons'), function (i)
                    plotfunPtsColour (i, dx=dx, dy=dy, z=z, border=border, 
                                      cols=cols, ...))
    } else if (class (obj) == 'SpatialLinesDataFrame')
    {
        plotfunLines <- function (i, z=z, bg=bg, cols=cols, ...) 
        {
            xyi <- slot (slot (i, 'Lines') [[1]], 'coords')
            lines (xyi, col=getColour (xyi, z=z, bg=bg, cols=cols), ...)
        }
        junk <- lapply (slot (obj, 'lines'), function (i)
                        plotfunLines (i, z=z, bg=bg, cols=cols, ...))
    } else if (class (obj) == 'SpatialPointsDataFrame')
    {
        xyi <- slot (obj, 'coords')
        points (xyi[,1], xyi[,2], col=getColour (xyi, z=z, bg=bg, cols=cols), ...)
    }
    if (verbose) cat ("\tdone\n")

    return (range (z, na.rm=TRUE))
}

