\name{fullmatch}
\alias{fullmatch}
\title{ Optimal full matching }
\description{
  Given two groups, such as a treatment and a control group, and
  a treatment-by-control discrepancy matrix indicating
  desirability and permissibility of potential matches,
  create optimal full matches of members of the groups.
  Optionally, incorporate restrictions on matched sets' ratios of
  treatment to control units.%, and/or omit some fraction of the control group.
}
\usage{
fullmatch(distance, min.controls = 0, max.controls = Inf, 
omit.fraction = NULL, tol = 0.001, subclass.indices = NULL)
}
\arguments{
  \item{distance}{ A matrix of nonnegative discrepancies, 
each indicating the permissibility and desirability of matching the unit 
corresponding to its row (a 'treatment') to the unit
corresponding to its column (a 'control'); or, better, a list of such matrices,
as produced by \code{\link{mdist}}.}
  \item{min.controls}{ The minimum ratio of controls to
  treatments that is to be permitted within a matched set: should
  be nonnegative and finite.  If
  \code{min.controls} is not a whole number, the reciprocal of a whole
  number, or zero, then it is rounded \emph{down} to the nearest
  whole number or reciprocal of a whole number.

  When matching within subclasses, \code{min.controls} may be a named
  numeric vector
  separately specifying the minimum permissible ratio of controls to
  treatments for each subclass.  The names of this vector should include
  names of all matrices in the list \code{distance}.
%  The names of this vector should include all nonempty levels of
%  \code{subclass.indices}, if \code{subclass.indices} is a factor,
%  or all nonempty levels of \code{interaction{subclass.indices}},
%  if \code{subclass.indices} is a data frame.
}

\item{max.controls}{ The maximum ratio of controls to treatments that is
  to be permitted within a matched set: should be positive and numeric.
  If \code{max.controls} is not a whole number, the reciprocal of a
  whole number, or \code{Inf}, then it is rounded \emph{up} to the
  nearest whole number or reciprocal of a whole number.

  When matching within subclasses, \code{max.controls} may be a named
  numeric vector separately specifying the maximum permissible ratio
  of controls to treatments in each subclass. }

\item{omit.fraction}{Optionally, specify what fraction of
  controls or treated subjects are to be rejected.  If
  \code{omit.fraction} is a positive fraction less than one, then
  \code{fullmatch} leaves up to that fraction of the control reservoir
  unmatched.  If \code{omit.fraction} is a negative number greater than
  -1, then \code{fullmatch} leaves up to |\code{omit.fraction}| of the
  treated group unmatched.  Positive values are only accepted if
  \code{max.controls} >= 1; negative values, only if \code{min.controls}
  <= 1.  If \code{omit.fraction} is not specified, then only those
  treated and control subjects without permissible matches among the
  control and treated subjects, respectively, are omitted.
  
  When matching within subclasses (so that \code{distance} is a list of matrices, as produced by
  \code{makedist} or \code{mahal.dist} or \code{pscore.dist}), \code{omit.fraction} specifies the fraction of
  controls to be rejected in each subproblem, a parameter that can be
  made to differ by subclass by setting \code{omit.fraction} equal to a
  named numeric vector of fractions.  }

\item{tol}{ Because of internal rounding, \code{fullmatch} may
  solve a slightly different matching problem than the one
  specified, in which the match generated by
  \code{fullmatch} may not coincide with an optimal solution of
  the specified problem.  \code{tol} times the number of subjects
  to be matched specifies the extent to
  which \code{fullmatch}'s output is permitted to differ from an
  optimal solution to the original problem, as measured by the
  sum of discrepancies for all treatments and controls placed
  into the same matched sets.
 }

  \item{subclass.indices}{ An old argument included for
    back-compatibility; no longer needed.}
}

\details{
  Finite entries in matrix slots of \code{distance} indicate
  permissible matches, with smaller
  discrepancies indicating more desirable matches. 
  Matrix \code{distance} must have row and column names.
  
  Consider using \code{\link{mdist}} to generate the distances.
  \code{fullmatch} tries to guess the
  order in which units would have been given in a data frame, and to
  order the factor that it returns accordingly.  If the dimnames of
  \code{distance}, or the matrices it lists, are not simply row numbers
  of the data frame you're working with, then you should compare the
  names of fullmatch's output to your row names in order to be sure
  things are in the proper order.  You can relieve yourself of these
  worries by using \code{\link{mdist}} (or \code{\link{makedist}}, \code{\link{pscore.dist}},
  or \code{\link{mahal.dist}}, which [as of version 0.6] are dispatched
  as needed by \code{\link{mdist}}) to produce the distances, as
  it passes the ordering of units to \code{fullmatch}, which then uses
  it to order its outputs.
  
  The value of \code{tol} can have a substantial effect on
  computation time; with smaller values, computation takes longer.
  Not every tolerance can be met, and how small a tolerance is too small
  varies with the machine and with the details of the problem.  If
  \code{fullmatch} can't guarantee that the tolerance is as small as the
  given value of argument \code{tol}, then matching proceeds but a
  warning is issued.  }

  \value{ Primarily, a named vector of class \code{c('optmatch',
  'factor')}.  Elements of this vector correspond to members of the
  treatment and control groups in reference to which the matching
  problem was posed, and are named accordingly; the names are taken from
  the row and column names of \code{distance}.  Each element of
  the vector is either \code{NA}, indicating unavailability of any
  suitable matches for that element, or the concatenation of: (i) a character abbreviation of
  the name of the subclass, if matching within subclasses, or the string
  '\code{m}' if not; (ii) the string \code{.}; and (iii) a
  nonnegative integer or the string \code{NA}.  In this last place,
  positive whole numbers indicate placement of the unit into a matched
  set and \code{NA}
  indicates that all or part of the matching problem given to
  \code{fullmatch} was found to be infeasible.  The functions
  \code{\link{matched}}, \code{\link{unmatched}}, and
  \code{\link{matchfailed}} distinguish these scenarios.

%  In some cases, only proper subsets of the initial treatment and/or
%  control groups will be represented in the value of \code{fullmatch}.
%  Whether this occurs is determined by the status of argument
%  \code{subclass.indices}.  If \code{subclass.indices} is null, then all
%  elements of the treatment and control groups, \emph{i.e.} the rows and
%  columns of \code{distance}, are represented in the value of
%  \code{fullmatch}.  Otherwise, the vector has an element for each unit
%  represented in \code{subclass.indices}, \emph{i.e.} for each element
%  of factor \code{subclass.indices} or for each row of data frame
%  \code{subclass.indices}.

  Secondarily, \code{fullmatch} returns various data about the matching
  process and its result, stored as attributes of the named vector
  which is its primary output.  In particular, the \code{exceedances}
  attribute gives upper bounds, not necessarily sharp, for the amount by
  which the sum of distances between matched units in the result of
  \code{fullmatch} exceeds the least possible sum of distances between
  matched units in a feasible solution to the matching problem given to
  \code{fullmatch}.  (Such a bound is also printed by
  \code{print.optmatch} and \code{summary.optmatch}.)  
}
\references{
  Hansen, B.B. and Klopfer, S.O. (2006), \sQuote{ Optimal full matching and related designs via network flows}, \emph{Journal of
    Computational and Graphical Statistics}, \bold{15}, 609--627.
  
  Hansen, B.B. (2004), \sQuote{Full Matching in an Observational Study
  of Coaching for the SAT}, \emph{Journal of the American
  Statistical Association}, \bold{99}, 609--618.  

  Rosenbaum, P. (1991), 
\sQuote{A Characterization of Optimal Designs for Observational
  Studies}, \emph{Journal of the Royal Statistical Society, Series B},
  \bold{53}, 597--610.
}
\author{Ben Hansen}

\seealso{\code{\link{matched}}, \code{\link{mdist}},
  \code{\link{caliper}}}
\examples{
data(nuclearplants)
### Full matching on a Mahalanobis distance
mhd  <- mdist(pr ~ t1 + t2, data = nuclearplants)
( fm1 <- fullmatch(mhd) )
summary(fm1)
### Full matching with restrictions
( fm2 <- fullmatch(mhd, min=.5, max=4) )
summary(fm2)
### Full matching to half of available controls
( fm3 <- fullmatch(mhd, omit.fraction=.5) )
summary(fm3)
### Full matching within a propensity score caliper.
ppty <- glm(pr~.-(pr+cost), family=binomial(), data=nuclearplants)
### Note that units without counterparts within the
### caliper are automatically dropped.
( fm4 <- fullmatch(mhd+caliper(1, ppty)) )
summary(fm4)

### Propensity balance assessment. Requires RItools package.
\dontrun{library(RItools) ; summary(fm4,ppty)}

### Creating a data frame with the matched sets attached.
### mdist(), caliper() and the like cooperate with fullmatch()
### to make sure observations are in the proper order:
all.equal(names(fm4), row.names(nuclearplants))
### So our data frame including the matched sets is just
cbind(nuclearplants, matches=fm4)

### In contrast, if your matching distance is an ordinary matrix
### (as earlier versions of optmatch required), you'll
### have to align it by observation name with your data set. 
cbind(nuclearplants, matches = fm4[row.names(nuclearplants)])

}

\keyword{ nonparametric }% at least one, from doc/KEYWORDS
\keyword{ optimize }% __ONLY ONE__ keyword per line

