/*
 OneMap: software for genetic mapping in outcrossing species
 Copyright (C) 2007-9 Gabriel R A Margarido and Marcelo Mollinari

    This file is part of OneMap.

    OneMap is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    OneMap is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
*/

/*
  File: threepts.c
  Description: Set of OneMap functions to be used with software R
               Implements the methodology of Wu et al. (2002): "Simultaneous
               maximum likelihood estimation of linkage and linkage phases in
               outcrossing species"
               Theoretical Population Biology 61, 349-363

  Written by Gabriel Rodrigues Alves Margarido
  Escola Superior de Agricultura "Luiz de Queiroz"

  Departamento de Gentica - So Paulo, Brazil
  Contact: gramarga@gmail.com
  First version: 02/13/2007
  Last update: 03/02/2009
*/

/* This code was meant to be easily understood and modified, thus the matrix
     definitions were made in a very explicit way, consuming a lot of space
     in this file
   The meaning of variables and the steps of the algorithm will not be
     commented - for details check the paper cited above */

#include <math.h>
#include <stdlib.h>
#include <stdio.h>
#include <R.h>
#include <Rmath.h>
#include <R_ext/PrtUtil.h>
#include <R_ext/Applic.h>
#include "twopts.h"

#define MAXITER 1000
#define THRESH log(1E-6)


/* mdrct3pt is a function used to calculate the ELEMENTWISE product of
     two matrices with dimensions 16x4 */

void mdrct3pt(double A[64], double B[64], double res[64]) {
  int i;
  for(i=0;i<64;i++) res[i] = A[i]*B[i];
}

/* mkron is a function used to calculate the KRONECKER product of
     two matrices and/or vectors */

void mkron(double *A, int rowA, int colA, double *B, int rowB, int colB, double *res) {
  int i, j, na, nb;
  na = rowA*colA;
  nb = rowB*colB;
  for (i=0;i<na;i++)
    for (j=0;j<nb;j++)
      res[((j%rowB)+(i%rowA)*rowB)+((j/rowB)+(i/rowA)*colB)*(rowA*rowB)] = A[i]*B[j];
}

/* THE 2 FUNCTIONS ABOVE WERE TESTED ONLY FOR THIS METHODOLOGY
   NOT TO BE USED WITH LARGE MATRICES AND/OR VECTORS */


/* H11 to H41 are functions that calculate the transition probability
     matrix between three markers under assignments A11, A12, A13, A14,
     A21, A22, A23, A24, A31, A32, A33, A34, A41, A42, A43 and A44, in
     that order, according to Wu et al. (2002) */

void H11(double g00, double g01, double g10, double g11, double H[64]) {
  H[0]=g00*g00; H[16]=g00*g01; H[32]=g00*g01; H[48]=g01*g01;
  H[1]=g00*g11; H[17]=g00*g10; H[33]=g01*g11; H[49]=g01*g10;
  H[2]=g00*g11; H[18]=g01*g11; H[34]=g00*g10; H[50]=g01*g10;
  H[3]=g11*g11; H[19]=g10*g11; H[35]=g10*g11; H[51]=g10*g10;
  H[4]=g00*g10; H[20]=g00*g11; H[36]=g01*g10; H[52]=g01*g11;
  H[5]=g00*g01; H[21]=g00*g00; H[37]=g01*g01; H[53]=g00*g01;
  H[6]=g10*g11; H[22]=g11*g11; H[38]=g10*g10; H[54]=g10*g11;
  H[7]=g01*g11; H[23]=g00*g11; H[39]=g01*g10; H[55]=g00*g10;
  H[8]=g00*g10; H[24]=g01*g10; H[40]=g00*g11; H[56]=g01*g11;
  H[9]=g10*g11; H[25]=g10*g10; H[41]=g11*g11; H[57]=g10*g11;
  H[10]=g00*g01; H[26]=g01*g01; H[42]=g00*g00; H[58]=g00*g01;
  H[11]=g01*g11; H[27]=g01*g10; H[43]=g00*g11; H[59]=g00*g10;
  H[12]=g10*g10; H[28]=g10*g11; H[44]=g10*g11; H[60]=g11*g11;
  H[13]=g01*g10; H[29]=g00*g10; H[45]=g01*g11; H[61]=g00*g11;
  H[14]=g01*g10; H[30]=g01*g11; H[46]=g00*g10; H[62]=g00*g11;
  H[15]=g01*g01; H[31]=g00*g01; H[47]=g00*g01; H[63]=g00*g00;
}


void H12(double g00, double g01, double g10, double g11, double H[64]) {
  H[0]=g00*g01; H[16]=g00*g00; H[32]=g01*g01; H[48]=g01*g00;
  H[1]=g00*g10; H[17]=g00*g11; H[33]=g01*g10; H[49]=g01*g11;
  H[2]=g11*g01; H[18]=g11*g00; H[34]=g10*g01; H[50]=g10*g00;
  H[3]=g11*g10; H[19]=g11*g11; H[35]=g10*g10; H[51]=g10*g11;
  H[4]=g00*g11; H[20]=g00*g10; H[36]=g01*g11; H[52]=g01*g10;
  H[5]=g00*g00; H[21]=g00*g01; H[37]=g01*g00; H[53]=g01*g01;
  H[6]=g11*g11; H[22]=g11*g10; H[38]=g10*g11; H[54]=g10*g10;
  H[7]=g11*g00; H[23]=g11*g01; H[39]=g10*g00; H[55]=g10*g01;
  H[8]=g10*g01; H[24]=g10*g00; H[40]=g11*g01; H[56]=g11*g00;
  H[9]=g10*g10; H[25]=g10*g11; H[41]=g11*g10; H[57]=g11*g11;
  H[10]=g01*g01; H[26]=g01*g00; H[42]=g00*g01; H[58]=g00*g00;
  H[11]=g01*g10; H[27]=g01*g11; H[43]=g00*g10; H[59]=g00*g11;
  H[12]=g10*g11; H[28]=g10*g10; H[44]=g11*g11; H[60]=g11*g10;
  H[13]=g10*g00; H[29]=g10*g01; H[45]=g11*g00; H[61]=g11*g01;
  H[14]=g01*g11; H[30]=g01*g10; H[46]=g00*g11; H[62]=g00*g10;
  H[15]=g01*g00; H[31]=g01*g01; H[47]=g00*g00; H[63]=g00*g01;
}


void H13(double g00, double g01, double g10, double g11, double H[64]) {
  H[0]=g01*g00; H[16]=g01*g01; H[32]=g00*g00; H[48]=g00*g01;
  H[1]=g01*g11; H[17]=g01*g10; H[33]=g00*g11; H[49]=g00*g10;
  H[2]=g10*g00; H[18]=g10*g01; H[34]=g11*g00; H[50]=g11*g01;
  H[3]=g10*g11; H[19]=g10*g10; H[35]=g11*g11; H[51]=g11*g10;
  H[4]=g01*g10; H[20]=g01*g11; H[36]=g00*g10; H[52]=g00*g11;
  H[5]=g01*g01; H[21]=g01*g00; H[37]=g00*g01; H[53]=g00*g00;
  H[6]=g10*g10; H[22]=g10*g11; H[38]=g11*g10; H[54]=g11*g11;
  H[7]=g10*g01; H[23]=g10*g00; H[39]=g11*g01; H[55]=g11*g00;
  H[8]=g11*g00; H[24]=g11*g01; H[40]=g10*g00; H[56]=g10*g01;
  H[9]=g11*g11; H[25]=g11*g10; H[41]=g10*g11; H[57]=g10*g10;
  H[10]=g00*g00; H[26]=g00*g01; H[42]=g01*g00; H[58]=g01*g01;
  H[11]=g00*g11; H[27]=g00*g10; H[43]=g01*g11; H[59]=g01*g10;
  H[12]=g11*g10; H[28]=g11*g11; H[44]=g10*g10; H[60]=g10*g11;
  H[13]=g11*g01; H[29]=g11*g00; H[45]=g10*g01; H[61]=g10*g00;
  H[14]=g00*g10; H[30]=g00*g11; H[46]=g01*g10; H[62]=g01*g11;
  H[15]=g00*g01; H[31]=g00*g00; H[47]=g01*g01; H[63]=g01*g00;
}


void H14(double g00, double g01, double g10, double g11, double H[64]) {
  H[0]=g01*g01; H[16]=g01*g00; H[32]=g00*g01; H[48]=g00*g00;
  H[1]=g01*g10; H[17]=g01*g11; H[33]=g00*g10; H[49]=g00*g11;
  H[2]=g10*g01; H[18]=g10*g00; H[34]=g11*g01; H[50]=g11*g00;
  H[3]=g10*g10; H[19]=g10*g11; H[35]=g11*g10; H[51]=g11*g11;
  H[4]=g01*g11; H[20]=g01*g10; H[36]=g00*g11; H[52]=g00*g10;
  H[5]=g01*g00; H[21]=g01*g01; H[37]=g00*g00; H[53]=g00*g01;
  H[6]=g10*g11; H[22]=g10*g10; H[38]=g11*g11; H[54]=g11*g10;
  H[7]=g10*g00; H[23]=g10*g01; H[39]=g11*g00; H[55]=g11*g01;
  H[8]=g11*g01; H[24]=g11*g00; H[40]=g10*g01; H[56]=g10*g00;
  H[9]=g11*g10; H[25]=g11*g11; H[41]=g10*g10; H[57]=g10*g11;
  H[10]=g00*g01; H[26]=g00*g00; H[42]=g01*g01; H[58]=g01*g00;
  H[11]=g00*g10; H[27]=g00*g11; H[43]=g01*g10; H[59]=g01*g11;
  H[12]=g11*g11; H[28]=g11*g10; H[44]=g10*g11; H[60]=g10*g10;
  H[13]=g11*g00; H[29]=g11*g01; H[45]=g10*g00; H[61]=g10*g01;
  H[14]=g00*g11; H[30]=g00*g10; H[46]=g11*g01; H[62]=g10*g01;
  H[15]=g00*g00; H[31]=g00*g01; H[47]=g01*g00; H[63]=g01*g01;
}


void H21(double g00, double g01, double g10, double g11, double H[64]) {
  H[0]=g00*g10; H[16]=g00*g11; H[32]=g01*g10; H[48]=g01*g11;
  H[1]=g00*g01; H[17]=g00*g00; H[33]=g01*g01; H[49]=g00*g01;
  H[2]=g10*g11; H[18]=g11*g11; H[34]=g10*g10; H[50]=g10*g11;
  H[3]=g01*g11; H[19]=g00*g11; H[35]=g01*g10; H[51]=g00*g10;
  H[4]=g00*g00; H[20]=g00*g01; H[36]=g00*g01; H[52]=g01*g01;
  H[5]=g00*g11; H[21]=g00*g10; H[37]=g01*g11; H[53]=g01*g10;
  H[6]=g00*g11; H[22]=g01*g11; H[38]=g00*g10; H[54]=g01*g10;
  H[7]=g11*g11; H[23]=g10*g11; H[39]=g10*g11; H[55]=g10*g10;
  H[8]=g10*g10; H[24]=g10*g11; H[40]=g10*g11; H[56]=g11*g11;
  H[9]=g01*g10; H[25]=g00*g10; H[41]=g01*g11; H[57]=g00*g11;
  H[10]=g01*g10; H[26]=g01*g11; H[42]=g00*g10; H[58]=g00*g11;
  H[11]=g01*g01; H[27]=g00*g01; H[43]=g00*g01; H[59]=g00*g00;
  H[12]=g00*g10; H[28]=g01*g10; H[44]=g00*g11; H[60]=g01*g11;
  H[13]=g10*g11; H[29]=g10*g10; H[45]=g11*g11; H[61]=g10*g11;
  H[14]=g00*g01; H[30]=g01*g01; H[46]=g00*g00; H[62]=g00*g01;
  H[15]=g01*g11; H[31]=g01*g10; H[47]=g00*g11; H[63]=g00*g10;
}


void H22(double g00, double g01, double g10, double g11, double H[64]) {
  H[0]=g00*g11; H[16]=g00*g10; H[32]=g01*g11; H[48]=g01*g10;
  H[1]=g00*g00; H[17]=g00*g01; H[33]=g00*g01; H[49]=g01*g01;
  H[2]=g11*g11; H[18]=g10*g11; H[34]=g10*g11; H[50]=g10*g10;
  H[3]=g00*g11; H[19]=g01*g11; H[35]=g00*g10; H[51]=g01*g10;
  H[4]=g00*g01; H[20]=g00*g00; H[36]=g01*g01; H[52]=g00*g01;
  H[5]=g00*g10; H[21]=g00*g11; H[37]=g01*g10; H[53]=g01*g11;
  H[6]=g01*g11; H[22]=g00*g11; H[38]=g01*g10; H[54]=g00*g10;
  H[7]=g10*g11; H[23]=g11*g11; H[39]=g10*g10; H[55]=g10*g11;
  H[8]=g10*g11; H[24]=g10*g10; H[40]=g11*g11; H[56]=g10*g11;
  H[9]=g00*g10; H[25]=g01*g10; H[41]=g00*g11; H[57]=g01*g11;
  H[10]=g01*g11; H[26]=g01*g10; H[42]=g00*g11; H[58]=g00*g10;
  H[11]=g00*g01; H[27]=g01*g01; H[43]=g00*g00; H[59]=g00*g01;
  H[12]=g01*g10; H[28]=g00*g10; H[44]=g01*g11; H[60]=g00*g11;
  H[13]=g10*g10; H[29]=g10*g11; H[45]=g10*g11; H[61]=g11*g11;
  H[14]=g01*g01; H[30]=g00*g01; H[46]=g00*g01; H[62]=g00*g00;
  H[15]=g01*g10; H[31]=g01*g11; H[47]=g00*g10; H[63]=g00*g11;
}


void H23(double g00, double g01, double g10, double g11, double H[64]) {
  H[0]=g01*g10; H[16]=g01*g11; H[32]=g00*g10; H[48]=g00*g11;
  H[1]=g01*g01; H[17]=g00*g01; H[33]=g00*g01; H[49]=g00*g00;
  H[2]=g10*g10; H[18]=g10*g11; H[34]=g10*g11; H[50]=g11*g11;
  H[3]=g01*g10; H[19]=g00*g10; H[35]=g01*g11; H[51]=g00*g11;
  H[4]=g00*g01; H[20]=g01*g01; H[36]=g00*g00; H[52]=g00*g01;
  H[5]=g01*g11; H[21]=g01*g10; H[37]=g00*g11; H[53]=g00*g10;
  H[6]=g00*g10; H[22]=g01*g10; H[38]=g00*g11; H[54]=g01*g11;
  H[7]=g10*g11; H[23]=g10*g10; H[39]=g11*g11; H[55]=g10*g11;
  H[8]=g10*g11; H[24]=g11*g11; H[40]=g10*g10; H[56]=g10*g11;
  H[9]=g01*g11; H[25]=g00*g11; H[41]=g01*g10; H[57]=g00*g10;
  H[10]=g00*g10; H[26]=g00*g11; H[42]=g01*g10; H[58]=g01*g11;
  H[11]=g00*g01; H[27]=g00*g00; H[43]=g01*g01; H[59]=g00*g01;
  H[12]=g00*g11; H[28]=g01*g11; H[44]=g00*g10; H[60]=g01*g10;
  H[13]=g11*g11; H[29]=g10*g11; H[45]=g10*g11; H[61]=g10*g10;
  H[14]=g00*g00; H[30]=g00*g01; H[46]=g00*g01; H[62]=g01*g01;
  H[15]=g00*g11; H[31]=g00*g10; H[47]=g01*g11; H[63]=g01*g10;
}


void H24(double g00, double g01, double g10, double g11, double H[64]) {
  H[0]=g01*g11; H[16]=g01*g10; H[32]=g00*g11; H[48]=g00*g10;
  H[1]=g00*g01; H[17]=g01*g01; H[33]=g00*g00; H[49]=g00*g01;
  H[2]=g10*g11; H[18]=g10*g10; H[34]=g11*g11; H[50]=g10*g11;
  H[3]=g00*g10; H[19]=g01*g10; H[35]=g00*g11; H[51]=g01*g11;
  H[4]=g01*g01; H[20]=g00*g01; H[36]=g00*g01; H[52]=g00*g00;
  H[5]=g01*g10; H[21]=g01*g11; H[37]=g00*g10; H[53]=g00*g11;
  H[6]=g01*g10; H[22]=g00*g10; H[38]=g01*g11; H[54]=g00*g11;
  H[7]=g10*g10; H[23]=g10*g11; H[39]=g10*g11; H[55]=g11*g11;
  H[8]=g11*g11; H[24]=g10*g11; H[40]=g10*g11; H[56]=g10*g10;
  H[9]=g00*g11; H[25]=g01*g11; H[41]=g00*g10; H[57]=g01*g10;
  H[10]=g00*g11; H[26]=g00*g10; H[42]=g01*g11; H[58]=g01*g10;
  H[11]=g00*g00; H[27]=g00*g01; H[43]=g00*g01; H[59]=g01*g01;
  H[12]=g01*g11; H[28]=g00*g11; H[44]=g01*g10; H[60]=g00*g10;
  H[13]=g10*g11; H[29]=g11*g11; H[45]=g10*g10; H[61]=g10*g11;
  H[14]=g00*g01; H[30]=g00*g00; H[46]=g01*g01; H[62]=g00*g01;
  H[15]=g00*g10; H[31]=g00*g11; H[47]=g01*g10; H[63]=g01*g11;
}


void H31(double g00, double g01, double g10, double g11, double H[64]) {
  H[0]=g10*g00; H[16]=g10*g01; H[32]=g11*g00; H[48]=g11*g01;
  H[1]=g10*g11; H[17]=g10*g10; H[33]=g11*g11; H[49]=g11*g10;
  H[2]=g01*g00; H[18]=g01*g01; H[34]=g00*g00; H[50]=g00*g01;
  H[3]=g01*g11; H[19]=g01*g10; H[35]=g00*g11; H[51]=g00*g10;
  H[4]=g10*g10; H[20]=g10*g11; H[36]=g11*g10; H[52]=g11*g11;
  H[5]=g10*g01; H[21]=g10*g00; H[37]=g11*g01; H[53]=g11*g00;
  H[6]=g01*g10; H[22]=g01*g11; H[38]=g00*g10; H[54]=g00*g11;
  H[7]=g01*g01; H[23]=g01*g00; H[39]=g00*g01; H[55]=g00*g00;
  H[8]=g00*g00; H[24]=g00*g01; H[40]=g01*g00; H[56]=g01*g01;
  H[9]=g00*g11; H[25]=g00*g10; H[41]=g01*g11; H[57]=g01*g10;
  H[10]=g11*g00; H[26]=g11*g01; H[42]=g10*g00; H[58]=g10*g01;
  H[11]=g11*g11; H[27]=g11*g10; H[43]=g10*g11; H[59]=g10*g10;
  H[12]=g00*g10; H[28]=g00*g11; H[44]=g01*g10; H[60]=g01*g11;
  H[13]=g00*g01; H[29]=g00*g00; H[45]=g01*g01; H[61]=g01*g00;
  H[14]=g11*g10; H[30]=g11*g11; H[46]=g10*g10; H[62]=g10*g11;
  H[15]=g11*g01; H[31]=g11*g00; H[47]=g10*g01; H[63]=g10*g00;
}


void H32(double g00, double g01, double g10, double g11, double H[64]) {
  H[0]=g10*g01; H[16]=g10*g00; H[32]=g11*g01; H[48]=g11*g00;
  H[1]=g10*g10; H[17]=g10*g11; H[33]=g11*g10; H[49]=g11*g11;
  H[2]=g01*g01; H[18]=g01*g00; H[34]=g00*g01; H[50]=g00*g00;
  H[3]=g01*g10; H[19]=g01*g11; H[35]=g00*g10; H[51]=g00*g11;
  H[4]=g10*g11; H[20]=g10*g10; H[36]=g11*g11; H[52]=g11*g10;
  H[5]=g10*g00; H[21]=g10*g01; H[37]=g11*g00; H[53]=g11*g01;
  H[6]=g01*g11; H[22]=g01*g10; H[38]=g00*g11; H[54]=g00*g10;
  H[7]=g01*g00; H[23]=g01*g01; H[39]=g00*g00; H[55]=g00*g01;
  H[8]=g00*g01; H[24]=g00*g00; H[40]=g01*g01; H[56]=g01*g00;
  H[9]=g00*g10; H[25]=g00*g11; H[41]=g01*g10; H[57]=g01*g11;
  H[10]=g11*g01; H[26]=g11*g00; H[42]=g10*g01; H[58]=g10*g00;
  H[11]=g11*g10; H[27]=g11*g11; H[43]=g10*g10; H[59]=g10*g11;
  H[12]=g00*g11; H[28]=g00*g10; H[44]=g01*g11; H[60]=g01*g10;
  H[13]=g00*g00; H[29]=g00*g01; H[45]=g01*g00; H[61]=g01*g01;
  H[14]=g11*g11; H[30]=g11*g10; H[46]=g10*g11; H[62]=g10*g10;
  H[15]=g11*g00; H[31]=g11*g01; H[47]=g10*g00; H[63]=g10*g01;
}


void H33(double g00, double g01, double g10, double g11, double H[64]) {
  H[0]=g11*g00; H[16]=g11*g01; H[32]=g10*g00; H[48]=g10*g01;
  H[1]=g11*g11; H[17]=g11*g10; H[33]=g10*g11; H[49]=g10*g10;
  H[2]=g00*g00; H[18]=g00*g01; H[34]=g01*g00; H[50]=g01*g01;
  H[3]=g00*g11; H[19]=g00*g10; H[35]=g01*g11; H[51]=g01*g10;
  H[4]=g11*g10; H[20]=g11*g11; H[36]=g10*g10; H[52]=g10*g11;
  H[5]=g11*g01; H[21]=g11*g00; H[37]=g10*g01; H[53]=g10*g00;
  H[6]=g00*g10; H[22]=g00*g11; H[38]=g01*g10; H[54]=g01*g11;
  H[7]=g00*g01; H[23]=g00*g00; H[39]=g01*g01; H[55]=g01*g00;
  H[8]=g01*g00; H[24]=g01*g01; H[40]=g00*g00; H[56]=g00*g01;
  H[9]=g01*g11; H[25]=g01*g10; H[41]=g00*g11; H[57]=g00*g10;
  H[10]=g10*g00; H[26]=g10*g01; H[42]=g11*g00; H[58]=g11*g01;
  H[11]=g10*g11; H[27]=g10*g10; H[43]=g11*g11; H[59]=g11*g10;
  H[12]=g01*g10; H[28]=g01*g11; H[44]=g00*g10; H[60]=g00*g11;
  H[13]=g01*g01; H[29]=g01*g00; H[45]=g00*g01; H[61]=g00*g00;
  H[14]=g10*g10; H[30]=g10*g11; H[46]=g11*g10; H[62]=g11*g11;
  H[15]=g10*g01; H[31]=g10*g00; H[47]=g11*g01; H[63]=g11*g00;
}


void H34(double g00, double g01, double g10, double g11, double H[64]) {
  H[0]=g11*g01; H[16]=g11*g00; H[32]=g10*g01; H[48]=g10*g00;
  H[1]=g11*g10; H[17]=g11*g11; H[33]=g10*g10; H[49]=g10*g11;
  H[2]=g00*g01; H[18]=g00*g00; H[34]=g01*g01; H[50]=g01*g00;
  H[3]=g00*g10; H[19]=g00*g11; H[35]=g01*g10; H[51]=g01*g11;
  H[4]=g11*g11; H[20]=g11*g10; H[36]=g10*g11; H[52]=g10*g10;
  H[5]=g11*g00; H[21]=g11*g01; H[37]=g10*g00; H[53]=g10*g01;
  H[6]=g00*g11; H[22]=g00*g10; H[38]=g01*g11; H[54]=g01*g10;
  H[7]=g00*g00; H[23]=g00*g01; H[39]=g01*g00; H[55]=g01*g01;
  H[8]=g01*g01; H[24]=g01*g00; H[40]=g00*g01; H[56]=g00*g00;
  H[9]=g01*g10; H[25]=g01*g11; H[41]=g00*g10; H[57]=g00*g11;
  H[10]=g10*g01; H[26]=g10*g00; H[42]=g11*g01; H[58]=g11*g00;
  H[11]=g10*g10; H[27]=g10*g11; H[43]=g11*g10; H[59]=g11*g11;
  H[12]=g01*g11; H[28]=g01*g10; H[44]=g00*g11; H[60]=g00*g10;
  H[13]=g01*g00; H[29]=g01*g01; H[45]=g00*g00; H[61]=g00*g01;
  H[14]=g10*g11; H[30]=g10*g10; H[46]=g11*g11; H[62]=g11*g10;
  H[15]=g10*g00; H[31]=g10*g01; H[47]=g11*g00; H[63]=g11*g01;
}


void H41(double g00, double g01, double g10, double g11, double H[64]) {
  H[0]=g10*g10; H[16]=g10*g11; H[32]=g11*g10; H[48]=g11*g11;
  H[1]=g10*g01; H[17]=g10*g00; H[33]=g11*g01; H[49]=g11*g00;
  H[2]=g01*g10; H[18]=g01*g11; H[34]=g00*g10; H[50]=g00*g11;
  H[3]=g01*g01; H[19]=g01*g00; H[35]=g00*g01; H[51]=g00*g00;
  H[4]=g10*g00; H[20]=g10*g01; H[36]=g11*g00; H[52]=g11*g01;
  H[5]=g10*g11; H[21]=g10*g10; H[37]=g11*g11; H[53]=g11*g10;
  H[6]=g01*g00; H[22]=g01*g01; H[38]=g00*g00; H[54]=g00*g01;
  H[7]=g01*g11; H[23]=g01*g10; H[39]=g00*g11; H[55]=g00*g10;
  H[8]=g00*g10; H[24]=g00*g11; H[40]=g01*g10; H[56]=g01*g11;
  H[9]=g00*g01; H[25]=g00*g00; H[41]=g01*g01; H[57]=g01*g00;
  H[10]=g11*g10; H[26]=g11*g11; H[42]=g10*g10; H[58]=g10*g11;
  H[11]=g11*g01; H[27]=g11*g00; H[43]=g10*g01; H[59]=g10*g00;
  H[12]=g00*g00; H[28]=g00*g01; H[44]=g01*g00; H[60]=g01*g01;
  H[13]=g00*g11; H[29]=g00*g10; H[45]=g01*g11; H[61]=g01*g10;
  H[14]=g11*g00; H[30]=g11*g01; H[46]=g10*g00; H[62]=g10*g01;
  H[15]=g11*g11; H[31]=g11*g10; H[47]=g10*g11; H[63]=g10*g10;
}


void H42(double g00, double g01, double g10, double g11, double H[64]) {
  H[0]=g10*g11; H[16]=g10*g10; H[32]=g11*g11; H[48]=g11*g10;
  H[1]=g10*g00; H[17]=g10*g01; H[33]=g11*g00; H[49]=g11*g01;
  H[2]=g01*g11; H[18]=g01*g10; H[34]=g00*g11; H[50]=g00*g10;
  H[3]=g01*g00; H[19]=g01*g01; H[35]=g00*g00; H[51]=g00*g01;
  H[4]=g10*g01; H[20]=g10*g00; H[36]=g11*g01; H[52]=g11*g00;
  H[5]=g10*g10; H[21]=g10*g11; H[37]=g11*g10; H[53]=g11*g11;
  H[6]=g01*g01; H[22]=g01*g00; H[38]=g00*g01; H[54]=g00*g00;
  H[7]=g01*g10; H[23]=g01*g11; H[39]=g00*g10; H[55]=g00*g11;
  H[8]=g00*g11; H[24]=g00*g10; H[40]=g01*g11; H[56]=g01*g10;
  H[9]=g00*g00; H[25]=g00*g01; H[41]=g01*g00; H[57]=g01*g01;
  H[10]=g11*g11; H[26]=g11*g10; H[42]=g10*g11; H[58]=g10*g10;
  H[11]=g11*g00; H[27]=g11*g01; H[43]=g10*g00; H[59]=g10*g01;
  H[12]=g00*g01; H[28]=g00*g00; H[44]=g01*g01; H[60]=g01*g00;
  H[13]=g00*g10; H[29]=g00*g11; H[45]=g01*g10; H[61]=g01*g11;
  H[14]=g11*g01; H[30]=g11*g00; H[46]=g10*g01; H[62]=g10*g00;
  H[15]=g11*g10; H[31]=g11*g11; H[47]=g10*g10; H[63]=g10*g11;
}


void H43(double g00, double g01, double g10, double g11, double H[64]) {
  H[0]=g11*g10; H[16]=g11*g11; H[32]=g10*g10; H[48]=g10*g11;
  H[1]=g11*g01; H[17]=g11*g00; H[33]=g10*g01; H[49]=g10*g00;
  H[2]=g00*g10; H[18]=g00*g11; H[34]=g01*g10; H[50]=g01*g11;
  H[3]=g00*g01; H[19]=g00*g00; H[35]=g01*g01; H[51]=g01*g00;
  H[4]=g11*g00; H[20]=g11*g01; H[36]=g10*g00; H[52]=g10*g01;
  H[5]=g11*g11; H[21]=g11*g10; H[37]=g10*g11; H[53]=g10*g10;
  H[6]=g00*g00; H[22]=g00*g01; H[38]=g01*g00; H[54]=g01*g01;
  H[7]=g00*g11; H[23]=g00*g10; H[39]=g01*g11; H[55]=g01*g10;
  H[8]=g01*g10; H[24]=g01*g11; H[40]=g00*g10; H[56]=g00*g11;
  H[9]=g01*g01; H[25]=g01*g00; H[41]=g00*g01; H[57]=g00*g00;
  H[10]=g10*g10; H[26]=g10*g11; H[42]=g11*g10; H[58]=g11*g11;
  H[11]=g10*g01; H[27]=g10*g00; H[43]=g11*g01; H[59]=g11*g00;
  H[12]=g01*g00; H[28]=g01*g01; H[44]=g00*g00; H[60]=g00*g01;
  H[13]=g01*g11; H[29]=g01*g10; H[45]=g00*g11; H[61]=g00*g10;
  H[14]=g10*g00; H[30]=g10*g01; H[46]=g11*g00; H[62]=g11*g01;
  H[15]=g10*g11; H[31]=g10*g10; H[47]=g11*g11; H[63]=g11*g10;
}


void H44(double g00, double g01, double g10, double g11, double H[64]) {
  H[0]=g11*g11; H[16]=g11*g10; H[32]=g10*g11; H[48]=g10*g10;
  H[1]=g11*g00; H[17]=g11*g01; H[33]=g10*g00; H[49]=g10*g01;
  H[2]=g00*g11; H[18]=g00*g10; H[34]=g01*g11; H[50]=g01*g10;
  H[3]=g00*g00; H[19]=g00*g01; H[35]=g01*g00; H[51]=g01*g01;
  H[4]=g11*g01; H[20]=g11*g00; H[36]=g10*g01; H[52]=g10*g00;
  H[5]=g11*g10; H[21]=g11*g11; H[37]=g10*g10; H[53]=g10*g11;
  H[6]=g00*g01; H[22]=g00*g00; H[38]=g01*g01; H[54]=g01*g00;
  H[7]=g00*g10; H[23]=g00*g11; H[39]=g01*g10; H[55]=g01*g11;
  H[8]=g01*g11; H[24]=g01*g10; H[40]=g00*g11; H[56]=g00*g10;
  H[9]=g01*g00; H[25]=g01*g01; H[41]=g00*g00; H[57]=g00*g01;
  H[10]=g10*g11; H[26]=g10*g10; H[42]=g11*g11; H[58]=g11*g10;
  H[11]=g10*g00; H[27]=g10*g01; H[43]=g11*g00; H[59]=g11*g01;
  H[12]=g01*g01; H[28]=g01*g00; H[44]=g00*g01; H[60]=g00*g00;
  H[13]=g01*g10; H[29]=g01*g11; H[45]=g00*g10; H[61]=g00*g11;
  H[14]=g10*g01; H[30]=g10*g00; H[46]=g11*g01; H[62]=g11*g00;
  H[15]=g10*g10; H[31]=g10*g11; H[47]=g11*g10; H[63]=g11*g11;
}

/* rf_3pt calculates recombination fractions and the log-likelihood between 3 markers for a given assignment */

void rf_3pt(double *I1, int p1, double *I2, int p2, double *I3, int p3, int *n, int ntot, void (*Hcall)(double, double, double, double, double [64]), double G00[64], double G01[64], double G10[64], double G11[64], double *theta12_assign, double *theta23_assign, double *theta13_assign, double *log_like_assign) {
  double g00, g01, g10, g11, two_n, suma, sumb, sumc, sumd, H[64], mid[64], *alpha, *beta, *gamma, *delta, *v, *kron12, *temp, log_Lold, log_Lnew, diff, mult;
  int i, iter;
  
  alpha = (double *)R_alloc(p1*p2*p3, sizeof(double));
  beta = (double *)R_alloc(p1*p2*p3, sizeof(double));
  gamma = (double *)R_alloc(p1*p2*p3, sizeof(double));
  delta = (double *)R_alloc(p1*p2*p3, sizeof(double));
  v = (double *)R_alloc(p1*p2*p3, sizeof(double));
  kron12 = (double *)R_alloc(p1*p2*16, sizeof(double));
  temp = (double *)R_alloc(p1*p2*4, sizeof(double));
  
  mkron(I1,p1,4,I2,p2,4,kron12);
  two_n = 2*ntot;

  g11 = 0.125; /* initial values for the probabilities of crossover events */
  g10 = 0.125;
  g01 = 0.125;
  g00 = 0.625;
  Hcall(g00, g01, g10, g11, H);
  mprod(kron12,p1*p2,16,(double *)H,16,4,temp);
  mprod(temp,p1*p2,4,I3,4,p3,v);

  /* the likelihood for the initial values */
  log_Lnew = 0.0;
  for (i=0;i<(p1*p2*p3);i++)
    log_Lnew += n[i]*log(v[i]);
  diff = 1;
  
  iter = 1;

  /* the probabilities will be calculated until convergence */
  while(diff>THRESH && iter<MAXITER) {
    iter++;
    log_Lold = log_Lnew;

    mdrct3pt(G00,H,mid);
    mprod(kron12,p1*p2,16,(double *)mid,16,4,temp);
    mprod(temp,p1*p2,4,I3,4,p3,alpha);

    mdrct3pt(G01,H,mid);
    mprod(kron12,p1*p2,16,(double *)mid,16,4,temp);
    mprod(temp,p1*p2,4,I3,4,p3,beta);

    mdrct3pt(G10,H,mid);
    mprod(kron12,p1*p2,16,(double *)mid,16,4,temp);
    mprod(temp,p1*p2,4,I3,4,p3,gamma);

    mdrct3pt(G11,H,mid);
    mprod(kron12,p1*p2,16,(double *)mid,16,4,temp);
    mprod(temp,p1*p2,4,I3,4,p3,delta);
    
    suma = sumb = sumc = sumd = 0.0;
    for (i=0;i<(p1*p2*p3);i++) {
      mult = n[i]/v[i];
      suma += mult*alpha[i];
      sumb += mult*beta[i];
      sumc += mult*gamma[i];
      sumd += mult*delta[i];
    }
    g00 = suma/(two_n); /* new values for the probabilities of crossover events */
    g01 = sumb/(two_n);
    g10 = sumc/(two_n);
    g11 = sumd/(two_n);

    Hcall(g00, g01, g10, g11, H);
    mprod(kron12,p1*p2,16,(double *)H,16,4,temp);
    mprod(temp,p1*p2,4,I3,4,p3,v);
    log_Lnew = 0.0;
    for (i=0;i<(p1*p2*p3);i++)
      log_Lnew += n[i]*log(v[i]);
    diff = log_sub(log_Lnew,log_Lold) - log_Lnew;
  }

  *theta12_assign = g10+g11; /* the recombination fractions are calculated and passed to R */
  *theta23_assign = g01+g11;
  *theta13_assign = g01+g10;
  *log_like_assign = log_Lnew;
}

void r3pts(double *I1, int *p1, double *I2, int *p2, double *I3, int *p3, int *n, int *ntot, double *theta12, double *theta23, double *theta13, double *log_like, double *posterior, double *LOD) {
  double G00[64], G01[64], G10[64], G11[64], H[64], log_sum_like, log_like_null, *kron12, *v, *temp;
  int i;
  
  kron12 = (double *)R_alloc(*p1 * *p2 * 16, sizeof(double));
  v = (double *)R_alloc(*p1 * *p2 * *p3, sizeof(double));
  temp = (double *)R_alloc(*p1 * *p2 * 4, sizeof(double));

  /* assignment 1-1 */
  /* the following matrices were obtained by the author and colleagues,
     according to the paper */
  G00[0]=2; G00[16]=1; G00[32]=1; G00[48]=0;
  G00[1]=1; G00[17]=1; G00[33]=0; G00[49]=0;
  G00[2]=1; G00[18]=0; G00[34]=1; G00[50]=0;
  G00[3]=0; G00[19]=0; G00[35]=0; G00[51]=0;
  G00[4]=1; G00[20]=1; G00[36]=0; G00[52]=0;
  G00[5]=1; G00[21]=2; G00[37]=0; G00[53]=1;
  G00[6]=0; G00[22]=0; G00[38]=0; G00[54]=0;
  G00[7]=0; G00[23]=1; G00[39]=0; G00[55]=1;
  G00[8]=1; G00[24]=0; G00[40]=1; G00[56]=0;
  G00[9]=0; G00[25]=0; G00[41]=0; G00[57]=0;
  G00[10]=1; G00[26]=0; G00[42]=2; G00[58]=1;
  G00[11]=0; G00[27]=0; G00[43]=1; G00[59]=1;
  G00[12]=0; G00[28]=0; G00[44]=0; G00[60]=0;
  G00[13]=0; G00[29]=1; G00[45]=0; G00[61]=1;
  G00[14]=0; G00[30]=0; G00[46]=1; G00[62]=1;
  G00[15]=0; G00[31]=1; G00[47]=1; G00[63]=2;
  
  G01[0]=0; G01[16]=1; G01[32]=1; G01[48]=2;
  G01[1]=0; G01[17]=0; G01[33]=1; G01[49]=1;
  G01[2]=0; G01[18]=1; G01[34]=0; G01[50]=1;
  G01[3]=0; G01[19]=0; G01[35]=0; G01[51]=0;
  G01[4]=0; G01[20]=0; G01[36]=1; G01[52]=1;
  G01[5]=1; G01[21]=0; G01[37]=2; G01[53]=1;
  G01[6]=0; G01[22]=0; G01[38]=0; G01[54]=0;
  G01[7]=1; G01[23]=0; G01[39]=1; G01[55]=0;
  G01[8]=0; G01[24]=1; G01[40]=0; G01[56]=1;
  G01[9]=0; G01[25]=0; G01[41]=0; G01[57]=0;
  G01[10]=1; G01[26]=2; G01[42]=0; G01[58]=1;
  G01[11]=1; G01[27]=1; G01[43]=0; G01[59]=0;
  G01[12]=0; G01[28]=0; G01[44]=0; G01[60]=0;
  G01[13]=1; G01[29]=0; G01[45]=1; G01[61]=0;
  G01[14]=1; G01[30]=1; G01[46]=0; G01[62]=0;
  G01[15]=2; G01[31]=1; G01[47]=1; G01[63]=0;
  
  G10[0]=0; G10[16]=0; G10[32]=0; G10[48]=0;
  G10[1]=0; G10[17]=1; G10[33]=0; G10[49]=1;
  G10[2]=0; G10[18]=0; G10[34]=1; G10[50]=1;
  G10[3]=0; G10[19]=1; G10[35]=1; G10[51]=2;
  G10[4]=1; G10[20]=0; G10[36]=1; G10[52]=0;
  G10[5]=0; G10[21]=0; G10[37]=0; G10[53]=0;
  G10[6]=1; G10[22]=0; G10[38]=2; G10[54]=1;
  G10[7]=0; G10[23]=0; G10[39]=1; G10[55]=1;
  G10[8]=1; G10[24]=1; G10[40]=0; G10[56]=0;
  G10[9]=1; G10[25]=2; G10[41]=0; G10[57]=1;
  G10[10]=0; G10[26]=0; G10[42]=0; G10[58]=0;
  G10[11]=0; G10[27]=1; G10[43]=0; G10[59]=1;
  G10[12]=2; G10[28]=1; G10[44]=1; G10[60]=0;
  G10[13]=1; G10[29]=1; G10[45]=0; G10[61]=0;
  G10[14]=1; G10[30]=0; G10[46]=1; G10[62]=0;
  G10[15]=0; G10[31]=0; G10[47]=0; G10[63]=0;
  
  G11[0]=0; G11[16]=0; G11[32]=0; G11[48]=0;
  G11[1]=1; G11[17]=0; G11[33]=1; G11[49]=0;
  G11[2]=1; G11[18]=1; G11[34]=0; G11[50]=0;
  G11[3]=2; G11[19]=1; G11[35]=1; G11[51]=0;
  G11[4]=0; G11[20]=1; G11[36]=0; G11[52]=1;
  G11[5]=0; G11[21]=0; G11[37]=0; G11[53]=0;
  G11[6]=1; G11[22]=2; G11[38]=0; G11[54]=1;
  G11[7]=1; G11[23]=1; G11[39]=0; G11[55]=0;
  G11[8]=0; G11[24]=0; G11[40]=1; G11[56]=1;
  G11[9]=1; G11[25]=0; G11[41]=2; G11[57]=1;
  G11[10]=0; G11[26]=0; G11[42]=0; G11[58]=0;
  G11[11]=1; G11[27]=0; G11[43]=1; G11[59]=0;
  G11[12]=0; G11[28]=1; G11[44]=1; G11[60]=2;
  G11[13]=0; G11[29]=0; G11[45]=1; G11[61]=1;
  G11[14]=0; G11[30]=1; G11[46]=0; G11[62]=1;
  G11[15]=0; G11[31]=0; G11[47]=0; G11[63]=0;
  
  rf_3pt(I1,*p1,I2,*p2,I3,*p3,n,*ntot,H11,G00,G01,G10,G11,&theta12[0],&theta23[0],&theta13[0],&log_like[0]);
  
  /* assignment 1-2 */
  G00[0]=1; G00[16]=2; G00[32]=0; G00[48]=1;
  G00[1]=1; G00[17]=1; G00[33]=0; G00[49]=0;
  G00[2]=0; G00[18]=1; G00[34]=0; G00[50]=1;
  G00[3]=0; G00[19]=0; G00[35]=0; G00[51]=0;
  G00[4]=1; G00[20]=1; G00[36]=0; G00[52]=0;
  G00[5]=2; G00[21]=1; G00[37]=1; G00[53]=0;
  G00[6]=0; G00[22]=0; G00[38]=0; G00[54]=0;
  G00[7]=1; G00[23]=0; G00[39]=1; G00[55]=0;
  G00[8]=0; G00[24]=1; G00[40]=0; G00[56]=1;
  G00[9]=0; G00[25]=0; G00[41]=0; G00[57]=0;
  G00[10]=0; G00[26]=1; G00[42]=1; G00[58]=2;
  G00[11]=0; G00[27]=0; G00[43]=1; G00[59]=1;
  G00[12]=0; G00[28]=0; G00[44]=0; G00[60]=0;
  G00[13]=1; G00[29]=0; G00[45]=1; G00[61]=0;
  G00[14]=0; G00[30]=0; G00[46]=1; G00[62]=1;
  G00[15]=1; G00[31]=0; G00[47]=2; G00[63]=1;
  
  G01[0]=1; G01[16]=0; G01[32]=2; G01[48]=1;
  G01[1]=0; G01[17]=0; G01[33]=1; G01[49]=1;
  G01[2]=1; G01[18]=0; G01[34]=1; G01[50]=0;
  G01[3]=0; G01[19]=0; G01[35]=0; G01[51]=0;
  G01[4]=0; G01[20]=0; G01[36]=1; G01[52]=1;
  G01[5]=0; G01[21]=1; G01[37]=1; G01[53]=2;
  G01[6]=0; G01[22]=0; G01[38]=0; G01[54]=0;
  G01[7]=0; G01[23]=1; G01[39]=0; G01[55]=1;
  G01[8]=1; G01[24]=0; G01[40]=1; G01[56]=0;
  G01[9]=0; G01[25]=0; G01[41]=0; G01[57]=0;
  G01[10]=2; G01[26]=1; G01[42]=1; G01[58]=0;
  G01[11]=1; G01[27]=1; G01[43]=0; G01[59]=0;
  G01[12]=0; G01[28]=0; G01[44]=0; G01[60]=0;
  G01[13]=0; G01[29]=1; G01[45]=0; G01[61]=1;
  G01[14]=1; G01[30]=1; G01[46]=0; G01[62]=0;
  G01[15]=1; G01[31]=2; G01[47]=0; G01[63]=1;
  
  G10[0]=0; G10[16]=0; G10[32]=0; G10[48]=0;
  G10[1]=1; G10[17]=0; G10[33]=1; G10[49]=0;
  G10[2]=0; G10[18]=0; G10[34]=1; G10[50]=1;
  G10[3]=1; G10[19]=0; G10[35]=2; G10[51]=1;
  G10[4]=0; G10[20]=1; G10[36]=0; G10[52]=1;
  G10[5]=0; G10[21]=0; G10[37]=0; G10[53]=0;
  G10[6]=0; G10[22]=1; G10[38]=1; G10[54]=2;
  G10[7]=0; G10[23]=0; G10[39]=1; G10[55]=1;
  G10[8]=1; G10[24]=1; G10[40]=0; G10[56]=0;
  G10[9]=2; G10[25]=1; G10[41]=1; G10[57]=0;
  G10[10]=0; G10[26]=0; G10[42]=0; G10[58]=0;
  G10[11]=1; G10[27]=0; G10[43]=1; G10[59]=0;
  G10[12]=1; G10[28]=2; G10[44]=0; G10[60]=1;
  G10[13]=1; G10[29]=1; G10[45]=0; G10[61]=0;
  G10[14]=0; G10[30]=1; G10[46]=0; G10[62]=1;
  G10[15]=0; G10[31]=0; G10[47]=0; G10[63]=0;
  
  G11[0]=0; G11[16]=0; G11[32]=0; G11[48]=0;
  G11[1]=0; G11[17]=1; G11[33]=0; G11[49]=1;
  G11[2]=1; G11[18]=1; G11[34]=0; G11[50]=0;
  G11[3]=1; G11[19]=2; G11[35]=0; G11[51]=1;
  G11[4]=1; G11[20]=0; G11[36]=1; G11[52]=0;
  G11[5]=0; G11[21]=0; G11[37]=0; G11[53]=0;
  G11[6]=2; G11[22]=1; G11[38]=1; G11[54]=0;
  G11[7]=1; G11[23]=1; G11[39]=0; G11[55]=0;
  G11[8]=0; G11[24]=0; G11[40]=1; G11[56]=1;
  G11[9]=0; G11[25]=1; G11[41]=1; G11[57]=2;
  G11[10]=0; G11[26]=0; G11[42]=0; G11[58]=0;
  G11[11]=0; G11[27]=1; G11[43]=0; G11[59]=1;
  G11[12]=1; G11[28]=0; G11[44]=2; G11[60]=1;
  G11[13]=0; G11[29]=0; G11[45]=1; G11[61]=1;
  G11[14]=1; G11[30]=0; G11[46]=1; G11[62]=0;
  G11[15]=0; G11[31]=0; G11[47]=0; G11[63]=0;
  
  rf_3pt(I1,*p1,I2,*p2,I3,*p3,n,*ntot,H12,G00,G01,G10,G11,&theta12[1],&theta23[1],&theta13[1],&log_like[1]);
  
  /* assignment 1-3 */
  G00[0]=1; G00[16]=0; G00[32]=2; G00[48]=1;
  G00[1]=0; G00[17]=0; G00[33]=1; G00[49]=1;
  G00[2]=1; G00[18]=0; G00[34]=1; G00[50]=0;
  G00[3]=0; G00[19]=0; G00[35]=0; G00[51]=0;
  G00[4]=0; G00[20]=0; G00[36]=1; G00[52]=1;
  G00[5]=0; G00[21]=1; G00[37]=1; G00[53]=2;
  G00[6]=0; G00[22]=0; G00[38]=0; G00[54]=0;
  G00[7]=0; G00[23]=1; G00[39]=0; G00[55]=1;
  G00[8]=1; G00[24]=0; G00[40]=1; G00[56]=0;
  G00[9]=0; G00[25]=0; G00[41]=0; G00[57]=0;
  G00[10]=2; G00[26]=1; G00[42]=1; G00[58]=0;
  G00[11]=1; G00[27]=1; G00[43]=0; G00[59]=0;
  G00[12]=0; G00[28]=0; G00[44]=0; G00[60]=0;
  G00[13]=0; G00[29]=1; G00[45]=0; G00[61]=1;
  G00[14]=1; G00[30]=1; G00[46]=0; G00[62]=0;
  G00[15]=1; G00[31]=2; G00[47]=0; G00[63]=1;
  
  G01[0]=1; G01[16]=2; G01[32]=0; G01[48]=1;
  G01[1]=1; G01[17]=1; G01[33]=0; G01[49]=0;
  G01[2]=0; G01[18]=1; G01[34]=0; G01[50]=1;
  G01[3]=0; G01[19]=0; G01[35]=0; G01[51]=0;
  G01[4]=1; G01[20]=1; G01[36]=0; G01[52]=0;
  G01[5]=2; G01[21]=1; G01[37]=1; G01[53]=0;
  G01[6]=0; G01[22]=0; G01[38]=0; G01[54]=0;
  G01[7]=1; G01[23]=0; G01[39]=1; G01[55]=0;
  G01[8]=0; G01[24]=1; G01[40]=0; G01[56]=1;
  G01[9]=0; G01[25]=0; G01[41]=0; G01[57]=0;
  G01[10]=0; G01[26]=1; G01[42]=1; G01[58]=2;
  G01[11]=0; G01[27]=0; G01[43]=1; G01[59]=1;
  G01[12]=0; G01[28]=0; G01[44]=0; G01[60]=0;
  G01[13]=1; G01[29]=0; G01[45]=1; G01[61]=0;
  G01[14]=0; G01[30]=0; G01[46]=1; G01[62]=1;
  G01[15]=1; G01[31]=0; G01[47]=2; G01[63]=1;
  
  G10[0]=0; G10[16]=0; G10[32]=0; G10[48]=0;
  G10[1]=0; G10[17]=1; G10[33]=0; G10[49]=1;
  G10[2]=1; G10[18]=1; G10[34]=0; G10[50]=0;
  G10[3]=1; G10[19]=2; G10[35]=0; G10[51]=1;
  G10[4]=1; G10[20]=0; G10[36]=1; G10[52]=0;
  G10[5]=0; G10[21]=0; G10[37]=0; G10[53]=0;
  G10[6]=2; G10[22]=1; G10[38]=1; G10[54]=0;
  G10[7]=1; G10[23]=1; G10[39]=0; G10[55]=0;
  G10[8]=0; G10[24]=0; G10[40]=1; G10[56]=1;
  G10[9]=0; G10[25]=1; G10[41]=1; G10[57]=2;
  G10[10]=0; G10[26]=0; G10[42]=0; G10[58]=0;
  G10[11]=0; G10[27]=1; G10[43]=0; G10[59]=1;
  G10[12]=1; G10[28]=0; G10[44]=2; G10[60]=1;
  G10[13]=0; G10[29]=0; G10[45]=1; G10[61]=1;
  G10[14]=1; G10[30]=0; G10[46]=1; G10[62]=0;
  G10[15]=0; G10[31]=0; G10[47]=0; G10[63]=0;
  
  G11[0]=0; G11[16]=0; G11[32]=0; G11[48]=0;
  G11[1]=1; G11[17]=0; G11[33]=1; G11[49]=0;
  G11[2]=0; G11[18]=0; G11[34]=1; G11[50]=1;
  G11[3]=1; G11[19]=0; G11[35]=2; G11[51]=1;
  G11[4]=0; G11[20]=1; G11[36]=0; G11[52]=1;
  G11[5]=0; G11[21]=0; G11[37]=0; G11[53]=0;
  G11[6]=0; G11[22]=1; G11[38]=1; G11[54]=2;
  G11[7]=0; G11[23]=0; G11[39]=1; G11[55]=1;
  G11[8]=1; G11[24]=1; G11[40]=0; G11[56]=0;
  G11[9]=2; G11[25]=1; G11[41]=1; G11[57]=0;
  G11[10]=0; G11[26]=0; G11[42]=0; G11[58]=0;
  G11[11]=1; G11[27]=0; G11[43]=1; G11[59]=0;
  G11[12]=1; G11[28]=2; G11[44]=0; G11[60]=1;
  G11[13]=1; G11[29]=1; G11[45]=0; G11[61]=0;
  G11[14]=0; G11[30]=1; G11[46]=0; G11[62]=1;
  G11[15]=0; G11[31]=0; G11[47]=0; G11[63]=0;
  
  rf_3pt(I1,*p1,I2,*p2,I3,*p3,n,*ntot,H13,G00,G01,G10,G11,&theta12[2],&theta23[2],&theta13[2],&log_like[2]);   
  
  /* assignment 1-4 */
  G00[0]=0; G00[16]=1; G00[32]=1; G00[48]=2;
  G00[1]=0; G00[17]=0; G00[33]=1; G00[49]=1;
  G00[2]=0; G00[18]=1; G00[34]=0; G00[50]=1;
  G00[3]=0; G00[19]=0; G00[35]=0; G00[51]=0;
  G00[4]=0; G00[20]=0; G00[36]=1; G00[52]=1;
  G00[5]=1; G00[21]=0; G00[37]=2; G00[53]=1;
  G00[6]=0; G00[22]=0; G00[38]=0; G00[54]=0;
  G00[7]=1; G00[23]=0; G00[39]=1; G00[55]=0;
  G00[8]=0; G00[24]=1; G00[40]=0; G00[56]=1;
  G00[9]=0; G00[25]=0; G00[41]=0; G00[57]=0;
  G00[10]=1; G00[26]=2; G00[42]=0; G00[58]=1;
  G00[11]=1; G00[27]=1; G00[43]=0; G00[59]=0;
  G00[12]=0; G00[28]=0; G00[44]=0; G00[60]=0;
  G00[13]=1; G00[29]=0; G00[45]=1; G00[61]=0;
  G00[14]=1; G00[30]=1; G00[46]=0; G00[62]=0;
  G00[15]=2; G00[31]=1; G00[47]=1; G00[63]=0;
  
  G01[0]=2; G01[16]=1; G01[32]=1; G01[48]=0;
  G01[1]=1; G01[17]=1; G01[33]=0; G01[49]=0;
  G01[2]=1; G01[18]=0; G01[34]=1; G01[50]=0;
  G01[3]=0; G01[19]=0; G01[35]=0; G01[51]=0;
  G01[4]=1; G01[20]=1; G01[36]=0; G01[52]=0;
  G01[5]=1; G01[21]=2; G01[37]=0; G01[53]=1;
  G01[6]=0; G01[22]=0; G01[38]=0; G01[54]=0;
  G01[7]=0; G01[23]=1; G01[39]=0; G01[55]=1;
  G01[8]=1; G01[24]=0; G01[40]=1; G01[56]=0;
  G01[9]=0; G01[25]=0; G01[41]=0; G01[57]=0;
  G01[10]=1; G01[26]=0; G01[42]=2; G01[58]=1;
  G01[11]=0; G01[27]=0; G01[43]=1; G01[59]=1;
  G01[12]=0; G01[28]=0; G01[44]=0; G01[60]=0;
  G01[13]=0; G01[29]=1; G01[45]=0; G01[61]=1;
  G01[14]=0; G01[30]=0; G01[46]=1; G01[62]=1;
  G01[15]=0; G01[31]=1; G01[47]=1; G01[63]=2;
  
  G10[0]=0; G10[16]=0; G10[32]=0; G10[48]=0;
  G10[1]=1; G10[17]=0; G10[33]=1; G10[49]=0;
  G10[2]=1; G10[18]=1; G10[34]=0; G10[50]=0;
  G10[3]=2; G10[19]=1; G10[35]=1; G10[51]=0;
  G10[4]=0; G10[20]=1; G10[36]=0; G10[52]=1;
  G10[5]=0; G10[21]=0; G10[37]=0; G10[53]=0;
  G10[6]=1; G10[22]=2; G10[38]=0; G10[54]=1;
  G10[7]=1; G10[23]=1; G10[39]=0; G10[55]=0;
  G10[8]=0; G10[24]=0; G10[40]=1; G10[56]=1;
  G10[9]=1; G10[25]=0; G10[41]=2; G10[57]=1;
  G10[10]=0; G10[26]=0; G10[42]=0; G10[58]=0;
  G10[11]=1; G10[27]=0; G10[43]=1; G10[59]=0;
  G10[12]=0; G10[28]=1; G10[44]=1; G10[60]=2;
  G10[13]=0; G10[29]=0; G10[45]=1; G10[61]=1;
  G10[14]=0; G10[30]=1; G10[46]=0; G10[62]=1;
  G10[15]=0; G10[31]=0; G10[47]=0; G10[63]=0;
  
  G11[0]=0; G11[16]=0; G11[32]=0; G11[48]=0;
  G11[1]=0; G11[17]=1; G11[33]=0; G11[49]=1;
  G11[2]=0; G11[18]=0; G11[34]=1; G11[50]=1;
  G11[3]=0; G11[19]=1; G11[35]=1; G11[51]=2;
  G11[4]=1; G11[20]=0; G11[36]=1; G11[52]=0;
  G11[5]=0; G11[21]=0; G11[37]=0; G11[53]=0;
  G11[6]=1; G11[22]=0; G11[38]=2; G11[54]=1;
  G11[7]=0; G11[23]=0; G11[39]=1; G11[55]=1;
  G11[8]=1; G11[24]=1; G11[40]=0; G11[56]=0;
  G11[9]=1; G11[25]=2; G11[41]=0; G11[57]=1;
  G11[10]=0; G11[26]=0; G11[42]=0; G11[58]=0;
  G11[11]=0; G11[27]=1; G11[43]=0; G11[59]=1;
  G11[12]=2; G11[28]=1; G11[44]=1; G11[60]=0;
  G11[13]=1; G11[29]=1; G11[45]=0; G11[61]=0;
  G11[14]=1; G11[30]=0; G11[46]=1; G11[62]=0;
  G11[15]=0; G11[31]=0; G11[47]=0; G11[63]=0;
  
  rf_3pt(I1,*p1,I2,*p2,I3,*p3,n,*ntot,H14,G00,G01,G10,G11,&theta12[3],&theta23[3],&theta13[3],&log_like[3]);
  
  /* assignment 2-1 */
  G00[0]=1; G00[16]=1; G00[32]=0; G00[48]=0;
  G00[1]=1; G00[17]=2; G00[33]=0; G00[49]=1;
  G00[2]=0; G00[18]=0; G00[34]=0; G00[50]=0;
  G00[3]=0; G00[19]=1; G00[35]=0; G00[51]=1;
  G00[4]=2; G00[20]=1; G00[36]=1; G00[52]=0;
  G00[5]=1; G00[21]=1; G00[37]=0; G00[53]=0;
  G00[6]=1; G00[22]=0; G00[38]=1; G00[54]=0;
  G00[7]=0; G00[23]=0; G00[39]=0; G00[55]=0;
  G00[8]=0; G00[24]=0; G00[40]=0; G00[56]=0;
  G00[9]=0; G00[25]=1; G00[41]=0; G00[57]=1;
  G00[10]=0; G00[26]=0; G00[42]=1; G00[58]=1;
  G00[11]=0; G00[27]=1; G00[43]=1; G00[59]=2;
  G00[12]=1; G00[28]=0; G00[44]=1; G00[60]=0;
  G00[13]=0; G00[29]=0; G00[45]=0; G00[61]=0;
  G00[14]=1; G00[30]=0; G00[46]=2; G00[62]=1;
  G00[15]=0; G00[31]=0; G00[47]=1; G00[63]=1;
  
  G01[0]=0; G01[16]=0; G01[32]=1; G01[48]=1;
  G01[1]=1; G01[17]=0; G01[33]=2; G01[49]=1;
  G01[2]=0; G01[18]=0; G01[34]=0; G01[50]=0;
  G01[3]=1; G01[19]=0; G01[35]=1; G01[51]=0;
  G01[4]=0; G01[20]=1; G01[36]=1; G01[52]=2;
  G01[5]=0; G01[21]=0; G01[37]=1; G01[53]=1;
  G01[6]=0; G01[22]=1; G01[38]=0; G01[54]=1;
  G01[7]=0; G01[23]=0; G01[39]=0; G01[55]=0;
  G01[8]=0; G01[24]=0; G01[40]=0; G01[56]=0;
  G01[9]=1; G01[25]=0; G01[41]=1; G01[57]=0;
  G01[10]=1; G01[26]=1; G01[42]=0; G01[58]=0;
  G01[11]=2; G01[27]=1; G01[43]=1; G01[59]=0;
  G01[12]=0; G01[28]=1; G01[44]=0; G01[60]=1;
  G01[13]=0; G01[29]=0; G01[45]=0; G01[61]=0;
  G01[14]=1; G01[30]=2; G01[46]=0; G01[62]=1;
  G01[15]=1; G01[31]=1; G01[47]=0; G01[63]=0;
  
  G10[0]=1; G10[16]=0; G10[32]=1; G10[48]=0;
  G10[1]=0; G10[17]=0; G10[33]=0; G10[49]=0;
  G10[2]=1; G10[18]=0; G10[34]=2; G10[50]=1;
  G10[3]=0; G10[19]=0; G10[35]=1; G10[51]=1;
  G10[4]=0; G10[20]=0; G10[36]=0; G10[52]=0;
  G10[5]=0; G10[21]=1; G10[37]=0; G10[53]=1;
  G10[6]=0; G10[22]=0; G10[38]=1; G10[54]=1;
  G10[7]=0; G10[23]=1; G10[39]=1; G10[55]=2;
  G10[8]=2; G10[24]=1; G10[40]=1; G10[56]=0;
  G10[9]=1; G10[25]=1; G10[41]=0; G10[57]=0;
  G10[10]=1; G10[26]=0; G10[42]=1; G10[58]=0;
  G10[11]=0; G10[27]=0; G10[43]=0; G10[59]=0;
  G10[12]=1; G10[28]=1; G10[44]=0; G10[60]=0;
  G10[13]=1; G10[29]=2; G10[45]=0; G10[61]=1;
  G10[14]=0; G10[30]=0; G10[46]=0; G10[62]=0;
  G10[15]=0; G10[31]=1; G10[47]=0; G10[63]=1;
  
  G11[0]=0; G11[16]=1; G11[32]=0; G11[48]=1;
  G11[1]=0; G11[17]=0; G11[33]=0; G11[49]=0;
  G11[2]=1; G11[18]=2; G11[34]=0; G11[50]=1;
  G11[3]=1; G11[19]=1; G11[35]=0; G11[51]=0;
  G11[4]=0; G11[20]=0; G11[36]=0; G11[52]=0;
  G11[5]=1; G11[21]=0; G11[37]=1; G11[53]=0;
  G11[6]=1; G11[22]=1; G11[38]=0; G11[54]=0;
  G11[7]=2; G11[23]=1; G11[39]=1; G11[55]=0;
  G11[8]=0; G11[24]=1; G11[40]=1; G11[56]=2;
  G11[9]=0; G11[25]=0; G11[41]=1; G11[57]=1;
  G11[10]=0; G11[26]=1; G11[42]=0; G11[58]=1;
  G11[11]=0; G11[27]=0; G11[43]=0; G11[59]=0;
  G11[12]=0; G11[28]=0; G11[44]=1; G11[60]=1;
  G11[13]=1; G11[29]=0; G11[45]=2; G11[61]=1;
  G11[14]=0; G11[30]=0; G11[46]=0; G11[62]=0;
  G11[15]=1; G11[31]=0; G11[47]=1; G11[63]=0;
  
  rf_3pt(I1,*p1,I2,*p2,I3,*p3,n,*ntot,H21,G00,G01,G10,G11,&theta12[4],&theta23[4],&theta13[4],&log_like[4]);
  
  /* assignment 2-2 */
  G00[0]=1; G00[16]=1; G00[32]=0; G00[48]=0;
  G00[1]=2; G00[17]=1; G00[33]=1; G00[49]=0;
  G00[2]=0; G00[18]=0; G00[34]=0; G00[50]=0;
  G00[3]=1; G00[19]=0; G00[35]=1; G00[51]=0;
  G00[4]=1; G00[20]=2; G00[36]=0; G00[52]=1;
  G00[5]=1; G00[21]=1; G00[37]=0; G00[53]=0;
  G00[6]=0; G00[22]=1; G00[38]=0; G00[54]=1;
  G00[7]=0; G00[23]=0; G00[39]=0; G00[55]=0;
  G00[8]=0; G00[24]=0; G00[40]=0; G00[56]=0;
  G00[9]=1; G00[25]=0; G00[41]=1; G00[57]=0;
  G00[10]=0; G00[26]=0; G00[42]=1; G00[58]=1;
  G00[11]=1; G00[27]=0; G00[43]=2; G00[59]=1;
  G00[12]=0; G00[28]=1; G00[44]=0; G00[60]=1;
  G00[13]=0; G00[29]=0; G00[45]=0; G00[61]=0;
  G00[14]=0; G00[30]=1; G00[46]=1; G00[62]=2;
  G00[15]=0; G00[31]=0; G00[47]=1; G00[63]=1;

  G01[0]=0; G01[16]=0; G01[32]=1; G01[48]=1;
  G01[1]=0; G01[17]=1; G01[33]=1; G01[49]=2;
  G01[2]=0; G01[18]=0; G01[34]=0; G01[50]=0;
  G01[3]=0; G01[19]=1; G01[35]=0; G01[51]=1;
  G01[4]=1; G01[20]=0; G01[36]=2; G01[52]=1;
  G01[5]=0; G01[21]=0; G01[37]=1; G01[53]=1;
  G01[6]=1; G01[22]=0; G01[38]=1; G01[54]=0;
  G01[7]=0; G01[23]=0; G01[39]=0; G01[55]=0;
  G01[8]=0; G01[24]=0; G01[40]=0; G01[56]=0;
  G01[9]=0; G01[25]=1; G01[41]=0; G01[57]=1;
  G01[10]=1; G01[26]=1; G01[42]=0; G01[58]=0;
  G01[11]=1; G01[27]=2; G01[43]=0; G01[59]=1;
  G01[12]=1; G01[28]=0; G01[44]=1; G01[60]=0;
  G01[13]=0; G01[29]=0; G01[45]=0; G01[61]=0;
  G01[14]=2; G01[30]=1; G01[46]=1; G01[62]=0;
  G01[15]=1; G01[31]=1; G01[47]=0; G01[63]=0;

  G10[0]=0; G10[16]=1; G10[32]=0; G10[48]=1;
  G10[1]=0; G10[17]=0; G10[33]=0; G10[49]=0;
  G10[2]=0; G10[18]=1; G10[34]=1; G10[50]=2;
  G10[3]=0; G10[19]=0; G10[35]=1; G10[51]=1;
  G10[4]=0; G10[20]=0; G10[36]=0; G10[52]=0;
  G10[5]=1; G10[21]=0; G10[37]=1; G10[53]=0;
  G10[6]=0; G10[22]=0; G10[38]=1; G10[54]=1;
  G10[7]=1; G10[23]=0; G10[39]=2; G10[55]=1;
  G10[8]=1; G10[24]=2; G10[40]=0; G10[56]=1;
  G10[9]=1; G10[25]=1; G10[41]=0; G10[57]=0;
  G10[10]=0; G10[26]=1; G10[42]=0; G10[58]=1;
  G10[11]=0; G10[27]=0; G10[43]=0; G10[59]=0;
  G10[12]=1; G10[28]=1; G10[44]=0; G10[60]=0;
  G10[13]=2; G10[29]=1; G10[45]=1; G10[61]=0;
  G10[14]=0; G10[30]=0; G10[46]=0; G10[62]=0;
  G10[15]=1; G10[31]=0; G10[47]=1; G10[63]=0;

  G11[0]=1; G11[16]=0; G11[32]=1; G11[48]=0;
  G11[1]=0; G11[17]=0; G11[33]=0; G11[49]=0;
  G11[2]=2; G11[18]=1; G11[34]=1; G11[50]=0;
  G11[3]=1; G11[19]=1; G11[35]=0; G11[51]=0;
  G11[4]=0; G11[20]=0; G11[36]=0; G11[52]=0;
  G11[5]=0; G11[21]=1; G11[37]=0; G11[53]=1;
  G11[6]=1; G11[22]=1; G11[38]=0; G11[54]=0;
  G11[7]=1; G11[23]=2; G11[39]=0; G11[55]=1;
  G11[8]=1; G11[24]=0; G11[40]=2; G11[56]=1;
  G11[9]=0; G11[25]=0; G11[41]=1; G11[57]=1;
  G11[10]=1; G11[26]=0; G11[42]=1; G11[58]=0;
  G11[11]=0; G11[27]=0; G11[43]=0; G11[59]=0;
  G11[12]=0; G11[28]=0; G11[44]=1; G11[60]=1;
  G11[13]=0; G11[29]=1; G11[45]=1; G11[61]=2;
  G11[14]=0; G11[30]=0; G11[46]=0; G11[62]=0;
  G11[15]=0; G11[31]=1; G11[47]=0; G11[63]=1;
  
  rf_3pt(I1,*p1,I2,*p2,I3,*p3,n,*ntot,H22,G00,G01,G10,G11,&theta12[5],&theta23[5],&theta13[5],&log_like[5]);
  
  /* assignment 2-3 */
  G00[0]=0; G00[16]=0; G00[32]=1; G00[48]=1;
  G00[1]=0; G00[17]=1; G00[33]=1; G00[49]=2;
  G00[2]=0; G00[18]=0; G00[34]=0; G00[50]=0;
  G00[3]=0; G00[19]=1; G00[35]=0; G00[51]=1;
  G00[4]=1; G00[20]=0; G00[36]=2; G00[52]=1;
  G00[5]=0; G00[21]=0; G00[37]=1; G00[53]=1;
  G00[6]=1; G00[22]=0; G00[38]=1; G00[54]=0;
  G00[7]=0; G00[23]=0; G00[39]=0; G00[55]=0;
  G00[8]=0; G00[24]=0; G00[40]=0; G00[56]=0;
  G00[9]=0; G00[25]=1; G00[41]=0; G00[57]=1;
  G00[10]=1; G00[26]=1; G00[42]=0; G00[58]=0;
  G00[11]=1; G00[27]=2; G00[43]=0; G00[59]=1;
  G00[12]=1; G00[28]=0; G00[44]=1; G00[60]=0;
  G00[13]=0; G00[29]=0; G00[45]=0; G00[61]=0;
  G00[14]=2; G00[30]=1; G00[46]=1; G00[62]=0;
  G00[15]=1; G00[31]=1; G00[47]=0; G00[63]=0;

  G01[0]=1; G01[16]=1; G01[32]=0; G01[48]=0;
  G01[1]=2; G01[17]=1; G01[33]=1; G01[49]=0;
  G01[2]=0; G01[18]=0; G01[34]=0; G01[50]=0;
  G01[3]=1; G01[19]=0; G01[35]=1; G01[51]=0;
  G01[4]=1; G01[20]=2; G01[36]=0; G01[52]=1;
  G01[5]=1; G01[21]=1; G01[37]=0; G01[53]=0;
  G01[6]=0; G01[22]=1; G01[38]=0; G01[54]=1;
  G01[7]=0; G01[23]=0; G01[39]=0; G01[55]=0;
  G01[8]=0; G01[24]=0; G01[40]=0; G01[56]=0;
  G01[9]=1; G01[25]=0; G01[41]=1; G01[57]=0;
  G01[10]=0; G01[26]=0; G01[42]=1; G01[58]=1;
  G01[11]=1; G01[27]=0; G01[43]=2; G01[59]=1;
  G01[12]=0; G01[28]=1; G01[44]=0; G01[60]=1;
  G01[13]=0; G01[29]=0; G01[45]=0; G01[61]=0;
  G01[14]=0; G01[30]=1; G01[46]=1; G01[62]=2;
  G01[15]=0; G01[31]=0; G01[47]=1; G01[63]=1;

  G10[0]=1; G10[16]=0; G10[32]=1; G10[48]=0;
  G10[1]=0; G10[17]=0; G10[33]=0; G10[49]=0;
  G10[2]=2; G10[18]=1; G10[34]=1; G10[50]=0;
  G10[3]=1; G10[19]=1; G10[35]=0; G10[51]=0;
  G10[4]=0; G10[20]=0; G10[36]=0; G10[52]=0;
  G10[5]=0; G10[21]=1; G10[37]=0; G10[53]=1;
  G10[6]=1; G10[22]=1; G10[38]=0; G10[54]=0;
  G10[7]=1; G10[23]=2; G10[39]=0; G10[55]=1;
  G10[8]=1; G10[24]=0; G10[40]=2; G10[56]=1;
  G10[9]=0; G10[25]=0; G10[41]=1; G10[57]=1;
  G10[10]=1; G10[26]=0; G10[42]=1; G10[58]=0;
  G10[11]=0; G10[27]=0; G10[43]=0; G10[59]=0;
  G10[12]=0; G10[28]=0; G10[44]=1; G10[60]=1;
  G10[13]=0; G10[29]=1; G10[45]=1; G10[61]=2;
  G10[14]=0; G10[30]=0; G10[46]=0; G10[62]=0;
  G10[15]=0; G10[31]=1; G10[47]=0; G10[63]=1;

  G11[0]=0; G11[16]=1; G11[32]=0; G11[48]=1;
  G11[1]=0; G11[17]=0; G11[33]=0; G11[49]=0;
  G11[2]=0; G11[18]=1; G11[34]=1; G11[50]=2;
  G11[3]=0; G11[19]=0; G11[35]=1; G11[51]=1;
  G11[4]=0; G11[20]=0; G11[36]=0; G11[52]=0;
  G11[5]=1; G11[21]=0; G11[37]=1; G11[53]=0;
  G11[6]=0; G11[22]=0; G11[38]=1; G11[54]=1;
  G11[7]=1; G11[23]=0; G11[39]=2; G11[55]=1;
  G11[8]=1; G11[24]=2; G11[40]=0; G11[56]=1;
  G11[9]=1; G11[25]=1; G11[41]=0; G11[57]=0;
  G11[10]=0; G11[26]=1; G11[42]=0; G11[58]=1;
  G11[11]=0; G11[27]=0; G11[43]=0; G11[59]=0;
  G11[12]=1; G11[28]=1; G11[44]=0; G11[60]=0;
  G11[13]=2; G11[29]=1; G11[45]=1; G11[61]=0;
  G11[14]=0; G11[30]=0; G11[46]=0; G11[62]=0;
  G11[15]=1; G11[31]=0; G11[47]=1; G11[63]=0;
  
  rf_3pt(I1,*p1,I2,*p2,I3,*p3,n,*ntot,H23,G00,G01,G10,G11,&theta12[6],&theta23[6],&theta13[6],&log_like[6]);
  
  /* assignment 2-4 */
  G00[0]=0; G00[16]=0; G00[32]=1; G00[48]=1;
  G00[1]=1; G00[17]=0; G00[33]=2; G00[49]=1;
  G00[2]=0; G00[18]=0; G00[34]=0; G00[50]=0;
  G00[3]=1; G00[19]=0; G00[35]=1; G00[51]=0;
  G00[4]=0; G00[20]=1; G00[36]=1; G00[52]=2;
  G00[5]=0; G00[21]=0; G00[37]=1; G00[53]=1;
  G00[6]=0; G00[22]=1; G00[38]=0; G00[54]=1;
  G00[7]=0; G00[23]=0; G00[39]=0; G00[55]=0;
  G00[8]=0; G00[24]=0; G00[40]=0; G00[56]=0;
  G00[9]=1; G00[25]=0; G00[41]=1; G00[57]=0;
  G00[10]=1; G00[26]=1; G00[42]=0; G00[58]=0;
  G00[11]=2; G00[27]=1; G00[43]=1; G00[59]=0;
  G00[12]=0; G00[28]=1; G00[44]=0; G00[60]=1;
  G00[13]=0; G00[29]=0; G00[45]=0; G00[61]=0;
  G00[14]=1; G00[30]=2; G00[46]=0; G00[62]=1;
  G00[15]=1; G00[31]=1; G00[47]=0; G00[63]=0;

  G01[0]=1; G01[16]=1; G01[32]=0; G01[48]=0;
  G01[1]=1; G01[17]=2; G01[33]=0; G01[49]=1;
  G01[2]=0; G01[18]=0; G01[34]=0; G01[50]=0;
  G01[3]=0; G01[19]=1; G01[35]=0; G01[51]=1;
  G01[4]=2; G01[20]=1; G01[36]=1; G01[52]=0;
  G01[5]=1; G01[21]=1; G01[37]=0; G01[53]=0;
  G01[6]=1; G01[22]=0; G01[38]=1; G01[54]=0;
  G01[7]=0; G01[23]=0; G01[39]=0; G01[55]=0;
  G01[8]=0; G01[24]=0; G01[40]=0; G01[56]=0;
  G01[9]=0; G01[25]=1; G01[41]=0; G01[57]=1;
  G01[10]=0; G01[26]=0; G01[42]=1; G01[58]=1;
  G01[11]=0; G01[27]=1; G01[43]=1; G01[59]=2;
  G01[12]=1; G01[28]=0; G01[44]=1; G01[60]=0;
  G01[13]=0; G01[29]=0; G01[45]=0; G01[61]=0;
  G01[14]=1; G01[30]=0; G01[46]=2; G01[62]=1;
  G01[15]=0; G01[31]=0; G01[47]=1; G01[63]=1;

  G10[0]=0; G10[16]=1; G10[32]=0; G10[48]=1;
  G10[1]=0; G10[17]=0; G10[33]=0; G10[49]=0;
  G10[2]=1; G10[18]=2; G10[34]=0; G10[50]=1;
  G10[3]=1; G10[19]=1; G10[35]=0; G10[51]=0;
  G10[4]=0; G10[20]=0; G10[36]=0; G10[52]=0;
  G10[5]=1; G10[21]=0; G10[37]=1; G10[53]=0;
  G10[6]=1; G10[22]=1; G10[38]=0; G10[54]=0;
  G10[7]=2; G10[23]=1; G10[39]=1; G10[55]=0;
  G10[8]=0; G10[24]=1; G10[40]=1; G10[56]=2;
  G10[9]=0; G10[25]=0; G10[41]=1; G10[57]=1;
  G10[10]=0; G10[26]=1; G10[42]=0; G10[58]=1;
  G10[11]=0; G10[27]=0; G10[43]=0; G10[59]=0;
  G10[12]=0; G10[28]=0; G10[44]=1; G10[60]=1;
  G10[13]=1; G10[29]=0; G10[45]=2; G10[61]=1;
  G10[14]=0; G10[30]=0; G10[46]=0; G10[62]=0;
  G10[15]=1; G10[31]=0; G10[47]=1; G10[63]=0;

  G11[0]=1; G11[16]=0; G11[32]=1; G11[48]=0;
  G11[1]=0; G11[17]=0; G11[33]=0; G11[49]=0;
  G11[2]=1; G11[18]=0; G11[34]=2; G11[50]=1;
  G11[3]=0; G11[19]=0; G11[35]=1; G11[51]=1;
  G11[4]=0; G11[20]=0; G11[36]=0; G11[52]=0;
  G11[5]=0; G11[21]=1; G11[37]=0; G11[53]=1;
  G11[6]=0; G11[22]=0; G11[38]=1; G11[54]=1;
  G11[7]=0; G11[23]=1; G11[39]=1; G11[55]=2;
  G11[8]=2; G11[24]=1; G11[40]=1; G11[56]=0;
  G11[9]=1; G11[25]=1; G11[41]=0; G11[57]=0;
  G11[10]=1; G11[26]=0; G11[42]=1; G11[58]=0;
  G11[11]=0; G11[27]=0; G11[43]=0; G11[59]=0;
  G11[12]=1; G11[28]=1; G11[44]=0; G11[60]=0;
  G11[13]=1; G11[29]=2; G11[45]=0; G11[61]=1;
  G11[14]=0; G11[30]=0; G11[46]=0; G11[62]=0;
  G11[15]=0; G11[31]=1; G11[47]=0; G11[63]=1;
  
  rf_3pt(I1,*p1,I2,*p2,I3,*p3,n,*ntot,H24,G00,G01,G10,G11,&theta12[7],&theta23[7],&theta13[7],&log_like[7]);
     
  /* assignment 3-1 */
  G00[0]=1; G00[16]=0; G00[32]=1; G00[48]=0;
  G00[1]=0; G00[17]=0; G00[33]=0; G00[49]=0;
  G00[2]=1; G00[18]=0; G00[34]=2; G00[50]=1;
  G00[3]=0; G00[19]=0; G00[35]=1; G00[51]=1;
  G00[4]=0; G00[20]=0; G00[36]=0; G00[52]=0;
  G00[5]=0; G00[21]=1; G00[37]=0; G00[53]=1;
  G00[6]=0; G00[22]=0; G00[38]=1; G00[54]=1;
  G00[7]=0; G00[23]=1; G00[39]=1; G00[55]=2;
  G00[8]=2; G00[24]=1; G00[40]=1; G00[56]=0;
  G00[9]=1; G00[25]=1; G00[41]=0; G00[57]=0;
  G00[10]=1; G00[26]=0; G00[42]=1; G00[58]=0;
  G00[11]=0; G00[27]=0; G00[43]=0; G00[59]=0;
  G00[12]=1; G00[28]=1; G00[44]=0; G00[60]=0;
  G00[13]=1; G00[29]=2; G00[45]=0; G00[61]=1;
  G00[14]=0; G00[30]=0; G00[46]=0; G00[62]=0;
  G00[15]=0; G00[31]=1; G00[47]=0; G00[63]=1;

  G01[0]=0; G01[16]=1; G01[32]=0; G01[48]=1;
  G01[1]=0; G01[17]=0; G01[33]=0; G01[49]=0;
  G01[2]=1; G01[18]=2; G01[34]=0; G01[50]=1;
  G01[3]=1; G01[19]=1; G01[35]=0; G01[51]=0;
  G01[4]=0; G01[20]=0; G01[36]=0; G01[52]=0;
  G01[5]=1; G01[21]=0; G01[37]=1; G01[53]=0;
  G01[6]=1; G01[22]=1; G01[38]=0; G01[54]=0;
  G01[7]=2; G01[23]=1; G01[39]=1; G01[55]=0;
  G01[8]=0; G01[24]=1; G01[40]=1; G01[56]=2;
  G01[9]=0; G01[25]=0; G01[41]=1; G01[57]=1;
  G01[10]=0; G01[26]=1; G01[42]=0; G01[58]=1;
  G01[11]=0; G01[27]=0; G01[43]=0; G01[59]=0;
  G01[12]=0; G01[28]=0; G01[44]=1; G01[60]=1;
  G01[13]=1; G01[29]=0; G01[45]=2; G01[61]=1;
  G01[14]=0; G01[30]=0; G01[46]=0; G01[62]=0;
  G01[15]=1; G01[31]=0; G01[47]=1; G01[63]=0;

  G10[0]=1; G10[16]=1; G10[32]=0; G10[48]=0;
  G10[1]=1; G10[17]=2; G10[33]=0; G10[49]=1;
  G10[2]=0; G10[18]=0; G10[34]=0; G10[50]=0;
  G10[3]=0; G10[19]=1; G10[35]=0; G10[51]=1;
  G10[4]=2; G10[20]=1; G10[36]=1; G10[52]=0;
  G10[5]=1; G10[21]=1; G10[37]=0; G10[53]=0;
  G10[6]=1; G10[22]=0; G10[38]=1; G10[54]=0;
  G10[7]=0; G10[23]=0; G10[39]=0; G10[55]=0;
  G10[8]=0; G10[24]=0; G10[40]=0; G10[56]=0;
  G10[9]=0; G10[25]=1; G10[41]=0; G10[57]=1;
  G10[10]=0; G10[26]=0; G10[42]=1; G10[58]=1;
  G10[11]=0; G10[27]=1; G10[43]=1; G10[59]=2;
  G10[12]=1; G10[28]=0; G10[44]=1; G10[60]=0;
  G10[13]=0; G10[29]=0; G10[45]=0; G10[61]=0;
  G10[14]=1; G10[30]=0; G10[46]=2; G10[62]=1;
  G10[15]=0; G10[31]=0; G10[47]=1; G10[63]=1;

  G11[0]=0; G11[16]=0; G11[32]=1; G11[48]=1;
  G11[1]=1; G11[17]=0; G11[33]=2; G11[49]=1;
  G11[2]=0; G11[18]=0; G11[34]=0; G11[50]=0;
  G11[3]=1; G11[19]=0; G11[35]=1; G11[51]=0;
  G11[4]=0; G11[20]=1; G11[36]=1; G11[52]=2;
  G11[5]=0; G11[21]=0; G11[37]=1; G11[53]=1;
  G11[6]=0; G11[22]=1; G11[38]=0; G11[54]=1;
  G11[7]=0; G11[23]=0; G11[39]=0; G11[55]=0;
  G11[8]=0; G11[24]=0; G11[40]=0; G11[56]=0;
  G11[9]=1; G11[25]=0; G11[41]=1; G11[57]=0;
  G11[10]=1; G11[26]=1; G11[42]=0; G11[58]=0;
  G11[11]=2; G11[27]=1; G11[43]=1; G11[59]=0;
  G11[12]=0; G11[28]=1; G11[44]=0; G11[60]=1;
  G11[13]=0; G11[29]=0; G11[45]=0; G11[61]=0;
  G11[14]=1; G11[30]=2; G11[46]=0; G11[62]=1;
  G11[15]=1; G11[31]=1; G11[47]=0; G11[63]=0;
  
  rf_3pt(I1,*p1,I2,*p2,I3,*p3,n,*ntot,H31,G00,G01,G10,G11,&theta12[8],&theta23[8],&theta13[8],&log_like[8]);
     
  /* assignment 3-2 */
  G00[0]=0; G00[16]=1; G00[32]=0; G00[48]=1;
  G00[1]=0; G00[17]=0; G00[33]=0; G00[49]=0;
  G00[2]=0; G00[18]=1; G00[34]=1; G00[50]=2;
  G00[3]=0; G00[19]=0; G00[35]=1; G00[51]=1;
  G00[4]=0; G00[20]=0; G00[36]=0; G00[52]=0;
  G00[5]=1; G00[21]=0; G00[37]=1; G00[53]=0;
  G00[6]=0; G00[22]=0; G00[38]=1; G00[54]=1;
  G00[7]=1; G00[23]=0; G00[39]=2; G00[55]=1;
  G00[8]=1; G00[24]=2; G00[40]=0; G00[56]=1;
  G00[9]=1; G00[25]=1; G00[41]=0; G00[57]=0;
  G00[10]=0; G00[26]=1; G00[42]=0; G00[58]=1;
  G00[11]=0; G00[27]=0; G00[43]=0; G00[59]=0;
  G00[12]=1; G00[28]=1; G00[44]=0; G00[60]=0;
  G00[13]=2; G00[29]=1; G00[45]=1; G00[61]=0;
  G00[14]=0; G00[30]=0; G00[46]=0; G00[62]=0;
  G00[15]=1; G00[31]=0; G00[47]=1; G00[63]=0;

  G01[0]=1; G01[16]=0; G01[32]=1; G01[48]=0;
  G01[1]=0; G01[17]=0; G01[33]=0; G01[49]=0;
  G01[2]=2; G01[18]=1; G01[34]=1; G01[50]=0;
  G01[3]=1; G01[19]=1; G01[35]=0; G01[51]=0;
  G01[4]=0; G01[20]=0; G01[36]=0; G01[52]=0;
  G01[5]=0; G01[21]=1; G01[37]=0; G01[53]=1;
  G01[6]=1; G01[22]=1; G01[38]=0; G01[54]=0;
  G01[7]=1; G01[23]=2; G01[39]=0; G01[55]=1;
  G01[8]=1; G01[24]=0; G01[40]=2; G01[56]=1;
  G01[9]=0; G01[25]=0; G01[41]=1; G01[57]=1;
  G01[10]=1; G01[26]=0; G01[42]=1; G01[58]=0;
  G01[11]=0; G01[27]=0; G01[43]=0; G01[59]=0;
  G01[12]=0; G01[28]=0; G01[44]=1; G01[60]=1;
  G01[13]=0; G01[29]=1; G01[45]=1; G01[61]=2;
  G01[14]=0; G01[30]=0; G01[46]=0; G01[62]=0;
  G01[15]=0; G01[31]=1; G01[47]=0; G01[63]=1;

  G10[0]=1; G10[16]=1; G10[32]=0; G10[48]=0;
  G10[1]=2; G10[17]=1; G10[33]=1; G10[49]=0;
  G10[2]=0; G10[18]=0; G10[34]=0; G10[50]=0;
  G10[3]=1; G10[19]=0; G10[35]=1; G10[51]=0;
  G10[4]=1; G10[20]=2; G10[36]=0; G10[52]=1;
  G10[5]=1; G10[21]=1; G10[37]=0; G10[53]=0;
  G10[6]=0; G10[22]=1; G10[38]=0; G10[54]=1;
  G10[7]=0; G10[23]=0; G10[39]=0; G10[55]=0;
  G10[8]=0; G10[24]=0; G10[40]=0; G10[56]=0;
  G10[9]=1; G10[25]=0; G10[41]=1; G10[57]=0;
  G10[10]=0; G10[26]=0; G10[42]=1; G10[58]=1;
  G10[11]=1; G10[27]=0; G10[43]=2; G10[59]=1;
  G10[12]=0; G10[28]=1; G10[44]=0; G10[60]=1;
  G10[13]=0; G10[29]=0; G10[45]=0; G10[61]=0;
  G10[14]=0; G10[30]=1; G10[46]=1; G10[62]=2;
  G10[15]=0; G10[31]=0; G10[47]=1; G10[63]=1;

  G11[0]=0; G11[16]=0; G11[32]=1; G11[48]=1;
  G11[1]=0; G11[17]=1; G11[33]=1; G11[49]=2;
  G11[2]=0; G11[18]=0; G11[34]=0; G11[50]=0;
  G11[3]=0; G11[19]=1; G11[35]=0; G11[51]=1;
  G11[4]=1; G11[20]=0; G11[36]=2; G11[52]=1;
  G11[5]=0; G11[21]=0; G11[37]=1; G11[53]=1;
  G11[6]=1; G11[22]=0; G11[38]=1; G11[54]=0;
  G11[7]=0; G11[23]=0; G11[39]=0; G11[55]=0;
  G11[8]=0; G11[24]=0; G11[40]=0; G11[56]=0;
  G11[9]=0; G11[25]=1; G11[41]=0; G11[57]=1;
  G11[10]=1; G11[26]=1; G11[42]=0; G11[58]=0;
  G11[11]=1; G11[27]=2; G11[43]=0; G11[59]=1;
  G11[12]=1; G11[28]=0; G11[44]=1; G11[60]=0;
  G11[13]=0; G11[29]=0; G11[45]=0; G11[61]=0;
  G11[14]=2; G11[30]=1; G11[46]=1; G11[62]=0;
  G11[15]=1; G11[31]=1; G11[47]=0; G11[63]=0;
  
  rf_3pt(I1,*p1,I2,*p2,I3,*p3,n,*ntot,H32,G00,G01,G10,G11,&theta12[9],&theta23[9],&theta13[9],&log_like[9]);
     
  /* assignment 3-3 */
  G00[0]=1; G00[16]=0; G00[32]=1; G00[48]=0;
  G00[1]=0; G00[17]=0; G00[33]=0; G00[49]=0;
  G00[2]=2; G00[18]=1; G00[34]=1; G00[50]=0;
  G00[3]=1; G00[19]=1; G00[35]=0; G00[51]=0;
  G00[4]=0; G00[20]=0; G00[36]=0; G00[52]=0;
  G00[5]=0; G00[21]=1; G00[37]=0; G00[53]=1;
  G00[6]=1; G00[22]=1; G00[38]=0; G00[54]=0;
  G00[7]=1; G00[23]=2; G00[39]=0; G00[55]=1;
  G00[8]=1; G00[24]=0; G00[40]=2; G00[56]=1;
  G00[9]=0; G00[25]=0; G00[41]=1; G00[57]=1;
  G00[10]=1; G00[26]=0; G00[42]=1; G00[58]=0;
  G00[11]=0; G00[27]=0; G00[43]=0; G00[59]=0;
  G00[12]=0; G00[28]=0; G00[44]=1; G00[60]=1;
  G00[13]=0; G00[29]=1; G00[45]=1; G00[61]=2;
  G00[14]=0; G00[30]=0; G00[46]=0; G00[62]=0;
  G00[15]=0; G00[31]=1; G00[47]=0; G00[63]=1;

  G01[0]=0; G01[16]=1; G01[32]=0; G01[48]=1;
  G01[1]=0; G01[17]=0; G01[33]=0; G01[49]=0;
  G01[2]=0; G01[18]=1; G01[34]=1; G01[50]=2;
  G01[3]=0; G01[19]=0; G01[35]=1; G01[51]=1;
  G01[4]=0; G01[20]=0; G01[36]=0; G01[52]=0;
  G01[5]=1; G01[21]=0; G01[37]=1; G01[53]=0;
  G01[6]=0; G01[22]=0; G01[38]=1; G01[54]=1;
  G01[7]=1; G01[23]=0; G01[39]=2; G01[55]=1;
  G01[8]=1; G01[24]=2; G01[40]=0; G01[56]=1;
  G01[9]=1; G01[25]=1; G01[41]=0; G01[57]=0;
  G01[10]=0; G01[26]=1; G01[42]=0; G01[58]=1;
  G01[11]=0; G01[27]=0; G01[43]=0; G01[59]=0;
  G01[12]=1; G01[28]=1; G01[44]=0; G01[60]=0;
  G01[13]=2; G01[29]=1; G01[45]=1; G01[61]=0;
  G01[14]=0; G01[30]=0; G01[46]=0; G01[62]=0;
  G01[15]=1; G01[31]=0; G01[47]=1; G01[63]=0;

  G10[0]=0; G10[16]=0; G10[32]=1; G10[48]=1;
  G10[1]=0; G10[17]=1; G10[33]=1; G10[49]=2;
  G10[2]=0; G10[18]=0; G10[34]=0; G10[50]=0;
  G10[3]=0; G10[19]=1; G10[35]=0; G10[51]=1;
  G10[4]=1; G10[20]=0; G10[36]=2; G10[52]=1;
  G10[5]=0; G10[21]=0; G10[37]=1; G10[53]=1;
  G10[6]=1; G10[22]=0; G10[38]=1; G10[54]=0;
  G10[7]=0; G10[23]=0; G10[39]=0; G10[55]=0;
  G10[8]=0; G10[24]=0; G10[40]=0; G10[56]=0;
  G10[9]=0; G10[25]=1; G10[41]=0; G10[57]=1;
  G10[10]=1; G10[26]=1; G10[42]=0; G10[58]=0;
  G10[11]=1; G10[27]=2; G10[43]=0; G10[59]=1;
  G10[12]=1; G10[28]=0; G10[44]=1; G10[60]=0;
  G10[13]=0; G10[29]=0; G10[45]=0; G10[61]=0;
  G10[14]=2; G10[30]=1; G10[46]=1; G10[62]=0;
  G10[15]=1; G10[31]=1; G10[47]=0; G10[63]=0;

  G11[0]=1; G11[16]=1; G11[32]=0; G11[48]=0;
  G11[1]=2; G11[17]=1; G11[33]=1; G11[49]=0;
  G11[2]=0; G11[18]=0; G11[34]=0; G11[50]=0;
  G11[3]=1; G11[19]=0; G11[35]=1; G11[51]=0;
  G11[4]=1; G11[20]=2; G11[36]=0; G11[52]=1;
  G11[5]=1; G11[21]=1; G11[37]=0; G11[53]=0;
  G11[6]=0; G11[22]=1; G11[38]=0; G11[54]=1;
  G11[7]=0; G11[23]=0; G11[39]=0; G11[55]=0;
  G11[8]=0; G11[24]=0; G11[40]=0; G11[56]=0;
  G11[9]=1; G11[25]=0; G11[41]=1; G11[57]=0;
  G11[10]=0; G11[26]=0; G11[42]=1; G11[58]=1;
  G11[11]=1; G11[27]=0; G11[43]=2; G11[59]=1;
  G11[12]=0; G11[28]=1; G11[44]=0; G11[60]=1;
  G11[13]=0; G11[29]=0; G11[45]=0; G11[61]=0;
  G11[14]=0; G11[30]=1; G11[46]=1; G11[62]=2;
  G11[15]=0; G11[31]=0; G11[47]=1; G11[63]=1;
  
  rf_3pt(I1,*p1,I2,*p2,I3,*p3,n,*ntot,H33,G00,G01,G10,G11,&theta12[10],&theta23[10],&theta13[10],&log_like[10]);
    
  /* assignment 3-4 */
  G00[0]=0; G00[16]=1; G00[32]=0; G00[48]=1;
  G00[1]=0; G00[17]=0; G00[33]=0; G00[49]=0;
  G00[2]=1; G00[18]=2; G00[34]=0; G00[50]=1;
  G00[3]=1; G00[19]=1; G00[35]=0; G00[51]=0;
  G00[4]=0; G00[20]=0; G00[36]=0; G00[52]=0;
  G00[5]=1; G00[21]=0; G00[37]=1; G00[53]=0;
  G00[6]=1; G00[22]=1; G00[38]=0; G00[54]=0;
  G00[7]=2; G00[23]=1; G00[39]=1; G00[55]=0;
  G00[8]=0; G00[24]=1; G00[40]=1; G00[56]=2;
  G00[9]=0; G00[25]=0; G00[41]=1; G00[57]=1;
  G00[10]=0; G00[26]=1; G00[42]=0; G00[58]=1;
  G00[11]=0; G00[27]=0; G00[43]=0; G00[59]=0;
  G00[12]=0; G00[28]=0; G00[44]=1; G00[60]=1;
  G00[13]=1; G00[29]=0; G00[45]=2; G00[61]=1;
  G00[14]=0; G00[30]=0; G00[46]=0; G00[62]=0;
  G00[15]=1; G00[31]=0; G00[47]=1; G00[63]=0;

  G01[0]=1; G01[16]=0; G01[32]=1; G01[48]=0;
  G01[1]=0; G01[17]=0; G01[33]=0; G01[49]=0;
  G01[2]=1; G01[18]=0; G01[34]=2; G01[50]=1;
  G01[3]=0; G01[19]=0; G01[35]=1; G01[51]=1;
  G01[4]=0; G01[20]=0; G01[36]=0; G01[52]=0;
  G01[5]=0; G01[21]=1; G01[37]=0; G01[53]=1;
  G01[6]=0; G01[22]=0; G01[38]=1; G01[54]=1;
  G01[7]=0; G01[23]=1; G01[39]=1; G01[55]=2;
  G01[8]=2; G01[24]=1; G01[40]=1; G01[56]=0;
  G01[9]=1; G01[25]=1; G01[41]=0; G01[57]=0;
  G01[10]=1; G01[26]=0; G01[42]=1; G01[58]=0;
  G01[11]=0; G01[27]=0; G01[43]=0; G01[59]=0;
  G01[12]=1; G01[28]=1; G01[44]=0; G01[60]=0;
  G01[13]=1; G01[29]=2; G01[45]=0; G01[61]=1;
  G01[14]=0; G01[30]=0; G01[46]=0; G01[62]=0;
  G01[15]=0; G01[31]=1; G01[47]=0; G01[63]=1;

  G10[0]=0; G10[16]=0; G10[32]=1; G10[48]=1;
  G10[1]=1; G10[17]=0; G10[33]=2; G10[49]=1;
  G10[2]=0; G10[18]=0; G10[34]=0; G10[50]=0;
  G10[3]=1; G10[19]=0; G10[35]=1; G10[51]=0;
  G10[4]=0; G10[20]=1; G10[36]=1; G10[52]=2;
  G10[5]=0; G10[21]=0; G10[37]=1; G10[53]=1;
  G10[6]=0; G10[22]=1; G10[38]=0; G10[54]=1;
  G10[7]=0; G10[23]=0; G10[39]=0; G10[55]=0;
  G10[8]=0; G10[24]=0; G10[40]=0; G10[56]=0;
  G10[9]=1; G10[25]=0; G10[41]=1; G10[57]=0;
  G10[10]=1; G10[26]=1; G10[42]=0; G10[58]=0;
  G10[11]=2; G10[27]=1; G10[43]=1; G10[59]=0;
  G10[12]=0; G10[28]=1; G10[44]=0; G10[60]=1;
  G10[13]=0; G10[29]=0; G10[45]=0; G10[61]=0;
  G10[14]=1; G10[30]=2; G10[46]=0; G10[62]=1;
  G10[15]=1; G10[31]=1; G10[47]=0; G10[63]=0;

  G11[0]=1; G11[16]=1; G11[32]=0; G11[48]=0;
  G11[1]=1; G11[17]=2; G11[33]=0; G11[49]=1;
  G11[2]=0; G11[18]=0; G11[34]=0; G11[50]=0;
  G11[3]=0; G11[19]=1; G11[35]=0; G11[51]=1;
  G11[4]=2; G11[20]=1; G11[36]=1; G11[52]=0;
  G11[5]=1; G11[21]=1; G11[37]=0; G11[53]=0;
  G11[6]=1; G11[22]=0; G11[38]=1; G11[54]=0;
  G11[7]=0; G11[23]=0; G11[39]=0; G11[55]=0;
  G11[8]=0; G11[24]=0; G11[40]=0; G11[56]=0;
  G11[9]=0; G11[25]=1; G11[41]=0; G11[57]=1;
  G11[10]=0; G11[26]=0; G11[42]=1; G11[58]=1;
  G11[11]=0; G11[27]=1; G11[43]=1; G11[59]=2;
  G11[12]=1; G11[28]=0; G11[44]=1; G11[60]=0;
  G11[13]=0; G11[29]=0; G11[45]=0; G11[61]=0;
  G11[14]=1; G11[30]=0; G11[46]=2; G11[62]=1;
  G11[15]=0; G11[31]=0; G11[47]=1; G11[63]=1;
  
  rf_3pt(I1,*p1,I2,*p2,I3,*p3,n,*ntot,H34,G00,G01,G10,G11,&theta12[11],&theta23[11],&theta13[11],&log_like[11]);
  
  /* assignment 4-1 */
  G00[0]=0; G00[16]=0; G00[32]=0; G00[48]=0;
  G00[1]=0; G00[17]=1; G00[33]=0; G00[49]=1;
  G00[2]=0; G00[18]=0; G00[34]=1; G00[50]=1;
  G00[3]=0; G00[19]=1; G00[35]=1; G00[51]=2;
  G00[4]=1; G00[20]=0; G00[36]=1; G00[52]=0;
  G00[5]=0; G00[21]=0; G00[37]=0; G00[53]=0;
  G00[6]=1; G00[22]=0; G00[38]=2; G00[54]=1;
  G00[7]=0; G00[23]=0; G00[39]=1; G00[55]=1;
  G00[8]=1; G00[24]=1; G00[40]=0; G00[56]=0;
  G00[9]=1; G00[25]=2; G00[41]=0; G00[57]=1;
  G00[10]=0; G00[26]=0; G00[42]=0; G00[58]=0;
  G00[11]=0; G00[27]=1; G00[43]=0; G00[59]=1;
  G00[12]=2; G00[28]=1; G00[44]=1; G00[60]=0;
  G00[13]=1; G00[29]=1; G00[45]=0; G00[61]=0;
  G00[14]=1; G00[30]=0; G00[46]=1; G00[62]=0;
  G00[15]=0; G00[31]=0; G00[47]=0; G00[63]=0;

  G01[0]=0; G01[16]=0; G01[32]=0; G01[48]=0;
  G01[1]=1; G01[17]=0; G01[33]=1; G01[49]=0;
  G01[2]=1; G01[18]=1; G01[34]=0; G01[50]=0;
  G01[3]=2; G01[19]=1; G01[35]=1; G01[51]=0;
  G01[4]=0; G01[20]=1; G01[36]=0; G01[52]=1;
  G01[5]=0; G01[21]=0; G01[37]=0; G01[53]=0;
  G01[6]=1; G01[22]=2; G01[38]=0; G01[54]=1;
  G01[7]=1; G01[23]=1; G01[39]=0; G01[55]=0;
  G01[8]=0; G01[24]=0; G01[40]=1; G01[56]=1;
  G01[9]=1; G01[25]=0; G01[41]=2; G01[57]=1;
  G01[10]=0; G01[26]=0; G01[42]=0; G01[58]=0;
  G01[11]=1; G01[27]=0; G01[43]=1; G01[59]=0;
  G01[12]=0; G01[28]=1; G01[44]=1; G01[60]=2;
  G01[13]=0; G01[29]=0; G01[45]=1; G01[61]=1;
  G01[14]=0; G01[30]=1; G01[46]=0; G01[62]=1;
  G01[15]=0; G01[31]=0; G01[47]=0; G01[63]=0;

  G10[0]=2; G10[16]=1; G10[32]=1; G10[48]=0;
  G10[1]=1; G10[17]=1; G10[33]=0; G10[49]=0;
  G10[2]=1; G10[18]=0; G10[34]=1; G10[50]=0;
  G10[3]=0; G10[19]=0; G10[35]=0; G10[51]=0;
  G10[4]=1; G10[20]=1; G10[36]=0; G10[52]=0;
  G10[5]=1; G10[21]=2; G10[37]=0; G10[53]=1;
  G10[6]=0; G10[22]=0; G10[38]=0; G10[54]=0;
  G10[7]=0; G10[23]=1; G10[39]=0; G10[55]=1;
  G10[8]=1; G10[24]=0; G10[40]=1; G10[56]=0;
  G10[9]=0; G10[25]=0; G10[41]=0; G10[57]=0;
  G10[10]=1; G10[26]=0; G10[42]=2; G10[58]=1;
  G10[11]=0; G10[27]=0; G10[43]=1; G10[59]=1;
  G10[12]=0; G10[28]=0; G10[44]=0; G10[60]=0;
  G10[13]=0; G10[29]=1; G10[45]=0; G10[61]=1;
  G10[14]=0; G10[30]=0; G10[46]=1; G10[62]=1;
  G10[15]=0; G10[31]=1; G10[47]=1; G10[63]=2;

  G11[0]=0; G11[16]=1; G11[32]=1; G11[48]=2;
  G11[1]=0; G11[17]=0; G11[33]=1; G11[49]=1;
  G11[2]=0; G11[18]=1; G11[34]=0; G11[50]=1;
  G11[3]=0; G11[19]=0; G11[35]=0; G11[51]=0;
  G11[4]=0; G11[20]=0; G11[36]=1; G11[52]=1;
  G11[5]=1; G11[21]=0; G11[37]=2; G11[53]=1;
  G11[6]=0; G11[22]=0; G11[38]=0; G11[54]=0;
  G11[7]=1; G11[23]=0; G11[39]=1; G11[55]=0;
  G11[8]=0; G11[24]=1; G11[40]=0; G11[56]=1;
  G11[9]=0; G11[25]=0; G11[41]=0; G11[57]=0;
  G11[10]=1; G11[26]=2; G11[42]=0; G11[58]=1;
  G11[11]=1; G11[27]=1; G11[43]=0; G11[59]=0;
  G11[12]=0; G11[28]=0; G11[44]=0; G11[60]=0;
  G11[13]=1; G11[29]=0; G11[45]=1; G11[61]=0;
  G11[14]=1; G11[30]=1; G11[46]=0; G11[62]=0;
  G11[15]=2; G11[31]=1; G11[47]=1; G11[63]=0;
  
  rf_3pt(I1,*p1,I2,*p2,I3,*p3,n,*ntot,H41,G00,G01,G10,G11,&theta12[12],&theta23[12],&theta13[12],&log_like[12]);
  
  /* assignment 4-2 */
  G00[0]=0; G00[16]=0; G00[32]=0; G00[48]=0;
  G00[1]=1; G00[17]=0; G00[33]=1; G00[49]=0;
  G00[2]=0; G00[18]=0; G00[34]=1; G00[50]=1;
  G00[3]=1; G00[19]=0; G00[35]=2; G00[51]=1;
  G00[4]=0; G00[20]=1; G00[36]=0; G00[52]=1;
  G00[5]=0; G00[21]=0; G00[37]=0; G00[53]=0;
  G00[6]=0; G00[22]=1; G00[38]=1; G00[54]=2;
  G00[7]=0; G00[23]=0; G00[39]=1; G00[55]=1;
  G00[8]=1; G00[24]=1; G00[40]=0; G00[56]=0;
  G00[9]=2; G00[25]=1; G00[41]=1; G00[57]=0;
  G00[10]=0; G00[26]=0; G00[42]=0; G00[58]=0;
  G00[11]=1; G00[27]=0; G00[43]=1; G00[59]=0;
  G00[12]=1; G00[28]=2; G00[44]=0; G00[60]=1;
  G00[13]=1; G00[29]=1; G00[45]=0; G00[61]=0;
  G00[14]=0; G00[30]=1; G00[46]=0; G00[62]=1;
  G00[15]=0; G00[31]=0; G00[47]=0; G00[63]=0;

  G01[0]=0; G01[16]=0; G01[32]=0; G01[48]=0;
  G01[1]=0; G01[17]=1; G01[33]=0; G01[49]=1;
  G01[2]=1; G01[18]=1; G01[34]=0; G01[50]=0;
  G01[3]=1; G01[19]=2; G01[35]=0; G01[51]=1;
  G01[4]=1; G01[20]=0; G01[36]=1; G01[52]=0;
  G01[5]=0; G01[21]=0; G01[37]=0; G01[53]=0;
  G01[6]=2; G01[22]=1; G01[38]=1; G01[54]=0;
  G01[7]=1; G01[23]=1; G01[39]=0; G01[55]=0;
  G01[8]=0; G01[24]=0; G01[40]=1; G01[56]=1;
  G01[9]=0; G01[25]=1; G01[41]=1; G01[57]=2;
  G01[10]=0; G01[26]=0; G01[42]=0; G01[58]=0;
  G01[11]=0; G01[27]=1; G01[43]=0; G01[59]=1;
  G01[12]=1; G01[28]=0; G01[44]=2; G01[60]=1;
  G01[13]=0; G01[29]=0; G01[45]=1; G01[61]=1;
  G01[14]=1; G01[30]=0; G01[46]=1; G01[62]=0;
  G01[15]=0; G01[31]=0; G01[47]=0; G01[63]=0;

  G10[0]=1; G10[16]=2; G10[32]=0; G10[48]=1;
  G10[1]=1; G10[17]=1; G10[33]=0; G10[49]=0;
  G10[2]=0; G10[18]=1; G10[34]=0; G10[50]=1;
  G10[3]=0; G10[19]=0; G10[35]=0; G10[51]=0;
  G10[4]=1; G10[20]=1; G10[36]=0; G10[52]=0;
  G10[5]=2; G10[21]=1; G10[37]=1; G10[53]=0;
  G10[6]=0; G10[22]=0; G10[38]=0; G10[54]=0;
  G10[7]=1; G10[23]=0; G10[39]=1; G10[55]=0;
  G10[8]=0; G10[24]=1; G10[40]=0; G10[56]=1;
  G10[9]=0; G10[25]=0; G10[41]=0; G10[57]=0;
  G10[10]=0; G10[26]=1; G10[42]=1; G10[58]=2;
  G10[11]=0; G10[27]=0; G10[43]=1; G10[59]=1;
  G10[12]=0; G10[28]=0; G10[44]=0; G10[60]=0;
  G10[13]=1; G10[29]=0; G10[45]=1; G10[61]=0;
  G10[14]=0; G10[30]=0; G10[46]=1; G10[62]=1;
  G10[15]=1; G10[31]=0; G10[47]=2; G10[63]=1;

  G11[0]=1; G11[16]=0; G11[32]=2; G11[48]=1;
  G11[1]=0; G11[17]=0; G11[33]=1; G11[49]=1;
  G11[2]=1; G11[18]=0; G11[34]=1; G11[50]=0;
  G11[3]=0; G11[19]=0; G11[35]=0; G11[51]=0;
  G11[4]=0; G11[20]=0; G11[36]=1; G11[52]=1;
  G11[5]=0; G11[21]=1; G11[37]=1; G11[53]=2;
  G11[6]=0; G11[22]=0; G11[38]=0; G11[54]=0;
  G11[7]=0; G11[23]=1; G11[39]=0; G11[55]=1;
  G11[8]=1; G11[24]=0; G11[40]=1; G11[56]=0;
  G11[9]=0; G11[25]=0; G11[41]=0; G11[57]=0;
  G11[10]=2; G11[26]=1; G11[42]=1; G11[58]=0;
  G11[11]=1; G11[27]=1; G11[43]=0; G11[59]=0;
  G11[12]=0; G11[28]=0; G11[44]=0; G11[60]=0;
  G11[13]=0; G11[29]=1; G11[45]=0; G11[61]=1;
  G11[14]=1; G11[30]=1; G11[46]=0; G11[62]=0;
  G11[15]=1; G11[31]=2; G11[47]=0; G11[63]=1;
  
  rf_3pt(I1,*p1,I2,*p2,I3,*p3,n,*ntot,H42,G00,G01,G10,G11,&theta12[13],&theta23[13],&theta13[13],&log_like[13]);
  
  /* assignment 4-3 */
  G00[0]=0; G00[16]=0; G00[32]=0; G00[48]=0;
  G00[1]=0; G00[17]=1; G00[33]=0; G00[49]=1;
  G00[2]=1; G00[18]=1; G00[34]=0; G00[50]=0;
  G00[3]=1; G00[19]=2; G00[35]=0; G00[51]=1;
  G00[4]=1; G00[20]=0; G00[36]=1; G00[52]=0;
  G00[5]=0; G00[21]=0; G00[37]=0; G00[53]=0;
  G00[6]=2; G00[22]=1; G00[38]=1; G00[54]=0;
  G00[7]=1; G00[23]=1; G00[39]=0; G00[55]=0;
  G00[8]=0; G00[24]=0; G00[40]=1; G00[56]=1;
  G00[9]=0; G00[25]=1; G00[41]=1; G00[57]=2;
  G00[10]=0; G00[26]=0; G00[42]=0; G00[58]=0;
  G00[11]=0; G00[27]=1; G00[43]=0; G00[59]=1;
  G00[12]=1; G00[28]=0; G00[44]=2; G00[60]=1;
  G00[13]=0; G00[29]=0; G00[45]=1; G00[61]=1;
  G00[14]=1; G00[30]=0; G00[46]=1; G00[62]=0;
  G00[15]=0; G00[31]=0; G00[47]=0; G00[63]=0;

  G01[0]=0; G01[16]=0; G01[32]=0; G01[48]=0;
  G01[1]=1; G01[17]=0; G01[33]=1; G01[49]=0;
  G01[2]=0; G01[18]=0; G01[34]=1; G01[50]=1;
  G01[3]=1; G01[19]=0; G01[35]=2; G01[51]=1;
  G01[4]=0; G01[20]=1; G01[36]=0; G01[52]=1;
  G01[5]=0; G01[21]=0; G01[37]=0; G01[53]=0;
  G01[6]=0; G01[22]=1; G01[38]=1; G01[54]=2;
  G01[7]=0; G01[23]=0; G01[39]=1; G01[55]=1;
  G01[8]=1; G01[24]=1; G01[40]=0; G01[56]=0;
  G01[9]=2; G01[25]=1; G01[41]=1; G01[57]=0;
  G01[10]=0; G01[26]=0; G01[42]=0; G01[58]=0;
  G01[11]=1; G01[27]=0; G01[43]=1; G01[59]=0;
  G01[12]=1; G01[28]=2; G01[44]=0; G01[60]=1;
  G01[13]=1; G01[29]=1; G01[45]=0; G01[61]=0;
  G01[14]=0; G01[30]=1; G01[46]=0; G01[62]=1;
  G01[15]=0; G01[31]=0; G01[47]=0; G01[63]=0;

  G10[0]=1; G10[16]=0; G10[32]=2; G10[48]=1;
  G10[1]=0; G10[17]=0; G10[33]=1; G10[49]=1;
  G10[2]=1; G10[18]=0; G10[34]=1; G10[50]=0;
  G10[3]=0; G10[19]=0; G10[35]=0; G10[51]=0;
  G10[4]=0; G10[20]=0; G10[36]=1; G10[52]=1;
  G10[5]=0; G10[21]=1; G10[37]=1; G10[53]=2;
  G10[6]=0; G10[22]=0; G10[38]=0; G10[54]=0;
  G10[7]=0; G10[23]=1; G10[39]=0; G10[55]=1;
  G10[8]=1; G10[24]=0; G10[40]=1; G10[56]=0;
  G10[9]=0; G10[25]=0; G10[41]=0; G10[57]=0;
  G10[10]=2; G10[26]=1; G10[42]=1; G10[58]=0;
  G10[11]=1; G10[27]=1; G10[43]=0; G10[59]=0;
  G10[12]=0; G10[28]=0; G10[44]=0; G10[60]=0;
  G10[13]=0; G10[29]=1; G10[45]=0; G10[61]=1;
  G10[14]=1; G10[30]=1; G10[46]=0; G10[62]=0;
  G10[15]=1; G10[31]=2; G10[47]=0; G10[63]=1;

  G11[0]=1; G11[16]=2; G11[32]=0; G11[48]=1;
  G11[1]=1; G11[17]=1; G11[33]=0; G11[49]=0;
  G11[2]=0; G11[18]=1; G11[34]=0; G11[50]=1;
  G11[3]=0; G11[19]=0; G11[35]=0; G11[51]=0;
  G11[4]=1; G11[20]=1; G11[36]=0; G11[52]=0;
  G11[5]=2; G11[21]=1; G11[37]=1; G11[53]=0;
  G11[6]=0; G11[22]=0; G11[38]=0; G11[54]=0;
  G11[7]=1; G11[23]=0; G11[39]=1; G11[55]=0;
  G11[8]=0; G11[24]=1; G11[40]=0; G11[56]=1;
  G11[9]=0; G11[25]=0; G11[41]=0; G11[57]=0;
  G11[10]=0; G11[26]=1; G11[42]=1; G11[58]=2;
  G11[11]=0; G11[27]=0; G11[43]=1; G11[59]=1;
  G11[12]=0; G11[28]=0; G11[44]=0; G11[60]=0;
  G11[13]=1; G11[29]=0; G11[45]=1; G11[61]=0;
  G11[14]=0; G11[30]=0; G11[46]=1; G11[62]=1;
  G11[15]=1; G11[31]=0; G11[47]=2; G11[63]=1;
  
  rf_3pt(I1,*p1,I2,*p2,I3,*p3,n,*ntot,H43,G00,G01,G10,G11,&theta12[14],&theta23[14],&theta13[14],&log_like[14]);
  
  /* assignment 4-4 */
  G00[0]=0; G00[16]=0; G00[32]=0; G00[48]=0;
  G00[1]=1; G00[17]=0; G00[33]=1; G00[49]=0;
  G00[2]=1; G00[18]=1; G00[34]=0; G00[50]=0;
  G00[3]=2; G00[19]=1; G00[35]=1; G00[51]=0;
  G00[4]=0; G00[20]=1; G00[36]=0; G00[52]=1;
  G00[5]=0; G00[21]=0; G00[37]=0; G00[53]=0;
  G00[6]=1; G00[22]=2; G00[38]=0; G00[54]=1;
  G00[7]=1; G00[23]=1; G00[39]=0; G00[55]=0;
  G00[8]=0; G00[24]=0; G00[40]=1; G00[56]=1;
  G00[9]=1; G00[25]=0; G00[41]=2; G00[57]=1;
  G00[10]=0; G00[26]=0; G00[42]=0; G00[58]=0;
  G00[11]=1; G00[27]=0; G00[43]=1; G00[59]=0;
  G00[12]=0; G00[28]=1; G00[44]=1; G00[60]=2;
  G00[13]=0; G00[29]=0; G00[45]=1; G00[61]=1;
  G00[14]=0; G00[30]=1; G00[46]=0; G00[62]=1;
  G00[15]=0; G00[31]=0; G00[47]=0; G00[63]=0;

  G01[0]=0; G01[16]=0; G01[32]=0; G01[48]=0;
  G01[1]=0; G01[17]=1; G01[33]=0; G01[49]=1;
  G01[2]=0; G01[18]=0; G01[34]=1; G01[50]=1;
  G01[3]=0; G01[19]=1; G01[35]=1; G01[51]=2;
  G01[4]=1; G01[20]=0; G01[36]=1; G01[52]=0;
  G01[5]=0; G01[21]=0; G01[37]=0; G01[53]=0;
  G01[6]=1; G01[22]=0; G01[38]=2; G01[54]=1;
  G01[7]=0; G01[23]=0; G01[39]=1; G01[55]=1;
  G01[8]=1; G01[24]=1; G01[40]=0; G01[56]=0;
  G01[9]=1; G01[25]=2; G01[41]=0; G01[57]=1;
  G01[10]=0; G01[26]=0; G01[42]=0; G01[58]=0;
  G01[11]=0; G01[27]=1; G01[43]=0; G01[59]=1;
  G01[12]=2; G01[28]=1; G01[44]=1; G01[60]=0;
  G01[13]=1; G01[29]=1; G01[45]=0; G01[61]=0;
  G01[14]=1; G01[30]=0; G01[46]=1; G01[62]=0;
  G01[15]=0; G01[31]=0; G01[47]=0; G01[63]=0;

  G10[0]=0; G10[16]=1; G10[32]=1; G10[48]=2;
  G10[1]=0; G10[17]=0; G10[33]=1; G10[49]=1;
  G10[2]=0; G10[18]=1; G10[34]=0; G10[50]=1;
  G10[3]=0; G10[19]=0; G10[35]=0; G10[51]=0;
  G10[4]=0; G10[20]=0; G10[36]=1; G10[52]=1;
  G10[5]=1; G10[21]=0; G10[37]=2; G10[53]=1;
  G10[6]=0; G10[22]=0; G10[38]=0; G10[54]=0;
  G10[7]=1; G10[23]=0; G10[39]=1; G10[55]=0;
  G10[8]=0; G10[24]=1; G10[40]=0; G10[56]=1;
  G10[9]=0; G10[25]=0; G10[41]=0; G10[57]=0;
  G10[10]=1; G10[26]=2; G10[42]=0; G10[58]=1;
  G10[11]=1; G10[27]=1; G10[43]=0; G10[59]=0;
  G10[12]=0; G10[28]=0; G10[44]=0; G10[60]=0;
  G10[13]=1; G10[29]=0; G10[45]=1; G10[61]=0;
  G10[14]=1; G10[30]=1; G10[46]=0; G10[62]=0;
  G10[15]=2; G10[31]=1; G10[47]=1; G10[63]=0;

  G11[0]=2; G11[16]=1; G11[32]=1; G11[48]=0;
  G11[1]=1; G11[17]=1; G11[33]=0; G11[49]=0;
  G11[2]=1; G11[18]=0; G11[34]=1; G11[50]=0;
  G11[3]=0; G11[19]=0; G11[35]=0; G11[51]=0;
  G11[4]=1; G11[20]=1; G11[36]=0; G11[52]=0;
  G11[5]=1; G11[21]=2; G11[37]=0; G11[53]=1;
  G11[6]=0; G11[22]=0; G11[38]=0; G11[54]=0;
  G11[7]=0; G11[23]=1; G11[39]=0; G11[55]=1;
  G11[8]=1; G11[24]=0; G11[40]=1; G11[56]=0;
  G11[9]=0; G11[25]=0; G11[41]=0; G11[57]=0;
  G11[10]=1; G11[26]=0; G11[42]=2; G11[58]=1;
  G11[11]=0; G11[27]=0; G11[43]=1; G11[59]=1;
  G11[12]=0; G11[28]=0; G11[44]=0; G11[60]=0;
  G11[13]=0; G11[29]=1; G11[45]=0; G11[61]=1;
  G11[14]=0; G11[30]=0; G11[46]=1; G11[62]=1;
  G11[15]=0; G11[31]=1; G11[47]=1; G11[63]=2;
  
  rf_3pt(I1,*p1,I2,*p2,I3,*p3,n,*ntot,H44,G00,G01,G10,G11,&theta12[15],&theta23[15],&theta13[15],&log_like[15]);
     
  /* the following statements calculate the posterior probability of each assignment */
  log_sum_like = -1e50;
  for (i=0;i<16;i++) log_sum_like = log_add(log_sum_like,log_like[i]);
  for (i=0;i<16;i++) posterior[i] = exp(log_like[i]-log_sum_like);
     
  /* the likelihood under null hipothesis of no linkage is calculated below */
  H11(0.25,0.25,0.25,0.25,H); /* crossover events under null hipothesis */
  mkron(I1,*p1,4,I2,*p2,4,kron12);
  mprod(kron12,*p1 * *p2,16,(double *)H,16,4,temp);
  mprod(temp,*p1 * *p2,4,I3,4,*p3,v);
  log_like_null = 0.0;
  for (i=0;i<(*p1 * *p2 * *p3);i++)
    log_like_null += n[i]*log(v[i]);
  
  /* the LOD score is calculated for each assignment */
  for (i=0;i<16;i++) LOD[i] = (log_like[i]-log_like_null)/log(10);
}



