% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/oem_xtx.R
\name{oem.xtx}
\alias{oem.xtx}
\title{Orthogonalizing EM with precomputed XtX}
\usage{
oem.xtx(xtx, xty, family = c("gaussian", "binomial"),
  penalty = c("elastic.net", "lasso", "ols", "mcp", "scad", "mcp.net",
  "scad.net", "grp.lasso", "grp.lasso.net", "grp.mcp", "grp.scad",
  "grp.mcp.net", "grp.scad.net", "sparse.grp.lasso"), lambda = numeric(0),
  nlambda = 100L, lambda.min.ratio = NULL, alpha = 1, gamma = 3,
  tau = 0.5, groups = numeric(0), scale.factor = numeric(0),
  penalty.factor = NULL, group.weights = NULL, maxit = 500L,
  tol = 1e-07, irls.maxit = 100L, irls.tol = 0.001)
}
\arguments{
\item{xtx}{input matrix equal to \code{crossprod(x) / nrow(x)}. 
where \code{x} is the design matrix.
It is highly recommended to scale by the number of rows in \code{x}.
If \code{xtx} is scaled, \code{xty} must also be scaled or else results may be meaningless!}

\item{xty}{numeric vector of length \code{nvars}. Equal to \code{crosprod(x, y) / nobs}. 
It is highly recommended to scale by the number of rows in \code{x}.}

\item{family}{\code{"gaussian"} for least squares problems, \code{"binomial"} for binary response. 
(only \code{gaussian} implemented currently)}

\item{penalty}{Specification of penalty type. Choices include:
\itemize{
   \item{\code{"elastic.net"}}{ - elastic net penalty, extra parameters: \code{"alpha"}}
   \item{\code{"lasso"}}{ - lasso penalty}
   \item{\code{"ols"}}{ - ordinary least squares}
   \item{\code{"mcp"}}{ - minimax concave penalty, extra parameters: \code{"gamma"}}
   \item{\code{"scad"}}{ - smoothly clipped absolute deviation, extra parameters: \code{"gamma"}}
   \item{\code{"mcp.net"}}{ - minimax concave penalty + l2 penalty, extra parameters: 
   \code{"gamma"}, \code{"alpha"}}
   \item{\code{"scad.net"}}{ - smoothly clipped absolute deviation + l2 penalty, extra parameters: 
   \code{"gamma"}, \code{"alpha"}}
   \item{\code{"grp.lasso"}}{ - group lasso penalty}
   \item{\code{"grp.lasso.net"}}{ - group lasso penalty + l2 penalty, extra parameters: \code{"alpha"}}
   \item{\code{"grp.mcp"}}{ - group minimax concave penalty, extra parameters: \code{"gamma"}}
   \item{\code{"grp.scad"}}{ - group smoothly clipped absolute deviation, extra parameters: \code{"gamma"}}
   \item{\code{"grp.mcp.net"}}{ - group minimax concave penalty + l2 penalty, extra parameters: \code{"gamma"}, \code{"alpha"}}
   \item{\code{"grp.scad.net"}}{ - group smoothly clipped absolute deviation + l2 penalty, extra parameters: \code{"gamma"}, \code{"alpha"}}
   \item{\code{"sparse.grp.lasso"}}{ - sparse group lasso penalty (group lasso + lasso), extra parameters: \code{"tau"}}
}
Careful consideration is required for the group lasso, group MCP, and group SCAD penalties. Groups as specified by the \code{groups} argument 
should be chosen in a sensible manner.}

\item{lambda}{A user supplied lambda sequence. By default, the program computes
its own lambda sequence based on \code{nlambda} and \code{lambda.min.ratio}. Supplying
a value of lambda overrides this.}

\item{nlambda}{The number of lambda values - default is 100.}

\item{lambda.min.ratio}{Smallest value for lambda, as a fraction of \code{lambda.max}, the (data derived) entry
value (i.e. the smallest value for which all coefficients are zero). The default
depends on the sample size nobs relative to the number of variables nvars. The default is 0.0001}

\item{alpha}{mixing value for \code{elastic.net}, \code{mcp.net}, \code{scad.net}, \code{grp.mcp.net}, \code{grp.scad.net}. 
penalty applied is (1 - alpha) * (ridge penalty) + alpha * (lasso/mcp/mcp/grp.lasso penalty)}

\item{gamma}{tuning parameter for SCAD and MCP penalties. must be >= 1}

\item{tau}{mixing value for \code{sparse.grp.lasso}. penalty applied is (1 - tau) * (group lasso penalty) + tau * (lasso penalty)}

\item{groups}{A vector of describing the grouping of the coefficients. See the example below. All unpenalized variables
should be put in group 0}

\item{scale.factor}{of length \code{nvars === ncol(xtx) == length(xty)} for scaling columns of \code{x}. The standard deviation
for each column of \code{x} is a common choice for \code{scale.factor}. Coefficients will be returned on original scale. Default is 
no scaling.}

\item{penalty.factor}{Separate penalty factors can be applied to each coefficient. 
This is a number that multiplies lambda to allow differential shrinkage. Can be 0 for some variables, 
which implies no shrinkage, and that variable is always included in the model. Default is 1 for all 
variables.}

\item{group.weights}{penalty factors applied to each group for the group lasso. Similar to \code{penalty.factor}, 
this is a number that multiplies lambda to allow differential shrinkage. Can be 0 for some groups, 
which implies no shrinkage, and that group is always included in the model. Default is sqrt(group size) for all
groups.}

\item{maxit}{integer. Maximum number of OEM iterations}

\item{tol}{convergence tolerance for OEM iterations}

\item{irls.maxit}{integer. Maximum number of IRLS iterations}

\item{irls.tol}{convergence tolerance for IRLS iterations. Only used if \code{family != "gaussian"}}
}
\value{
An object with S3 class \code{"oem"}
}
\description{
Orthogonalizing EM with precomputed XtX
}
\examples{
set.seed(123)
n.obs <- 1e4
n.vars <- 100

true.beta <- c(runif(15, -0.25, 0.25), rep(0, n.vars - 15))

x <- matrix(rnorm(n.obs * n.vars), n.obs, n.vars)
y <- rnorm(n.obs, sd = 3) + x \%*\% true.beta

fit <- oem(x = x, y = y, 
           penalty = c("lasso", "elastic.net", 
                        "ols", 
                        "mcp",       "scad", 
                        "mcp.net",   "scad.net",
                        "grp.lasso", "grp.lasso.net",
                        "grp.mcp",   "grp.scad",
                        "sparse.grp.lasso"), 
           standardize = FALSE, intercept = FALSE,
           groups = rep(1:20, each = 5))
           
xtx <- crossprod(x) / n.obs
xty <- crossprod(x, y) / n.obs

fit.xtx <- oem.xtx(xtx = xtx, xty = xty, 
                   penalty = c("lasso", "elastic.net", 
                               "ols", 
                               "mcp",       "scad", 
                               "mcp.net",   "scad.net",
                               "grp.lasso", "grp.lasso.net",
                               "grp.mcp",   "grp.scad",
                               "sparse.grp.lasso"), 
                   groups = rep(1:20, each = 5))    
                   
max(abs(fit$beta[[1]][-1,] - fit.xtx$beta[[1]]))
max(abs(fit$beta[[2]][-1,] - fit.xtx$beta[[2]]))       

layout(matrix(1:2, ncol = 2))
plot(fit.xtx)
plot(fit.xtx, which.model = 2)

}
