% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mainODPC.R
\name{crit.odpc}
\alias{crit.odpc}
\title{Automatic Choice of Tuning Parameters for One-Sided Dynamic Principal Components via the Minimization of an Information Criterion}
\usage{
crit.odpc(
  Z,
  k_list = 1:5,
  max_num_comp = 5,
  ncores = 1,
  method,
  tol = 1e-04,
  niter_max = 500
)
}
\arguments{
\item{Z}{Data matrix. Each column is a different time series.}

\item{k_list}{List of values of k to choose from.}

\item{max_num_comp}{Maximum possible number of components to compute.}

\item{ncores}{Number of cores to use in parallel computations.}

\item{method}{A string specifying the algorithm used. Options are 'ALS', 'mix' or 'gradient'. See details in \code{\link{odpc}}.}

\item{tol}{Relative precision. Default is 1e-4.}

\item{niter_max}{Integer. Maximum number of iterations. Default is 500.}
}
\value{
An object of class odpcs, that is, a list of length equal to the number of computed components, each computed using the optimal value of k. 
The i-th entry of this list is an object of class \code{odpc}, that is, a list with entries
\item{f}{Coordinates of the i-th dynamic principal component corresponding to the periods \eqn{k_1 + 1,\dots,T}.}
\item{mse}{Mean squared error of the reconstruction using the first i components.}
\item{k1}{Number of lags used to define the i-th dynamic principal component f.}
\item{k2}{Number of lags of f used to reconstruct.}
\item{alpha}{Vector of intercepts corresponding to f.}
\item{a}{Vector that defines the i-th dynamic principal component}
\item{B}{Matrix of loadings corresponding to f. Row number \eqn{k} is the vector of \eqn{k-1} lag loadings.}
\item{call}{The matched call.}
\item{conv}{Logical. Did the iterations converge?}
\code{components}, \code{fitted}, \code{plot} and \code{print} methods are available for this class.
}
\description{
Computes One-Sided Dynamic Principal Components, choosing the number of components and lags automatically, to minimize an 
information criterion.
}
\details{
We apply the same stepwise approach taken in \code{\link{cv.odpc}}, but now to minimize an
information criterion instead of the cross-validated forecasting error. The criterion is
inspired by the \eqn{IC_{p3}} criterion proposed in Bai and Ng (2002).
Let \eqn{\widehat{\sigma}^{2}_{1,k}} be the reconstruction mean squared error for
the first ODPC defined using \eqn{k} lags. Let \eqn{T^{\ast,1,k}=T-2k}.
Then we choose the
value \eqn{k^{\ast,1}} in \code{k_list} that minimizes
\deqn{
 {BNG}_{1,k}=\log\left( \widehat{\sigma}^{2}_{1,k} \right)
 + ( k+1 ) \frac{\log\left(\min(T^{\ast,1,k},m)\right)}{\min(T^{\ast,1,k},m)}.
 }
Suppose now that \code{max_num_comp} \eqn{\geq 2} and we
have computed \eqn{q-1} dynamic principal components, \eqn{q-1 <} \code{max_num_comp}, each with \eqn{k_{1}^{i}=k_{2}^{i}=k^{\ast, i}} lags, \eqn{i=1,\dots,q-1}.
Let \eqn{\widehat{\sigma}^{2}_{q,k}} be the reconstruction mean squared error for
the fit obtained using \eqn{q} components, where the first \eqn{q-1} components are defined using 
\eqn{k^{\ast, i}}, \eqn{i=1,\dots,q-1} and the last component is defined using \eqn{k} lags.
Let \eqn{T^{\ast,q,k}=T-\max\lbrace 2k^{\ast,1},\dots,2k^{\ast,q-1},2k \rbrace}.
Let \eqn{k^{\ast,q}} be the value in \code{k_list} that minimizes
\deqn{
 {BNG}_{q,k}=\log\left( \widehat{\sigma}^{2}_{q,k} \right)
 + \left(\sum_{i=1}^{q-1}(k^{\ast,i}+1) + k+1 \right) \frac{\log\left(\min(T^{\ast,q,k},m)\right)}{\min(T^{\ast,q,k},m)}  .
 }
If \eqn{{BNG}_{q,k^{\ast,q}}} is larger than \eqn{{BNG}_{q-1,k^{\ast,q-1}}}
we stop and the final model is the ODPC with \eqn{q-1} components. Else we add the \eqn{q}-th component defined using \eqn{k^{\ast,q}}
and continue as before.
}
\examples{
T <- 50 #length of series
m <- 10 #number of series
set.seed(1234)
f <- rnorm(T + 1)
x <- matrix(0, T, m)
u <- matrix(rnorm(T * m), T, m)
for (i in 1:m) {
  x[, i] <- 10 * sin(2 * pi * (i/m)) * f[1:T] + 10 * cos(2 * pi * (i/m)) * f[2:(T + 1)] + u[, i]
}
# Choose parameters to perform a one step ahead forecast. Use 1 or 2 lags, only one component 
fit <- crit.odpc(x, k_list = 1:2, max_num_comp = 1)
}
\references{
Peña D., Smucler E. and Yohai V.J. (2017). “Forecasting Multiple Time Series with One-Sided Dynamic Principal Components.” Available at https://arxiv.org/abs/1708.04705.

Bai J. and Ng S. (2002). “Determining the Number of Factors in Approximate Factor Models.” Econometrica, 70(1), 191–221.
}
\seealso{
\code{\link{odpc}}, \code{\link{cv.odpc}}, \code{\link{forecast.odpcs}}
}
