% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/misc.R
\name{curl}
\alias{curl}
\title{Curl of 2D Vector Field}
\usage{
curl(u, v, x, y, geographical = FALSE, method = 1)
}
\arguments{
\item{u}{matrix containing the 'x' component of a vector field}

\item{v}{matrix containing the 'y' component of a vector field}

\item{x}{the x values for the matrices, a vector of length equal to the
number of rows in \code{u} and \code{v}.}

\item{y}{the y values for the matrices, a vector of length equal to the
number of cols in \code{u} and \code{v}.}

\item{geographical}{logical value indicating whether \code{x} and \code{y}
are longitude and latitude, in which case spherical trigonometry is used.}

\item{method}{A number indicating the method to be used to calculate the
first-difference approximations to the derivatives.  See \dQuote{Details}.}
}
\value{
A list containing vectors \code{x} and \code{y}, along with matrix
\code{curl}.  See \dQuote{Details} for the lengths and dimensions, for
various values of \code{method}.
}
\description{
Calculate the z component of the curl of an x-y vector field.
}
\details{
The computed component of the curl is defined by \eqn{\partial }{dv/dx -
du/dy}\eqn{ v/\partial x - \partial u/\partial y}{dv/dx - du/dy} and the
estimate is made using first-difference approximations to the derivatives.
Two methods are provided, selected by the value of \code{method}.
\itemize{
\item For \code{method=1}, a centred-difference, 5-point stencil is used in
the interior of the domain.  For example, \eqn{\partial v/\partial x}{dv/dx}
is given by the ratio of \eqn{v_{i+1,j}-v_{i-1,j}}{v[i+1,j]-v[i-1,j]} to the
x extent of the grid cell at index \eqn{j}{j}. (The cell extents depend on
the value of \code{geographical}.)  Then, the edges are filled in with
nearest-neighbour values. Finally, the corners are filled in with the
adjacent value along a diagonal.  If \code{geographical=TRUE}, then \code{x}
and \code{y} are taken to be longitude and latitude in degrees, and the
earth shape is approximated as a sphere with radius 6371km.  The resultant
\code{x} and \code{y} are identical to the provided values, and the
resultant \code{curl} is a matrix with dimension identical to that of
\code{u}.
\item For \code{method=2}, each interior cell in the grid is considered
individually, with derivatives calculated at the cell center. For example,
\eqn{\partial v/\partial x}{dv/dx} is given by the ratio of
\eqn{0.5*(v_{i+1,j}+v_{i+1,j+1}) -
0.5*(v_{i,j}+v_{i,j+1})}{0.5*(v[i+1,j]+v[i+1,j+1]) - 0.5*(v[i,j]+v[i,j+1])}
to the average of the x extent of the grid cell at indices \eqn{j}{j} and
\eqn{j+1}{j+1}. (The cell extents depend on the value of
\code{geographical}.)  The returned \code{x} and \code{y} values are the
mid-points of the supplied values. Thus, the returned \code{x} and \code{y}
are shorter than the supplied values by 1 item, and the returned \code{curl}
matrix dimensions are similarly reduced compared with the dimensions of
\code{u} and \code{v}.
}
}
\section{Development status.}{
 This function is under active development as
of December 2014 and is unlikely to be stabilized until February 2015.
}

\examples{
library(oce)
# 1. Shear flow with uniform curl.
x <- 1:4
y <- 1:10
u <- outer(x, y, function(x, y) y / 2)
v <- outer(x, y, function(x, y) -x / 2)
C <- curl(u, v, x, y, FALSE)

# 2. Rankine vortex: constant curl inside circle, zero outside
rankine <- function(x, y) {
    r <- sqrt(x^2 + y^2)
    theta <- atan2(y, x)
    speed <- ifelse(r < 1, 0.5 * r, 0.5 / r)
    list(u = -speed * sin(theta), v = speed * cos(theta))
}
x <- seq(-2, 2, length.out = 100)
y <- seq(-2, 2, length.out = 50)
u <- outer(x, y, function(x, y) rankine(x, y)$u)
v <- outer(x, y, function(x, y) rankine(x, y)$v)
C <- curl(u, v, x, y, FALSE)
# plot results
par(mfrow = c(2, 2))
imagep(x, y, u, zlab = "u", asp = 1)
imagep(x, y, v, zlab = "v", asp = 1)
imagep(x, y, C$curl, zlab = "curl", asp = 1)
hist(C$curl, breaks = 100)
}
\seealso{
Other things relating to vector calculus: 
\code{\link{grad}()}
}
\author{
Dan Kelley and Chantelle Layton
}
\concept{things relating to vector calculus}
