% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/argo.R
\name{handleFlags,argo-method}
\alias{handleFlags,argo-method}
\alias{handleFlags.argo}
\title{Handle Flags in ARGO Objects}
\usage{
\S4method{handleFlags}{argo}(
  object,
  flags = NULL,
  actions = NULL,
  where = NULL,
  debug = getOption("oceDebug")
)
}
\arguments{
\item{object}{an \linkS4class{argo} object.}

\item{flags}{A \link{list} specifying flag values upon which
actions will be taken. This can take two forms.
\itemize{
\item In the first form, the
list has named elements each containing a vector of integers. For example,
salinities flagged with values of 1 or 3:9 would be specified
by \code{flags=list(salinity=c(1,3:9))}. Several data items can be specified,
e.g. \code{flags=list(salinity=c(1,3:9), temperature=c(1,3:9))} indicates
that the actions are to take place for both salinity and temperature.
\item In the second form, \code{flags} is a list holding a single \emph{unnamed} vector, and
this means to apply the actions to \emph{all} the data entries.  For example,
\code{flags=list(c(1,3:9))} means to apply not just to salinity and temperature,
but to everything within the \code{data} slot.
}

If \code{flags} is not provided, then \code{\link[=defaultFlags]{defaultFlags()}} is called, to try to
determine a reasonable default.}

\item{actions}{an optional \link{list} that contains items with
names that match those in the \code{flags} argument.  If \code{actions}
is not supplied, the default will be to set all values identified by
\code{flags} to \code{NA}; this can also be specified by
specifying \code{actions=list("NA")}. It is also possible to specify
functions that calculate replacement values. These are provided
with \code{object} as the single argument, and must return a
replacement for the data item in question.
See \dQuote{Details} for the default that is used if \code{actions} is not supplied.}

\item{where}{an optional character value that permits the function to work with
objects that store flags in e.g. \code{object@metadata$flags$where}
instead of in \code{object@metadata$flags}, and data within
\code{object@data$where} instead of within \code{object@data}. The
default value of \code{NULL} means to look withing \code{object@metadata}
itself, and this is the default within \code{oce}.  (The purpose of \code{where}
is to make \code{oce} extensible by other packages, which may choose to store
data two levels deep in the \code{data} slot.)}

\item{debug}{An optional integer specifying the degree of debugging, with
value 0 meaning to skip debugging and 1 or higher meaning to print some
information about the arguments and the data. It is usually a good idea to set
this to 1 for initial work with a dataset, to see which flags are being
handled for each data item. If not supplied, this defaults to the value of
\code{\link{getOption}}\code{("oceDebug")}.}
}
\description{
Data-quality flags are stored in the \code{metadata}
slot of \linkS4class{oce} objects in a
\link{list} named \code{flags}.
The present function (a generic that has specialized versions
for various data classes) provides a way to
manipulate the contents of the \code{data} slot, based on
such data-quality flags. For example, a common operation is to replace
erroneous data with \code{NA}.

If the \code{flags} within \code{object}'s \code{metadata} slot is empty,
then \code{object} is returned, unaltered.
Otherwise, \code{handleFlags} examines \code{object@metadata$flags}
in the context of the \code{flags} argument, and then
carries out actions that are specified by the \code{actions} argument.
By default, this sets the returned \code{data} entries to \code{NA},
wherever the corresponding \code{metadata$flag} values
signal unreliable data. To maintain a hint as to why
\code{data} were changed, \code{metadata$flags} in the
returned value is a direct copy of the corresponding
entry in \code{object}.
}
\examples{
library(oce)
data(argo)
argoNew <- handleFlags(argo)
# Demonstrate replacement, looking at the second profile
f <- argo[["salinityFlag"]][,2]
df <- data.frame(flag=f, orig=argo[["salinity"]][,2], new=argoNew[["salinity"]][,2])
df[11:15,] # notice line 13

}
\references{
\enumerate{
\item Wong, Annie, Robert Keeley, Thierry Carval, and Argo Data Management Team.
"Argo Quality Control Manual for CTD and Trajectory Data," January 1, 2020.
\verb{https://archimer.ifremer.fr/doc/00228/33951/}.
}
}
\seealso{
Other functions relating to data-quality flags: 
\code{\link{defaultFlags}()},
\code{\link{handleFlags,adp-method}},
\code{\link{handleFlags,ctd-method}},
\code{\link{handleFlags,oce-method}},
\code{\link{handleFlags,section-method}},
\code{\link{handleFlags}()},
\code{\link{initializeFlagScheme,ctd-method}},
\code{\link{initializeFlagScheme,oce-method}},
\code{\link{initializeFlagScheme,section-method}},
\code{\link{initializeFlagSchemeInternal}()},
\code{\link{initializeFlagScheme}()},
\code{\link{initializeFlags,adp-method}},
\code{\link{initializeFlags,oce-method}},
\code{\link{initializeFlagsInternal}()},
\code{\link{initializeFlags}()},
\code{\link{setFlags,adp-method}},
\code{\link{setFlags,ctd-method}},
\code{\link{setFlags,oce-method}},
\code{\link{setFlags}()}

Other things related to argo data: 
\code{\link{[[,argo-method}},
\code{\link{[[<-,argo-method}},
\code{\link{argo-class}},
\code{\link{argoGrid}()},
\code{\link{argoNames2oceNames}()},
\code{\link{argo}},
\code{\link{as.argo}()},
\code{\link{plot,argo-method}},
\code{\link{read.argo.copernicus}()},
\code{\link{read.argo}()},
\code{\link{subset,argo-method}},
\code{\link{summary,argo-method}}
}
\author{
Dan Kelley
}
\concept{functions relating to data-quality flags}
\concept{things related to argo data}
