% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/section.R
\docType{methods}
\name{plot,section-method}
\alias{plot,section-method}
\alias{plot.section}
\title{Plot a Section}
\usage{
\S4method{plot}{section}(x, which = c(1, 2, 3, 99), eos, at = NULL,
  labels = TRUE, grid = FALSE, contourLevels = NULL,
  contourLabels = NULL, stationIndices, coastline = "best",
  xlim = NULL, ylim = NULL, zlim = NULL, map.xlim = NULL,
  map.ylim = NULL, clongitude, clatitude, span, projection = NULL,
  xtype = "distance", ytype = "depth", ztype = "contour",
  zbreaks = NULL, zcol = NULL, legend.loc = "bottomright",
  showStations = FALSE, showStart = TRUE, showBottom = TRUE,
  axes = TRUE, mgp, mar, col, cex, pch, labcex = 1, debug, ...)
}
\arguments{
\item{x}{a \code{section} object, e.g. as created by \code{\link{as.section}}
or \code{\link{read.section}}.}

\item{which}{a list of desired plot types, as explained in \dQuote{Details}.
There may be up to four panels in total, and the desired plots are placed in
these panels, in reading order.  If only one panel is plotted, \code{par} is
not adjusted, which makes it easy to add to the plot with subsequent plotting
commands.}

\item{eos}{Character indication of the seawater equation of state
to use. The permitted choices are \code{"gsw"} and \code{"unesco"}.
If \code{eos} is not supplied, it defaults to
\code{getOption("oceEOS", default="gsw")}.}

\item{at}{If \code{NULL} (the default), the x axis will indicate the distance
of the stations from the first in the section.  (This may give errors in the
contouring routine, if the stations are not present in a geographical order.)
If a list, then it indicates the values at which stations will be plotted.}

\item{labels}{Either a logical, indicating whether to put labels on the x axis,
or a vector that is a list of labels to be placed at the x positions indicated
by \code{at}.}

\item{grid}{If \code{TRUE}, points are drawn at data locations.}

\item{contourLevels}{Optional contour levels.}

\item{contourLabels}{Optional contour labels.}

\item{stationIndices}{Optional list of the indices of stations to use.  Note
that an index is \emph{not} a station number, e.g. to show the first 4
stations, use \code{station.indices=1:4}.}

\item{coastline}{String giving the coastline to be used in a station map
The permitted choices are \code{"best"} (the default) to pick
a variant that suits the scale, \code{"coastlineWorld"} for the coarse
version that is provided by \CRANpkg{oce},
\code{"coastlineWorldMedium"} or \code{"coastlineWorldFine"} for two
coastlines provided by the \CRANpkg{ocedata} package, or \code{"none"}, to avoid
drawing a coastline.}

\item{xlim}{Optional limit for x axis (only in sections, not map).}

\item{ylim}{Optional limit for y axis (only in sections, not map)}

\item{zlim}{Optional two-element numerical vector specifying the
limit on the plotted field. This is used only if \code{ztype="image"};
see also \code{zbreaks} and \code{zcol}.}

\item{map.xlim, map.ylim}{Optional limits for station map; \code{map.ylim} is
ignored if \code{map.xlim} is provided.}

\item{clongitude, clatitude, span}{Optional map centre position and span (km).}

\item{projection}{Parameter specifying map
projection; see \code{\link{mapPlot}}.  If \code{projection="automatic"},
however, a projection is devised from the data, with \code{stereographic} if
the mean latitude exceeds 70N and \code{mollweide} otherwise.}

\item{xtype}{Type of x axis, for contour plots, either \code{"distance"} for
distance (in km) to the first point in the section, \code{"track"} for distance
along the cruise track, \code{"longitude"}, \code{"latitude"}, or
\code{"time"}.  Note that if the x values are not in order, they will be put in
order (which may make no sense) and a warning will be printed.}

\item{ytype}{Type of y axis for contour plots, either \code{"pressure"} for
pressure (in dbar, with zero at the surface) or \code{"depth"} for depth (in m
below the surface, calculated from pressure with \code{\link{swDepth}}).}

\item{ztype}{String indicating whether to how to indicate the "z"
data (in the R sense, i.e. this could be salinity, temperature, etc; it does
not mean the vertical coordinate) The choices are: \code{"contour"} for
contours, \code{"image"} for an image (drawn with \code{\link{imagep}} with
\code{filledContours=TRUE}), or \code{"points"} to draw points.
In the first two cases, the data must be gridded, with identical pressures at
each station.}

\item{zbreaks, zcol}{Indication of breaks and colors to be used if \code{ztype="points"} or
\code{"image"}. If not provided, reasonable default are used. If \code{zlim}
is given but \code{breaks} is not given, then \code{breaks} is computed to
run from \code{zlim[1]} to \code{zlim[2]}. If \code{zcol} is a function,
it will be invoked with an argument equal to
\code{1+length(zbreaks)}.}

\item{legend.loc}{Location of legend, as supplied to \code{\link{legend}}, or
set to the empty string to avoid plotting a legend.}

\item{showStations}{Logical indicating whether to draw station numbers on maps.}

\item{showStart}{Logical indicating whether to indicate the first station with
a different symbol than the others.}

\item{showBottom}{An indication of whether (and how) to indicate the ocean bottom.
If \code{FALSE}, then the bottom is not rendered. If \code{TRUE}, then it
is rendered with a gray polygon. If \code{showBottom} is a character string,
then there are three possibilities: is the string is \code{"polygon"} then
a polygon is drawn, if it is \code{"lines"} then a line is drawn, and if it
is \code{"points"} then points are drawn. If \code{showBottom} is an object
inheriting from \code{\link{topo-class}} then the station locations are
interpolated to that topography and the results are shown with a polygon.
In this last case, the interpolation is set at a grid that is roughly
in accordance with the resolution of the latitudes in the \code{topo} object.
See \dQuote{Examples}.}

\item{axes}{Logical value indicating whether to draw axes.}

\item{mgp}{A 3-element numerical vector to use for \code{par(mgp)}, and also for
\code{par(mar)}, computed from this. If not provided, this defaults to
\code{getOption("oceMgp")}.}

\item{mar}{Value to be used with \code{\link{par}("mar")}. If not provided,
a default is set up.}

\item{col}{Color, which defaults to \code{\link{par}("col")}.}

\item{cex}{Numerical character-expansion factor, which defaults to \code{\link{par}("cex")}.}

\item{pch}{Indication of symbol type; defaults to \code{\link{par}("pch")}.}

\item{labcex}{Size of characters in contour labels (passed to
\code{\link{contour}}).}

\item{debug}{an integer specifying whether debugging information is
to be printed during the processing. This is a general parameter that
is used by many \code{oce} functions. Generally, setting \code{debug=0}
turns off the printing, while higher values suggest that more information
be printed. If \code{debug} is not supplied, it defaults to
\code{\link{getOption}("oceDebug")}.}

\item{...}{Optional arguments passed to the contouring function.}
}
\value{
If the original section was gridded, the return value is that section.
Otherwise, the gridded section that was constructed for the plot is returned.
In both cases, the value is returned silently. The
purpose of returning the section is to enable subsequent processing
of the grid, including adding elements to the plot (see example 5).
}
\description{
Creates a summary plot for a CTD section, with one panel for each value of
\code{which}.
}
\details{
The type of plot is governed by \code{which}, as follows.

\itemize{
    \item \code{which=0} or \code{"potential temperature"} for temperature contours
    \item \code{which=1} or \code{"temperature"} for temperature contours (the default)
    \item \code{which=2} or \code{"salinity"} for salinity contours
    \item \code{which=3} or \code{"sigmaTheta"} for sigma-theta contours
    \item \code{which=4} or \code{"nitrate"} for nitrate concentration contours
    \item \code{which=5} or \code{"nitrite"} for nitrite concentration contours
    \item \code{which=6} or \code{"oxygen"} for oxygen concentration  contours
    \item \code{which=7} or \code{"phosphate"} for phosphate concentration contours
    \item \code{which=8} or \code{"silicate"} for silicate concentration contours
    \item \code{which=9} or \code{"u"} for eastward velocity
    \item \code{which=10} or \code{"uz"} for vertical derivative of eastward velocity
    \item \code{which=11} or \code{"v"} for northward velocity
    \item \code{which=12} or \code{"vz"} for vertical derivative of northward velocity
    \item \code{which=20} or \code{"data"} for a dot for each data location
    \item \code{which=99} or \code{"map"} for a location map
}

The y-axis for the contours is pressure, plotted in the conventional reversed
form, so that the water surface appears at the top of the plot.  The x-axis is
more complicated. If \code{at} is not supplied, then the routine calculates x
as the distance between the first station in the section and each of the other
stations. (This will produce an error if the stations are not ordered
geographically, because the \code{\link{contour}} routine cannot handle
non-increasing axis coordinates.) If \code{at} is specified, then it is taken
to be the location, in arbitrary units, along the x-axis of labels specified by
\code{labels}; the way this works is designed to be the same as for
\code{\link{axis}}.
}
\examples{
library(oce)
data(section)
sg <- sectionGrid(section)

## 1. start of section, default fields.
plot(head(section))

## 2. Gulf Stream
GS <- subset(section, 109<=stationId&stationId<=129)
GSg <- sectionGrid(GS, p=seq(0, 2000, 100))
plot(GSg, which=c(1, 99), map.ylim=c(34, 42))
par(mfrow=c(2, 1))
plot(GS, which=1, ylim=c(2000, 0), ztype='points',
     zbreaks=seq(0,30,2), pch=20, cex=3)
plot(GSg, which=1, ztype='image', zbreaks=seq(0,30,2))

par(mfrow=c(1, 1))

## 3. Image, with colored dots to indicate grid-data mismatch.
\dontrun{
plot(GSg, which=1, ztype='image')
T <- GS[['temperature']]
col <- oceColorsJet(100)[rescale(T, rlow=1, rhigh=100)]
points(GS[['distance']],GS[['depth']],pch=20,cex=3,col='white')
points(GS[['distance']],GS[['depth']],pch=20,cex=2.5,col=col)
}


\dontrun{
## 4. Image of Absolute Salinity, with 4-minute bathymetry
## It's easy to calculate the desired area for the bathymetry,
## but for brevity we'll hard-code it. Note that download.topo()
## caches the file locally.
f <- download.topo(west=-80, east=0, south=35, north=40, resolution=4)
t <- read.topo(f)
plot(section, which="SA", xtype="longitude", ztype="image", showBottom=t)
}

\dontrun{
## 5. Temperature with salinity added in red
s <- plot(section, which="temperature")
distance <- s[["distance", "byStation"]]
depth <- s[["station", 1]][["depth"]]
salinity <- matrix(s[["salinity"]], byrow=TRUE, nrow=length(s[["station"]]))
contour(distance, depth, salinity, col=2, add=TRUE)
}

}
\seealso{
The documentation for \code{\link{section-class}} explains the
structure of section objects, and also outlines the other functions dealing
with them.

Other functions that plot \code{oce} data: \code{\link{plot,adp-method}},
  \code{\link{plot,adv-method}},
  \code{\link{plot,amsr-method}},
  \code{\link{plot,argo-method}},
  \code{\link{plot,bremen-method}},
  \code{\link{plot,cm-method}},
  \code{\link{plot,coastline-method}},
  \code{\link{plot,ctd-method}},
  \code{\link{plot,gps-method}},
  \code{\link{plot,ladp-method}},
  \code{\link{plot,landsat-method}},
  \code{\link{plot,lisst-method}},
  \code{\link{plot,lobo-method}},
  \code{\link{plot,met-method}},
  \code{\link{plot,odf-method}},
  \code{\link{plot,rsk-method}},
  \code{\link{plot,satellite-method}},
  \code{\link{plot,sealevel-method}},
  \code{\link{plot,tidem-method}},
  \code{\link{plot,topo-method}},
  \code{\link{plot,windrose-method}},
  \code{\link{plotProfile}}, \code{\link{plotScan}},
  \code{\link{plotTS}}, \code{\link{tidem-class}}

Other things related to \code{section} data: \code{\link{[[,section-method}},
  \code{\link{[[<-,section-method}},
  \code{\link{as.section}},
  \code{\link{handleFlags,section-method}},
  \code{\link{initializeFlagScheme,section-method}},
  \code{\link{read.section}}, \code{\link{section-class}},
  \code{\link{sectionAddStation}},
  \code{\link{sectionGrid}}, \code{\link{sectionSmooth}},
  \code{\link{sectionSort}}, \code{\link{section}},
  \code{\link{subset,section-method}},
  \code{\link{summary,section-method}}
}
\author{
Dan Kelley
}
\concept{functions that plot \code{oce} data}
\concept{things related to \code{section} data}
