% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/colors.R
\name{colormap}
\alias{colormap}
\title{Calculate color map}
\usage{
colormap(z = NULL, zlim, zclip = FALSE, breaks, col = oceColorsJet,
  name, x0, x1, col0, col1, blend = 0, missingColor,
  debug = getOption("oceDebug"))
}
\arguments{
\item{z}{an optional vector or other set of numerical values to be examined.
If \code{z} is given, the return value will contain an item named
\code{zcol} that will be a vector of the same length as \code{z}, containing
a color for each point.  If \code{z} is not given, \code{zcol} will contain
just one item, the color \code{"black"}.}

\item{zlim}{optional vector containing two numbers that specify the \code{z}
limits for the color scale.  If provided, it overrides defaults as describe
in the following.  If \code{name} is given, then the \code{\link{range}} of
numerical values contained therein will be used for \code{zlim}.  Otherwise,
if \code{z} is given, then its \code{\link{rangeExtended}} sets \code{zlim}.
Otherwise, if \code{x0} and \code{x1} are given, then their
\code{\link{range}} sets \code{zlim}.  Otherwise, there is no way to infer
\code{zlim} and indeed there is no way to construct a colormap, so an error
is reported.  It is an error to specify both \code{zlim} and \code{breaks},
if the length of the latter does not equal 1.}

\item{zclip}{logical, with \code{TRUE} indicating that z values outside the
range of \code{zlim} or \code{breaks} should be painted with
\code{missingColor} and \code{FALSE} indicating that these values should be
painted with the nearest in-range color.}

\item{breaks}{an optional indication of break points between color levels
(see \code{\link{image}}).  If this is provided, the arguments \code{name}
through \code{blend} are all ignored (see \dQuote{Details}).  If it is
provided, then it may either be a vector of break points, or a single number
indicating the desired number of break points to be computed with
\code{\link{pretty}(z, breaks)}.  In either case of non-missing
\code{breaks}, the resultant break points must number 1 plus the number of
colors (see \code{col}).}

\item{col}{either a vector of colors or a function taking a numerical value
as its single argument and returning a vector of colors.  The value of
\code{col} is ignored if \code{name} is provided, or if \code{x0} through
\code{col1} are provided.}

\item{name}{an optional string naming a built-in colormap (one of
\code{"gmt_relief"}, \code{"gmt_ocean"}, \code{"gmt_globe"} or
\code{"gmt_gebco"}) or the name of a file or URL that contains a color map
specification in GMT format, e.g. one of the \code{.cpt} files from
\url{http://www.beamreach.org/maps/gmt/share/cpt}). If \code{name} is
provided, then \code{x0}, \code{x1}, \code{col0} and \code{col1} are all
ignored.}

\item{x0, x1, col0, col1}{Vectors that specify a color map.  They must all be
the same length, with \code{x0} and \code{x1} being numerical values, and
\code{col0} and \code{col1} being colors.  The colors may be strings (e.g.
\code{"red"}) or colors as defined by \code{\link{rgb}} or
\code{\link{hsv}}.}

\item{blend}{a number indicating how to blend colors within each band.
This is ignored except when \code{x0} through \code{col1} are supplied.  A
value of 0 means to use \code{col0[i]} through the interval \code{x0[i]} to
\code{x1[i]}.  A value of 1 means to use \code{col1[i]} in that interval.  A
value between 0 and 1 means to blend between the two colors according to
the stated fraction.  Values exceeding 1 are an error at present, but there
is a plan to use this to indicate subintervals, so a smooth palette can be
created from a few colors.}

\item{missingColor}{color to use for missing values.  If not provided, this
will be \code{"gray"}, unless \code{name} is given, in which case it comes
from that color table.}

\item{debug}{a flag that turns on debugging.  Set to 1 to get a moderate
amount of debugging information, or to 2 to get more.}
}
\value{
A list containing the following (not necessarily in this order)

\itemize{

\item \code{zcol}, a vector of colors for \code{z}, if \code{z} was
provided, otherwise \code{"black"}

\item \code{zlim}, a two-element vector suitable as the argument of the same
name supplied to \code{\link{image}} or \code{\link{imagep}}

\item \code{breaks} and \code{col}, vectors of breakpoints and colors,
suitable as the same-named arguments to \code{\link{image}} or
\code{\link{imagep}}

\item \code{zclip} the provided value of \code{zclip}.

\item \code{x0} and \code{x1}, numerical vectors of the sides of color
intervals, and \code{col0} and \code{col1}, vectors of corresponding
colors.  The meaning is the same as on input.  The purpose of returning
these four vectors is to permit users to alter color mapping, as in example
3 in \dQuote{Examples}.

\item \code{missingColor}, a color that could be used to specify missing
values, e.g. as the same-named argument to \code{\link{imagep}}.  If this is
supplied as an argument, its value is repeated in the return value.
Otherwise, its value is either \code{"gray"} or, in the case of \code{name}
being given, the value in the GMT color map specification.

\item \code{colfunction}, a univariate function that returns a vector
of colors, given a vector of \code{z} values; see Example 6.

}
}
\description{
Map values to colors, for use in palettes and plots. There are many ways to
use this function, and some study of the arguments should prove fruitful in
cases that extend far beyond the examples.
}
\details{
This is a multi-purpose function that generally links (``maps'') numerical
values to colors.  The return value can specify colors for points on a
graph, or \code{breaks} and \code{col} vectors that are suitable for use by
\code{\link{drawPalette}}, \code{\link{imagep}} or \code{\link{image}}.

There are three ways of specifying color schemes, and \code{colormap} works
by checking for each condition in turn.

\itemize{

\item{Case A.} Supply \code{z} but nothing else.  In this case,
\code{breaks} will be set to \code{\link{pretty}(z, 10)} and things are
otherwise as in case B.

\item{Case B.} Supply \code{breaks}.  In this case, \code{breaks} and
\code{col} are used together to specify a color scheme.  If \code{col} is a
function, then it is expected to take a single numerical argument that
specifies the number of colors, and this number will be set to
\code{length(breaks)-1}.  Otherwise, \code{col} may be a vector of colors,
and its length must be one less than the number of breaks.  (NB. if
\code{breaks} is given, then all other arguments except \code{col} and
\code{missingColor} are ignored.) \if{html}{The figure below explains the
(\code{breaks}, \code{col}) method of specifying a color mapping.  Note
that there must be one more break than color.  This is the method used by
e.g. \code{\link{image}}.}
\if{html}{\figure{colormap_fig_1.png}}

\item{Case C.} Do not supply \code{breaks}, but supply \code{name}
instead.  This \code{name} may be the name of a pre-defined color palette
(\code{"gmt_relief"}, \code{"gmt_ocean"}, \code{"gmt_globe"} or
\code{"gmt_gebco"}), or it may be the name of a file (including a URL)
containing a color map in the GMT format (see \dQuote{References}).  (NB.
if \code{name} is given, then all other arguments except \code{z} and
\code{missingColor} are ignored.)

\item{Case D.} Do not supply either \code{breaks} or \code{name}, but
instead supply each of \code{x0}, \code{x1}, \code{col0}, and \code{col1}.
These values are specify a value-color mapping that is similar to that used
for GMT color maps.  The method works by using \code{\link{seq}} to
interpolate between the elements of the \code{x0} vector.  The same is done
for \code{x1}.  Similarly, \code{\link{colorRampPalette}} is used to
interpolate between the colors in the \code{col0} vector, and the same is
done for \code{col1}.  \if{html}{The figure above explains the (\code{x0},
\code{x1}, \code{col0}, \code{col1}) method of specifying a color mapping.
Note that the each of the items has the same length. The case of
\code{blend=0}, which has color \code{col0[i]} between \code{x0[i]} and
\code{x1[i]}, is illustrated below.}
\if{html}{\figure{colormap_fig_2.png}}

}
}
\examples{
library(oce)
## Example 1. color scheme for points on xy plot
x <- seq(0, 1, length.out=40)
y <- sin(2 * pi * x)
par(mar=c(3, 3, 1, 1))
mar <- par('mar') # prevent margin creep by drawPalette()
## First, default breaks
c <- colormap(y)
drawPalette(c$zlim, col=c$col, breaks=c$breaks)
plot(x, y, bg=c$zcol, pch=21, cex=1)
grid()
par(mar=mar)
## Second, 100 breaks, yielding a smoother palette
c <- colormap(y, breaks=100)
drawPalette(c$zlim, col=c$col, breaks=c$breaks)
plot(x, y, bg=c$zcol, pch=21, cex=1)
grid()
par(mar=mar)

\dontrun{
## Example 2. topographic image with a standard color scheme
par(mfrow=c(1,1))
data(topoWorld)
cm <- colormap(name="gmt_globe")
imagep(topoWorld, breaks=cm$breaks, col=cm$col)

## Example 3. topographic image with modified colors,
## black for depths below 4km.
cm <- colormap(name="gmt_globe")
deep <- cm$x0 < -4000
cm$col0[deep] <- 'black'
cm$col1[deep] <- 'black'
cm <- colormap(x0=cm$x0, x1=cm$x1, col0=cm$col0, col1=cm$col1)
imagep(topoWorld, breaks=cm$breaks, col=cm$col)

## Example 4. image of world topography with water colorized
## smoothly from violet at 8km depth to blue
## at 4km depth, then blending in 0.5km increments
## to white at the coast, with tan for land.
cm <- colormap(x0=c(-8000, -4000,   0,  100),
               x1=c(-4000,     0, 100, 5000),
               col0=c("violet","blue","white","tan"),
               col1=c("blue","white","tan","yelloe"),
               blend=c(100, 8, 0))
lon <- topoWorld[['longitude']]
lat <- topoWorld[['latitude']]
z <- topoWorld[['z']]
imagep(lon, lat, z, breaks=cm$breaks, col=cm$col)
contour(lon, lat, z, levels=0, add=TRUE)
message("colormap() example 4 is broken")

## Example 5. visualize GMT style color map
cm <- colormap(name="gmt_globe", debug=4)
plot(seq_along(cm$x0), cm$x0, pch=21, bg=cm$col0)
grid()
points(seq_along(cm$x1), cm$x1, pch=21, bg=cm$col1)

## Example 6. colfunction
cm <- colormap(c(0, 1))
x <- 1:10
y <- (x - 5.5)^2
z <- seq(0, 1, length.out=length(x))
drawPalette(colormap=cm)
plot(x, y, pch=21, bg=cm$colfunction(z), cex=3)
}
}
\references{
Information on GMT software is given at
\code{http://gmt.soest.hawaii.edu} (link worked for years but failed
2015-12-12).  Diagrams showing the GMT color schemes are at
\code{http://www.geos.ed.ac.uk/it/howto/GMT/CPT/palettes.html} (link worked
for years but failed 2015-12-08), and numerical specifications for some
color maps are at \url{http://www.beamreach.org/maps/gmt/share/cpt},
\url{http://soliton.vm.bytemark.co.uk/pub/cpt-city}, and other sources.
}
\seealso{
Other things related to colors: \code{\link{oceColors9B}},
  \code{\link{oceColorsCDOM}},
  \code{\link{oceColorsChlorophyll}},
  \code{\link{oceColorsClosure}},
  \code{\link{oceColorsDensity}},
  \code{\link{oceColorsFreesurface}},
  \code{\link{oceColorsGebco}}, \code{\link{oceColorsJet}},
  \code{\link{oceColorsOxygen}},
  \code{\link{oceColorsPAR}},
  \code{\link{oceColorsPalette}},
  \code{\link{oceColorsPhase}},
  \code{\link{oceColorsSalinity}},
  \code{\link{oceColorsTemperature}},
  \code{\link{oceColorsTurbidity}},
  \code{\link{oceColorsTwo}},
  \code{\link{oceColorsVelocity}},
  \code{\link{oceColorsViridis}},
  \code{\link{oceColorsVorticity}}, \code{\link{ocecolors}}
}
\author{
Dan Kelley
}
\concept{things related to colors}
