// Copyright (C) 2013 - 2017  Metrum Research Group, LLC
//
// This file is part of mrgsolve.
//
// mrgsolve is free software: you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 2 of the License, or
// (at your option) any later version.
//
// mrgsolve is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with mrgsolve.  If not, see <http://www.gnu.org/licenses/>.


/**
 * @file quick.cpp
 * 
 */

#include "odeproblem.h"
#include "RcppInclude.h"
#include "dataobject.h"
#include "mrgsolve.h"

/** Perform a simple simulation run.
 * 
 * @param parin list of data and options for the simulation
 * @param inpar numeric parameter values
 * @param parnames parameter names
 * @param init numeric initial values
 * @param cmtnames compartment names
 * @param capture indices in capture vector to actually get
 * @param funs list of pointer addresses to model functions generated by 
 * <code>getNativeSymbolInfo()</code>
 * @param data the main data set
 * @param idata the idata data aset
 * @param OMEGA between-ID normal random effects
 * @param SIGMA within-ID normal random effects
 * @return list containing matrix of simulated data and a character vector of
 * tran names that may have been carried into the output
 *
 */

namespace {
  unsigned int timecol = 0;
  unsigned int cmtcol  = 1;
  unsigned int evidcol = 2;
  unsigned int amtcol  = 3;
  unsigned int ratecol = 4;
}

typedef  Rcpp::NumericMatrix::Column mcol;

// [[Rcpp::export]]
Rcpp::NumericMatrix QUICKSIM(const Rcpp::List& parin,
                             const Rcpp::NumericVector& param,
                             const Rcpp::NumericVector& init,
                             Rcpp::CharacterVector& parnames,
                             Rcpp::NumericMatrix& data,
                             Rcpp::IntegerVector& n,
                             const Rcpp::NumericMatrix& idata,
                             const Rcpp::IntegerVector& req,
                             const Rcpp::IntegerVector& capturei,
                             const Rcpp::List& funs,
                             const Rcpp::IntegerVector& nre) {
  
  dataobject idat(idata,parnames);
  
  const unsigned int capn = capturei.at(0);
  
  odeproblem* prob = new odeproblem(param, init, funs, capn);
  prob->copy_parin(parin);
  prob->neta(nre[0]);
  prob->neps(nre[1]);
  
  const unsigned int NN = n[0] * idata.nrow();
  const unsigned int nreq = req.size();
  
  Rcpp::NumericMatrix ans(NN,2+nreq+capn);
  
  mcol time = data(Rcpp::_,timecol);
  mcol evid = data(Rcpp::_,evidcol);
  mcol amt = data(Rcpp::_,amtcol);
  mcol cmt = data(Rcpp::_,cmtcol);
  mcol rate = data(Rcpp::_,ratecol);
  
  double tto =0;
  double tfrom = time[0];
  const int capstart = 2 + nreq;
  int crow = 0;
  unsigned int k; 
  double ID = 0;
  bool skip = false;
  
  size_t irow = idata.nrow();
  size_t drow = data.nrow();
  
  // Simulate each individual
  for(size_t i = 0; i < irow; ++i) {
    
    idat.copy_parameters(i,prob);
    
    ID = idata(i,0);
    prob->reset_newid(ID);
    
    if(i==0) prob->newind(0);
    
    prob->y_init(init);
    prob->config_call();
    prob->init_call(tto);
    prob->lsoda_init();
    
    tfrom = time[0];
    
    // Simulate the same data set
    for(size_t j = 0; j < drow; ++j) {
      
      skip = false;
      
      if(j==0) {
        prob->newind(1);
      } else {
        prob->newind(2); 
      }
      
      tto = time[j];
      
      prob->advance(tfrom,tto);
      
      switch(int(evid[j])) {
      case 0:
        break;
      case 1: 
        skip = true;
        if(rate[j] > 0) {
          prob->rate_bump(cmt[j],rate[j]);
        } else {
          prob->y_add(cmt[j],amt[j]);
        }
        prob->lsoda_init();
        break;
      case 9: 
        prob->rate_bump(cmt[j],-1.0*rate[j]);
        skip = true;
        prob->lsoda_init();
        break;
      default:
        break;
      }
      
      prob->table_call();
      tfrom = tto;
      
      if(!skip) {
        ans(crow,0) = ID;
        ans(crow,1) = time[j];
        for(k = 0; k < nreq;  ++k) {
          ans(crow,2+k) = prob->y(req[k]);
        }
        for(k = 0; k < capn; ++k) {
          ans(crow,capstart+k) = prob->capture(capturei[1+k]); 
        }
        ++crow;
      }
    }
  }
  delete prob;
  return ans;
}



// Rcpp::NumericMatrix QUICKSIM_DATA(const Rcpp::List& parin,
//                                   const Rcpp::NumericVector& param,
//                                   const Rcpp::NumericVector& init,
//                                   Rcpp::CharacterVector& parnames,
//                                   Rcpp::IntegerVector& n,
//                                   Rcpp::NumericMatrix& data,
//                                   const Rcpp::IntegerVector& req,
//                                   const Rcpp::IntegerVector& capturei,
//                                   const Rcpp::List& funs,
//                                   const Rcpp::IntegerVector& nre) {
//   
//   const unsigned int capn = capturei.at(0);
//   
//   odeproblem* prob = new odeproblem(param, init, funs, capn);
//   prob->copy_parin(parin);
//   prob->neta(nre[0]);
//   prob->neps(nre[1]);
//   
//   const unsigned int NN = n[0];
//   const unsigned int nreq = req.size();
//   
//   Rcpp::NumericMatrix ans(NN,2+nreq+capn);
//   
//   // mcol time = data(Rcpp::_,timecol);
//   // mcol evid = data(Rcpp::_,evidcol);
//   // mcol amt = data(Rcpp::_,amtcol);
//   // mcol cmt = data(Rcpp::_,cmtcol);
//   // mcol rate = data(Rcpp::_,ratecol);
//   // mcol id = data(Rcpp::_,idcol);
//     
//   double tto =0;
//   double tfrom = data(0,timecol);
//   const int capstart = 2 + nreq;
//   int crow = 0;
//   unsigned int k; 
//   double ID = data(0,idcol)+1;
//   bool skip = false;
//   
//   size_t drow = data.nrow();
//   
//   prob->y_init(init);
//   prob->config_call();
//   prob->init_call(tto);
//   prob->lsoda_init();
// 
//   // Simulate each individual
//   for(size_t i = 0; i < drow; ++i) {
//     
//     if(i==0) prob->newind(0);
//     
//     if(data(i,idcol) != ID) {
//       prob->reset_newid(data(i,idcol));
//       prob->init_call(tto);
//     }
//     
//     tto = data(i,timecol);
//     
//     prob->init_call_record(tto);
//   
//     skip = false;
//     
//     prob->advance(tfrom,tto);
//     
//     switch(int(data(i,evidcol))) {
//     case 0:
//       break;
//     case 1: 
//       skip = true;
//       if(data(i,ratecol) > 0) {
//         prob->rate_bump(data(i,cmtcol),data(i,ratecol));
//       } else {
//         prob->y_add(data(i,cmtcol),data(i,amtcol));
//       }
//       prob->lsoda_init();
//       break;
//     case 9: 
//       prob->rate_bump(data(i,cmtcol),-1.0*data(i,ratecol));
//       skip = true;
//       prob->lsoda_init();
//       break;
//     default:
//       break;
//     }
//     
//     prob->table_call();
//     tfrom = tto;
//     ID = data(i,idcol);
//     
//     if(!skip) {
//       ans(crow,0) = ID;
//       ans(crow,1) = data(i,timecol);
//       for(k = 0; k < nreq;  ++k) {
//         ans(crow,2+k) = prob->y(req[k]);
//       }
//       for(k = 0; k < capn; ++k) {
//         ans(crow,capstart+k) = prob->capture(capturei[1+k]); 
//       }
//       ++crow;
//     }
//   }
//   delete prob;
//   return ans;
// }




// Rcpp::NumericMatrix PREDSIM(const Rcpp::List& parin,
//                             const Rcpp::NumericVector& param,
//                             const Rcpp::NumericVector& init,
//                             Rcpp::CharacterVector& parnames,
//                             const Rcpp::NumericMatrix& idata,
//                             const Rcpp::IntegerVector& capturei,
//                             const Rcpp::List& funs) {
//   
//   dataobject idat(idata,parnames);
//   
//   const int capn = capturei.at(0);
//   odeproblem prob(param, init, funs, capn);
//   Rcpp::NumericMatrix ans(idata.nrow(),capn);
//   
//   int k; 
//   size_t irow = idata.nrow();
//   
//   // Simulate each individual
//   for(size_t i = 0; i < irow; ++i) {
//     idat.copy_parameters(i,&prob);
//     prob.init_call(0.0);
//     prob.table_call();
//     for(k = 0; k < capn; ++k) {
//       ans(i,k) = prob.capture(capturei[1+k]); 
//     }
//   }
//   return ans;
// }
