% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/randomEffects.R
\name{randomEffects}
\alias{randomEffects}
\title{Random effects estimation}
\usage{
randomEffects(
  m,
  Xformula = ~1,
  alpha = 0.95,
  ncores = 1,
  nlmPar = list(),
  fit = TRUE,
  retryFits = 0,
  retrySD = NULL,
  optMethod = "nlm",
  control = list(),
  modelName = NULL,
  ...
)
}
\arguments{
\item{m}{A \code{\link{momentuHMM}} object.}

\item{Xformula}{Formula for the design matrix of the random effects model. The default \code{Xformula=~1} specifies an intercept-only model with no additional individual covariate effects.}

\item{alpha}{Significance level of the confidence intervals. Default: 0.95 (i.e. 95\% CIs).}

\item{ncores}{number of cores to use for parallel processing}

\item{nlmPar}{List of parameters to pass to the optimization function \code{\link[stats]{nlm}}. See \code{\link{fitHMM}}.}

\item{fit}{\code{TRUE} if the HMM should be re-fitted at the shrinkage estimates, \code{FALSE} otherwise.}

\item{retryFits}{Non-negative integer indicating the number of times to attempt to iteratively fit the model using random perturbations of the current parameter estimates as the 
initial values for likelihood optimization. See \code{\link{fitHMM}}.}

\item{retrySD}{An optional list of scalars or vectors indicating the standard deviation to use for normal perturbations of each working scale parameter when \code{retryFits>0}. See \code{\link{fitHMM}}.}

\item{optMethod}{The optimization method to be used. See \code{\link{fitHMM}}.}

\item{control}{A list of control parameters to be passed to \code{\link[stats]{optim}} (ignored unless \code{optMethod="Nelder-Mead"} or \code{optMethod="SANN"}).}

\item{modelName}{An optional character string providing a name for the fitted model. See \code{\link{fitHMM}}.}

\item{...}{further arguments passed to or from other methods. Not currently used.}
}
\value{
A \code{randomEffects} model similar to a \code{\link{momentuHMM}} object, but including the additional random effect components:
\item{varcomp}{A list of length \code{nbStates*(nbStates-1)} with each element containing the random effect mean coefficient(s) (\code{mu}), random effect variance (\code{sigma}), 
and logit-scale shrinkage estimates for the state transition probability parameters (\code{ztilde}).}
\item{traceG}{The trace of the projection matrix for each random effect.}
}
\description{
Approximate individual-level random effects estimation for state transition probabilities based on Burnham & White (2002)
}
\examples{
\dontrun{
# simulated data with normal random effects
# and binary individual covariate 

nbAnimals <- 5 # should be larger for random effects estimation
obsPerAnimal <- 110
indCov <- rbinom(nbAnimals,1,0.5) # individual covariate
betaCov <- c(-0.5,0.5) # covariate effects
mu <- c(-0.1,0.1) # mean for random effects
sigma <- c(0.2,0.4) # sigma for random effects
beta0 <- cbind(rnorm(nbAnimals,mu[1],sigma[1]),
               rnorm(nbAnimals,mu[2],sigma[2]))

reData <- simData(nbAnimals=nbAnimals,obsPerAnimal=obsPerAnimal,nbStates=2,
                  dist=list(step="gamma"),formula=~0+ID+indCov,
                  Par=list(step=c(1,10,1,2)),
                  beta=rbind(beta0,betaCov),
                  covs=data.frame(indCov=rep(indCov,each=obsPerAnimal)))

# fit null model
nullFit <- fitHMM(reData,nbStates=2,
                  dist=list(step="gamma"),
                  Par0=list(step=c(1,10,1,2)))

# fit covariate model
covFit <- fitHMM(reData,nbStates=2,
                 dist=list(step="gamma"),formula=~indCov,
                 Par0=list(step=c(1,10,1,2)),
                 beta0=rbind(mu,betaCov)) 

# fit fixed effects model
fixFit <- fitHMM(reData,nbStates=2,
                 dist=list(step="gamma"),formula=~0+ID,
                 Par0=list(step=c(1,10,1,2)),
                 beta0=beta0)

# fit random effect model
reFit <- randomEffects(fixFit)

# fit random effect model with individual covariate
reCovFit <- randomEffects(fixFit, Xformula=~indCov)

# compare by AICc
AIC(nullFit,covFit,fixFit,reFit,reCovFit, n=nrow(reData))
}
}
\references{
Burnham, K.P. and White, G.C. 2002. Evaluation of some random effects methodology applicable to bird ringing data. Journal of Applied Statistics 29: 245-264.

McClintock, B.T. 2021. Worth the effort? A practical examination of random effects in hidden Markov models for animal telemetry data. Methods in Ecology and Evolution \doi{10.1111/2041-210X.13619}.
}
