% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/item.scores.R
\name{item.scores}
\alias{item.scores}
\title{Compute Scale Scores}
\usage{
item.scores(x, fun = c("mean", "sum", "median", "var", "sd", "min", "max"),
            prorated = TRUE, p.avail = NULL, n.avail = NULL, as.na = NULL,
            check = TRUE)
}
\arguments{
\item{x}{a matrix or data frame with numeric vectors.}

\item{fun}{a character string indicating the function used to compute scale scores, default: \code{"mean"}.}

\item{prorated}{logical: if \code{TRUE} (default), prorated scale scores are computed (see 'Details');
if \code{FALSE}, scale scores of only complete cases are computed.}

\item{p.avail}{a numeric value indicating the minimum proportion of available item responses needed for computing
a prorated scale score for each case, e.g. \code{p.avail = 0.8} indicates that scale scores are only
computed for cases with at least 80\% of item responses available. By default prorated scale scores are
computed for all cases with at least one item response. Note that either argument \code{p.avail} or
\code{n.avail} is used to specify the proration criterion.}

\item{n.avail}{an integer indicating the minimum number of available item responses needed for computing a prorated
scale score for each case, e.g. \code{n.avail = 2} indicates that scale scores are only computed for
cases with item responses on at least 2 items. By default prorated scale scores are computed for all
cases with at least one item response. Note that either argument \code{p.avail} or \code{n.avail}
is used to specify the proration criterion.}

\item{as.na}{a numeric vector indicating user-defined missing values,
i.e. these values are converted to \code{NA} before conducting the analysis.}

\item{check}{logical: if \code{TRUE}, argument specification is checked.}
}
\value{
Returns a numeric vector with the same length as \code{nrow(x)} containing (prorated) scale scores.
}
\description{
This function computes (prorated) scale scores by averaging the (available) items that measure a single construct by default.
}
\details{
Prorated mean scale scores are computed by averaging the available items, e.g., if a participant answers 4 out of 8 items,
the prorated scale score is the average of the 4 responses. Averaging the available items is equivalent to substituting the mean
of a participant's own observed items for each of the participant's missing items, i.e., \emph{person mean imputation}
(Mazza, Enders & Ruehlman, 2015) or \emph{ipsative mean imputation} (Schafer & Graham, 2002).

Proration may be reasonable when (1) a relatively high proportion of the items (e.g., 0.8) and never fewer than half are used
to form the scale score, (2) means of the items comprising a scale are similar and (3) the item-total correlations are similar
(Enders, 2010; Graham, 2009; Graham, 2012). Results of simulation studies indicate that proration is prone to substantial bias
when either the item means or the inter-item correlation vary (Lee, Bartholow, McCarthy, Pederson & Sher, 2014; Mazza et al., 2015).
}
\examples{
dat <- data.frame(item1 = c(3,  2,  4, 1,  5, 1,  3, NA),
                  item2 = c(2,  2, NA, 2,  4, 2, NA,  1),
                  item3 = c(1,  1,  2, 2,  4, 3, NA, NA),
                  item4 = c(4,  2,  4, 4, NA, 2, NA, NA),
                  item5 = c(3, NA, NA, 2,  4, 3, NA,  3))

# Prorated mean scale scores
item.scores(dat)

# Prorated standard deviation scale scores
item.scores(dat, fun = "sd")

# Sum scale scores without proration
item.scores(dat, fun = "sum", prorated = FALSE)

# Prorated mean scale scores,
# minimum proportion of available item responses = 0.8
item.scores(dat, p.avail = 0.8)

# Prorated mean scale scores,
# minimum number of available item responses = 3
item.scores(dat, n.avail = 3)
}
\references{
Enders, C. K. (2010). \emph{Applied missing data analysis}. New York, NY: Guilford Press.

Graham, J. W. (2009). Missing data analysis: Making it work in the real world. \emph{Annual Review of Psychology, 60}, 549-576.
https://doi.org/10.1146/annurev.psych.58.110405.085530

Graham, J. W. (2012). Missing data: Analysis and design. New York, NY: Springer

Lee, M. R., Bartholow, B. D., McCarhy, D. M., Pederson, S. L., & Sher, K. J. (2014). Two alternative approaches to conventional
person-mean imputation scoring of the self-rating of the effects of alcohol scale (SRE). \emph{Psychology of Addictive Behaviors, 29},
231-236. https://doi.org/10.1037/adb0000015

Mazza, G. L., Enders, C. G., & Ruehlman, L. S. (2015). Addressing item-level missing data: A comparison of proration and
full information maximum likelihood estimation. \emph{Multivariate Behavioral Research, 50}, 504-519.
https://doi.org/10.1080/00273171.2015.1068157

Schafer, J. L., & Graham, J. W. (2002). Missing data: Our view of the state of the art. \emph{Psychological Methods, 7}, 147-177.
https://doi.org/10.1037/1082-989X.7.2.147
}
\seealso{
\code{\link{group.scores}}, \code{\link{item.alpha}}, \code{\link{item.omega}},
}
\author{
Takuya Yanagida \email{takuya.yanagida@univie.ac.at}
}
