% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cohens.d.R
\name{cohens.d}
\alias{cohens.d}
\title{Cohen's d for Between- and Within-Subject Design}
\usage{
cohens.d(formula, data, paired = FALSE, weighted = TRUE, ref = NULL, correct = FALSE,
         digits = 2, conf.level = 0.95, as.na = NULL, check = TRUE,
         output = TRUE)
}
\arguments{
\item{formula}{in case of a between-subject design (i.e., \code{paired = FALSE}),
a formula of the form \code{y ~ group} for one outcome variable or
\code{cbind(y1, y2, y3) ~ group} for more than one outcome variable where
\code{y} is a numeric variable giving the data values and \code{group}
a numeric variable, character variable or factor with two values or factor
levels giving the corresponding group; in case of a within-subject design
(i.e., \code{paired = TRUE}), a formula of the form \code{post ~ pre} where
\code{post} and \code{pre} are numeric variables. Note that analysis for more
than one outcome variable is not permitted in within-subject design.}

\item{data}{a matrix or data frame containing the variables in the \code{formula}.}

\item{paired}{logical: if \code{TRUE}, Cohen's d for within-subject design is computed.}

\item{weighted}{logical: if \code{TRUE} (default), in case of a between-subject design the weighted
pooled standard deviation is used; in case of a within-subject design the correlation
between measures is controlled when computing the pooled standard deviation.}

\item{ref}{a numeric value or character string indicating the reference group in a between-subject
design or a character string indicating the reference variable in a within-subject
design. The standard deviation of the reference group or reference variable is used
to standardized the mean difference. If the standard deviation of the control group
is used (e.g. \code{group = "control"}), the effect size is usually called Glass' delta.}

\item{correct}{logical: if \code{TRUE}, correction factor to remove positive bias in small
samples is used. Note that correction factor is only applied when \code{weighted = TRUE}
and \code{ref = NULL}.}

\item{digits}{an integer value indicating the number of decimal places to be used for
displaying results.}

\item{conf.level}{a numeric value between 0 and 1 indicating the confidence level of the interval.}

\item{as.na}{a numeric vector indicating user-defined missing values,
i.e. these values are converted to \code{NA} before conducting the analysis.
Note that \code{as.na()} function is only applied to \code{y} but not to \code{group}
in a between-subject design, while \code{as.na()} function is applied to \code{pre} and
\code{post} in a within-subject design.}

\item{check}{logical: if \code{TRUE}, argument specification is checked.}

\item{output}{logical: if \code{TRUE}, output is shown on the console.}
}
\value{
Returns an object of class \code{cohens.d}, which is a list with following entries: function call (\code{call}),
matrix or data frame specified in \code{x} (\code{data}), specification of function arguments (\code{args}), and
list with results (\code{result}).
}
\description{
This function computes Cohen's d for between- and within-subject designs with confidence intervals.
By default, the function computes the standardized mean difference divided by the weighted pooled
standard deviation without applying the correction factor for removing the small sample bias.
}
\details{
Cohen (1988, p.67) proposed to compute the standardized mean difference by dividing the
mean difference by the unweighted pooled standard deviation (i.e., \code{weighted = FALSE}).

Glass et al. (1981, p. 29) suggested to use the standard deviation of the control group
(e.g., \code{ref = "control"}) to compute the standardized mean difference since the
standard deviation of the control group is unaffected by the treatment and will therefore
more closely reflect the population standard deviation.

Hedges (1981, p. 110) recommended to weight each group's standard deviation by its sample
size resulting in a weighted and pooled standard deviation (i.e., \code{weighted = TRUE}).
According to Hedges and Olkin (1985, p. 81), the standardized mean difference based
on the weighted and pooled standard deviation has a positive small sample bias, i.e.,
standardized mean difference is overestimates in small samples (i.e., sample size less
than 20 or less than 10 in each group). However, a correction factor can be applied to
remove the small sample bias (i.e., \code{correct = TRUE}). Note that a gamma function
is used for computing the correction factor when \eqn{n} < 200, while a approximation method
is used when \eqn{n} >= 200.

Note that the terminology is inconsistent because the standardized mean difference based
on the weighted and pooled standard deviation is usually called Cohen's d, but sometimes
called Hedges' g. Oftentimes, Cohen's d is called Hedges' d as soon as the correction factor
is applied. It is recommended to avoid the term Hedges' g (Cumming & Calin-Jageman, 2017, p. 171),
but to report which standard deviation was used to standardized the mean difference
(e.g., unweighted/weighted pooled standard deviation, or the standard deviation of the control group)
and whether a small sample correction factor was applied.

As for the terminology according to Lakens (2013), in between subject design (\code{paired = FALSE})
Cohen's \eqn{d_s} is computed when using \code{weighted = TRUE} and Hedges's \eqn{g_s} is computed when
using \code{correct = TRUE} in addition. In within-subject designs (\code{paired = TRUE}),
Cohen's \eqn{d_rm} is computed when using \code{weighted = TRUE}, while Cohen's \eqn{d_av} is computed
when using \code{weighted = FALSE}, and corresponding Hedges' \eqn{g_rm} and Hedges' \eqn{g_av} are computed
when using \code{correct = TRUE}.
}
\examples{
#--------------------------------------
# Between-subject design
dat.bs <- data.frame(group = c("cont", "cont", "cont", "treat",  "treat"),
                     y1 = c(1, 3, 2, 5, 7),
                     y2 = c(4, 3, 3, 6, 4),
                     y3 = c(7, 5, 7, 3, 2))

# Standardized mean difference divided by the weighted pooled
# standard deviation without small sample correction factor
cohens.d(y1 ~ group, data = dat.bs)

# Standardized mean difference divided by the unweighted pooled
# standard deviation without small sample correction factor
cohens.d(y1 ~ group, data = dat.bs, weighted = FALSE)

# Standardized mean difference divided by the weighted pooled
# standard deviation with small sample correction factor
cohens.d(y1 ~ group, data = dat.bs, correct = TRUE)

# Standardized mean difference divided by the standard deviation
# of the control group without small sample correction factor
cohens.d(y1 ~ group, data = dat.bs, ref = "cont")

# Cohens's d for for more than one outcome variable
cohens.d(cbind(y1, y2, y3) ~ group, data = dat.bs)

#--------------------------------------
# Within-subject design
dat.ws <- data.frame(pre = c(1, 3, 2, 5, 7),
                     post = c(2, 2, 1, 6, 8))

# Standardized mean difference divided by the pooled
# standard deviation while controlling for the correlation
# without small sample correction factor
cohens.d(post ~ pre, data = dat.ws, paired = TRUE)

# Standardized mean difference divided by the pooled
# standard deviation whithout controlling for the correlation
# without small sample correction factor
cohens.d(post ~ pre, data = dat.ws, paired = TRUE, weighted = FALSE)

# Standardized mean difference divided by the pooled
# standard deviation while controlling for the correlation
# with small sample correction factor
cohens.d(post ~ pre, data = dat.ws, paired = TRUE, correct = TRUE)

# Standardized mean difference divided by the standard deviation
# of the pretest without small sample correction factor
cohens.d(post ~ pre, data = dat.ws, paired = TRUE, ref = "pre")
}
\references{
Cohen, J. (1988). \emph{Statistical power analysis for the behavioral sciences} (2nd ed.).
Academic Press.

Cumming, G., & Calin-Jageman, R. (2017). \emph{Introduction to the new statistics: Estimation, open science,
& beyond}. Routledge.

Glass. G. V., McGaw, B., & Smith, M. L. (1981). \emph{Meta-analysis in social research}. Sage Publication.

Goulet-Pelletier, J.-C., & Cousineau, D. (2018) A review of effect sizes and their confidence intervals,
Part I: The Cohen's d family. \emph{The Quantitative Methods for Psychology, 14}, 242-265.
https://doi.org/10.20982/tqmp.14.4.p242

Hedges, L. V. (1981). Distribution theory for Glass's estimator of effect size and related estimators.
\emph{Journal of Educational Statistics, 6}(3), 106-128.

Hedges, L. V. & Olkin, I. (1985). \emph{Statistical methods for meta-analysis}. Academic Press.

Lakens, D. (2013). Calculating and reporting effect sizes to facilitate cumulative science:
A practical primer for t-tests and ANOVAs. \emph{Frontiers in Psychology, 4}, 1-12.
https://doi.org/10.3389/fpsyg.2013.00863
}
\seealso{
\code{\link{eta.sq}}, \code{\link{cont.coef}}, \code{\link{cramers.v}},\code{\link{cor.matrix}},
\code{\link{na.auxiliary}}.
}
\author{
Takuya Yanagida \email{takuya.yanagida@univie.ac.at}
}
