% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/measure_centrality.R
\name{centrality}
\alias{centrality}
\alias{node_degree}
\alias{tie_degree}
\alias{node_closeness}
\alias{tie_closeness}
\alias{node_betweenness}
\alias{tie_betweenness}
\alias{node_eigenvector}
\alias{tie_eigenvector}
\alias{node_reach}
\title{Measures of node and tie centrality}
\usage{
node_degree(object, normalized = TRUE, direction = c("all", "out", "in"))

tie_degree(object, normalized = TRUE)

node_closeness(object, normalized = TRUE, direction = "out", cutoff = NULL)

tie_closeness(object, normalized = TRUE)

node_betweenness(object, normalized = TRUE, cutoff = NULL)

tie_betweenness(object, normalized = TRUE)

node_eigenvector(object, normalized = TRUE, scale = FALSE)

tie_eigenvector(object, normalized = TRUE)

node_reach(object, normalized = TRUE, k = 2)
}
\arguments{
\item{object}{An object of a migraph-consistent class:
\itemize{
\item matrix (adjacency or incidence) from \code{{base}} R
\item edgelist, a data frame from \code{{base}} R or tibble from \code{{tibble}}
\item igraph, from the \code{{igraph}} package
\item network, from the \code{{network}} package
\item tbl_graph, from the \code{{tidygraph}} package
}}

\item{normalized}{Logical scalar, whether the centrality scores are normalized.
Different denominators are used depending on whether the object is one-mode or two-mode,
the type of centrality, and other arguments.}

\item{direction}{Character string, “out” bases the measure on outgoing ties,
“in” on incoming ties, and "all" on either/the sum of the two.
For two-mode networks, "all" uses as numerator the sum of differences
between the maximum centrality score for the mode
against all other centrality scores in the network,
whereas "in" uses as numerator the sum of differences
between the maximum centrality score for the mode
against only the centrality scores of the other nodes in that mode.}

\item{cutoff}{Maximum path length to use during calculations.}

\item{scale}{Logical scalar, whether to rescale the vector so the maximum score is 1.}

\item{k}{Integer of steps out to calculate reach}
}
\value{
A single centralization score if the object was one-mode,
and two centralization scores if the object was two-mode.

Depending on how and what kind of an object is passed to the function,
the function will return a \code{tidygraph} object where the nodes have been updated

A numeric vector giving the betweenness centrality measure of each node.

A numeric vector giving the eigenvector centrality measure of each node.
}
\description{
These functions calculate common centrality measures for one- and two-mode networks.
All measures attempt to use as much information as they are offered,
including whether the networks are directed, weighted, or multimodal.
If this would produce unintended results,
first transform the salient properties using e.g. \code{\link[=to_undirected]{to_undirected()}} functions.
All centrality and centralization measures return normalized measures by default,
including for two-mode networks.
}
\section{Functions}{
\itemize{
\item \code{node_degree}: Calculates the degree centrality of nodes in an unweighted network,
or weighted degree/strength of nodes in a weighted network.

\item \code{tie_degree}: Calculate the degree centrality of edges in a network

\item \code{node_closeness}: Calculate the closeness centrality of nodes in a network

\item \code{tie_closeness}: Calculate the closeness of each edge to each other edge
in the network.

\item \code{node_betweenness}: Calculate the betweenness centralities of nodes in a network

\item \code{tie_betweenness}: Calculate number of shortest paths going through an edge

\item \code{node_eigenvector}: Calculate the eigenvector centrality of nodes in a network

\item \code{tie_eigenvector}: Calculate the eigenvector centrality of edges in a network

\item \code{node_reach}: Calculate nodes' reach centrality
}}

\examples{
node_degree(mpn_elite_mex)
node_degree(ison_southern_women)
tie_degree(ison_adolescents)
node_closeness(mpn_elite_mex)
node_closeness(ison_southern_women)
(ec <- tie_closeness(ison_adolescents))
plot(ec)
ison_adolescents \%>\% 
  activate(edges) \%>\% mutate(weight = ec) \%>\% 
  autographr()
node_betweenness(mpn_elite_mex)
node_betweenness(ison_southern_women)
(tb <- tie_betweenness(ison_adolescents))
plot(tb)
ison_adolescents \%>\% 
  activate(edges) \%>\% mutate(weight = tb) \%>\% 
  autographr()
node_eigenvector(mpn_elite_mex)
node_eigenvector(ison_southern_women)
tie_eigenvector(ison_adolescents)
node_reach(ison_adolescents)
}
\references{
Faust, Katherine. 1997.
"Centrality in affiliation networks."
\emph{Social Networks} 19(2): 157-191.
\doi{10.1016/S0378-8733(96)00300-0}.

Borgatti, Stephen P., and Martin G. Everett. 1997.
"Network analysis of 2-mode data."
\emph{Social Networks} 19(3): 243-270.
\doi{10.1016/S0378-8733(96)00301-2}.

Borgatti, Stephen P., and Daniel S. Halgin. 2011.
"Analyzing affiliation networks."
In \emph{The SAGE Handbook of Social Network Analysis},
edited by John Scott and Peter J. Carrington, 417–33.
London, UK: Sage.
\doi{10.4135/9781446294413.n28}.

Bonacich, Phillip. 1991.
“Simultaneous Group and Individual Centralities.”
\emph{Social Networks} 13(2):155–68.
\doi{10.1016/0378-8733(91)90018-O}.
}
\seealso{
\code{\link[=to_undirected]{to_undirected()}} for removing edge directions
and \code{\link[=to_unweighted]{to_unweighted()}} for removing weights from a graph.

Other measures: 
\code{\link{centralisation}},
\code{\link{closure}},
\code{\link{cohesion}()},
\code{\link{diversity}},
\code{\link{features}},
\code{\link{holes}}
}
\concept{measures}
