% Generated by roxygen2 (4.1.1): do not edit by hand
% Please edit documentation in R/plsres.R
\name{plsres}
\alias{plsres}
\title{PLS results}
\usage{
plsres(y.pred, y.ref = NULL, ncomp.selected = NULL, xdecomp = NULL,
  ydecomp = NULL, info = "")
}
\arguments{
\item{y.pred}{predicted y values.}

\item{y.ref}{reference (measured) y values.}

\item{ncomp.selected}{selected (optimal) number of components.}

\item{xdecomp}{PLS decomposition of X data (object of class \code{ldecomp}).}

\item{ydecomp}{PLS decomposition of Y data (object of class \code{ldecomp}).}

\item{info}{information about the object.}
}
\value{
Returns an object of \code{plsres} class with following fields:
\item{ncomp }{number of components included to the model.}
\item{ncomp.selected }{selected (optimal) number of components.}
\item{y.ref }{a matrix with reference values for responses.}
\item{y.pred }{a matrix with predicted values for responses.}
\item{rmse }{a matrix with root mean squared error values for each response and component.}
\item{slope }{a matrix with slope values for each response and component.}
\item{r2 }{a matrix with determination coefficients for each response and component.}
\item{bias }{a matrix with bias values for each response and component.}
\item{sep }{a matrix with standard error values for each response and component.}
\item{rpd }{a matrix with RPD values for each response and component.}
\item{xdecomp }{decomposition of predictors (object of class \code{ldecomp}).}
\item{ydecomp }{decomposition of responses (object of class \code{ldecomp}).}
\item{info }{information about the object.}
}
\description{
\code{plsres} is used to store and visualize results of applying a PLS model to a new data.
}
\details{
Do not use \code{plsres} manually, the object is created automatically when one applies a PLS
model to a new data set, e.g. when calibrate and validate a PLS model (all calibration and
validation results in PLS model are stored as objects of \code{plsres} class) or use function
\code{\link{predict.pls}}.

The object gives access to all PLS results as well as to the plotting methods for visualisation
of the results. The \code{plsres} class also inherits all properties and methods of \code{regres}
 - general class for regression results.

If no reference values provided, regression statistics will not be calculated and most of the
plots not available. The class is also used for cross-validation results, in this case some of
the values and methods are not available (e.g. scores and scores plot, etc.).

All plots are based on \code{\link{mdaplot}} function, so most of its options can be used (e.g.
color grouping, etc.).

RPD is ratio of standard deviation of response values to standard error of prediction (SDy/SEP).
}
\examples{
### Examples of using PLS result class
library(mdatools)
## 1. Make a PLS model for concentration of first component
## using full-cross validation and get calibration results

data(simdata)
x = simdata$spectra.c
y = simdata$conc.c[, 1]

model = pls(x, y, ncomp = 8, cv = 1)
model = selectCompNum(model, 2)
res = model$calres

summary(res)
plot(res)

## 2. Make a PLS model for concentration of first component
## and apply model to a new dataset

data(simdata)
x = simdata$spectra.c
y = simdata$conc.c[, 1]

model = pls(x, y, ncomp = 6, cv = 1)
model = selectCompNum(model, 2)

x.new = simdata$spectra.t
y.new = simdata$conc.t[, 1]
res = predict(model, x.new, y.new)

summary(res)
plot(res)

## 3. Show variance and error plots for PLS results
par(mfrow = c(2, 2))
plotXCumVariance(res, type = 'h')
plotYCumVariance(res, type = 'b', show.labels = TRUE, legend.position = 'bottomright')
plotRMSE(res)
plotRMSE(res, type = 'h', show.labels = TRUE)
par(mfrow = c(1, 1))

## 4. Show scores plots for PLS results
## (for results plot we can use color grouping)
par(mfrow = c(2, 2))
plotXScores(res)
plotXScores(res, show.labels = TRUE, cgroup = y.new)
plotXYScores(res)
plotXYScores(res, comp = 2, show.labels = TRUE)
par(mfrow = c(1, 1))

## 5. Show predictions and residuals plots for PLS results
par(mfrow = c(2, 2))
plotXResiduals(res, show.label = TRUE, cgroup = y.new)
plotYResiduals(res, show.label = TRUE)
plotPredictions(res)
plotPredictions(res, ncomp = 4, xlab = 'C, reference', ylab = 'C, predictions')
par(mfrow = c(1, 1))
}
\seealso{
Methods for \code{plsres} objects:
\tabular{ll}{
   \code{print} \tab prints information about a \code{plsres} object.\cr
   \code{\link{summary.plsres}} \tab shows performance statistics for the results.\cr
   \code{\link{plot.plsres}} \tab shows plot overview of the results.\cr
   \code{\link{plotPredictions.plsres}} \tab shows predicted vs. measured plot.\cr
   \code{\link{plotXScores.plsres}} \tab shows scores plot for x decomposition.\cr
   \code{\link{plotXYScores.plsres}} \tab shows scores plot for x and y decomposition.\cr
   \code{\link{plotRMSE.plsres}} \tab shows RMSE plot.\cr
   \code{\link{plotXVariance.plsres}} \tab shows explained variance plot for x decomposition.\cr
   \code{\link{plotYVariance.plsres}} \tab shows explained variance plot for y decomposition.\cr
   \code{\link{plotXCumVariance.plsres}} \tab shows cumulative explained variance plot for y
   decomposition.\cr
   \code{\link{plotYCumVariance.plsres}} \tab shows cumulative explained variance plot for y
   decomposition.\cr
   \code{\link{plotXResiduals.plsres}} \tab shows T2 vs. Q plot for x decomposition.\cr
   \code{\link{plotYResiduals.regres}} \tab shows residuals plot for y values.\cr
}
See also \code{\link{pls}} - a class for PLS models.
}

